"""
Created on 2019年6月28日
@author: 刘益宗
device接口.
对应前端接口device
返回值统一为 {data:JSON结构体, msg:"", status:0}
"""

import json
import datetime
import os
import time
from threading import Timer
from copy import deepcopy
from threading import Thread
from flask import g
from flask import Blueprint
from flask.globals import request
from flask.json import jsonify
from H_9U import SysSettings
from H_9U.api.input import input_read_all_list, get_used_inputs_dict
from H_9U.api.output import output_read_all_list
from H_9U.service.device import device_svc
from H_9U.models.result import get_result_model, ResInfo
from H_9U.models.sysconst import ModelId, CardCategory, FunctionType, CardType, InterfaceType, DeviceType, \
    PermissionCode, DeviceConfig, HttpConfigType, LogConfig, DateConfig, UpgradeType, \
    Output_DL_Timing_Default
from H_9U.models.sysconst import Output_SL_Timing_Default
from H_9U.util.En_De_coryp import des_decrypt
from H_9U.util.cache import cacher
from H_9U.util.ip_ping import ping
from H_9U.util.logwrapper import api_log
from H_9U.util.log import logger
from H_9U.util.logutil import LogModule, LogOperation, add_opt_log, LogFunction
from H_9U.util.common import valid_json, valid_params, password_verifier
from H_9U.api.user import valid_license_user, inited_license_user
from H_9U.api.websender import push_data, push_data_org, clear_push_data
from H_9U.models.syncdataname import SyncDataName
from H_9U.util.permission_valid import get_code_list
from H_9U.service.middlewarefilepath import font_manager

device_bp = Blueprint('device', __name__)


@device_bp.route('/readDetail', methods=['GET', 'POST'])
@api_log
def device_read_detail():
    """
    读取设备详细信息
    :return:参考前端接口文档readDetail
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    rs = device_detail(device_id)
    return jsonify(rs)


def device_detail(device_id):
    """
    读取设备详情
    :param device_id: 设备Id
    :return: 结果对象，参考DeviceDetail
    """
    rs = device_svc.device_detail_read(device_id)
    if rs['status'] == 0 and rs['data']:
        # 网口 1 正常， 0 异常
        rs['data']['ethernetStatus'] = 1

        # 判断是否二合一卡
        for slot in rs['data']['slotList']:
            slot['cardCategory'] = CardCategory.get_card_category(slot['modelId'])
    return rs


def device_fiber_detail(device_id, slot_id):
    """
    读取设备详情
    :param device_id: 设备Id
    :return: 结果对象，参考DeviceDetail
    """
    return device_svc.read_fiber_detail(device_id, slot_id)

def device_init_status(device_id):
    return device_svc.device_init_status_read(device_id)


def device_detail_without_cache(device_id):
    """
    读取设备详情
    :param device_id: 设备Id
    :return: 结果对象，参考DeviceDetail
    """
    rs = device_svc.device_detail_without_cache(device_id)
    if rs['status'] == 0 and rs['data']:
        # 网口 1 正常， 0 异常
        rs['data']['ethernetStatus'] = 1

        # 判断是否二合一卡
        for slot in rs['data']['slotList']:
            slot['cardCategory'] = CardCategory.get_card_category(slot['modelId'])
    return rs


@device_bp.route('/readSlot', methods=['GET', 'POST'])
@api_log
def device_read_slot():
    """
    # 读取指定设备上指定槽（子卡）的信息
    :return:参考前端接口文档readSlot
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'slotId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, slot_id = values
    return jsonify(slot_detail(device_id, slot_id))


@device_bp.route('/readGenlock', methods=['GET', 'POST'])
@api_log
def device_read_genlock():
    """
    读取指定设备的Genlock的信息
    :return:参考前端接口文档readGenlock
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    genlock_id = params.get('genlockId', 0)

    return jsonify(device_svc.device_genlock_read(device_id, genlock_id))


@device_bp.route('/writeGenlock', methods=['GET', 'POST'])
@api_log
# @permission_required(PermissionName.OTHER_SETTING)
def device_write_genlock():
    """
    读取指定设备的Genlock的信息
    :return:参考前端接口文档readGenlock
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'genlockId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))

    user = g.user
    role_id = user['roleId']
    # 增加接口校验
    role_rs = get_code_list(PermissionCode.Device_Other, role_id)
    if role_rs['status'] != 0:
        return jsonify(role_rs)
    role_rs = get_code_list(PermissionCode.Open_Genlock, role_id)
    if role_rs['status'] != 0:
        return jsonify(role_rs)

    device_id, genlock_id = values
    rs = device_svc.device_genlock_write(device_id, genlock_id, params)
    if rs['status'] == 0:
        add_opt_log(LogModule.Device, LogFunction.Genlock, LogOperation.Edit, params)
        push_data_org(SyncDataName.Device_Write_Genlock, params)
        push_data(genLock_status_read, SyncDataName.Device_Read_GenLock_Status, device_id)
    return jsonify(rs)


@device_bp.route('/writeSlotFunction', methods=['GET', 'POST'])
@api_log
def device_write_slot_function():
    """
    设置子卡功能类型
    :return:参考前端接口文档writeSlotFunction
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'deviceId', 'slotId', 'functions')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, slot_id, functions = values
    if not functions:
        return jsonify(get_result_model(ResInfo.Params_Error))
    slot_rs = device_svc.device_slot_read(device_id, slot_id)
    if slot_rs['status'] != 0:
        return jsonify(get_result_model(ResInfo.Device_Detail_Error))
    is_4k = False
    if slot_rs['data']['cardType'] == CardType.In:
        card_model = slot_rs['data']['modelId']
        if card_model in [ModelId.Input_HDMI20_DP12, ModelId.Pd_Input_HDMI20_DP12]:
            is_4k = True
        __build_input_data(device_id, functions, slot_id)
        rs = device_svc.slot_function_write(device_id, slot_id, functions)
    else:
        funcs = __build_not_input_data(slot_rs, device_id, slot_id, functions)
        params['functions'] = funcs
        rs = device_svc.device_slot_write(device_id, slot_id, params)
    if rs['status'] == 0:
        add_opt_log(LogModule.Slot, LogFunction.Slot_Function, LogOperation.Edit, params)
        push_data(device_detail, SyncDataName.Device_Read_Detail, device_id)
        if slot_rs['data']['cardType'] == CardType.In:
            push_data(input_read_all_list, SyncDataName.Input_Read_List, device_id)
            if not is_4k:
                from H_9U.api.screen import screen_list
                push_data(screen_list, SyncDataName.Screen_Read_All_List, device_id)
        else:
            push_data(output_read_all_list, SyncDataName.Output_Read_List, device_id)

    return jsonify(rs)


def __build_input_data(device_id, functions, slot_id):
    d_rs = device_svc.device_detail_read(device_id)
    if d_rs['status'] != 0:
        return jsonify(get_result_model(ResInfo.Device_Detail_Error))
    model_id = d_rs['data']['modelId']
    if model_id not in DeviceType.HS_ALL and model_id not in DeviceType.Alita_ALL:  # 非H系列判断是否被使用
        used_rs = _get_used_inputs(device_id)
        if used_rs['status'] != 0:
            return jsonify(get_result_model(ResInfo.Device_Detail_Error))
        for f in functions:
            i_id = int(f['id'])
            if slot_id in used_rs['data'] and len(
                    [x for x in [2 * i_id, 2 * i_id + 1] if x in used_rs['data'][slot_id]]) > 0:
                return jsonify(get_result_model(ResInfo.Input_Is_Used))


def __build_not_input_data(slot_rs, device_id, slot_id, functions):
    o_rs = output_read_all_list(device_id)
    if o_rs['status'] != 0:
        logger.error("output list error in writeSlotFunction")
        return jsonify(get_result_model(ResInfo.Middle_Data_Err))
    outputs = [x for x in o_rs['data'] if x['slotId'] == slot_id]
    from H_9U.api.output import output_write_timing_original
    funcs = {}
    for func in functions:
        func_id = func['id']
        if [x for x in slot_rs['data']['functions'] if x['isUsed'] == 1 and x['id'] == func_id]:
            return jsonify(get_result_model(ResInfo.Output_Is_Used))
        functype = func['functionType']
        outs = [x for x in outputs if x['interfaceId'] // 2 == func_id]
        if functype == FunctionType.DL and len(outs) == 2:
            outs.sort(key=lambda x: x['outputId'], reverse=True)
            from H_9U.api.output import output_write_general_two
            general = outs[0]['general']
            general.pop('name')
            general['deviceId'] = device_id
            general['outputId'] = outs[1]['outputId']
            rs = output_write_general_two(device_id, outs[1]['outputId'], general)
            if rs['status'] != 0:
                logger.error("write output slot function reset timing err")
                return jsonify(get_result_model(ResInfo.Middle_Data_Err))
        for out in outs:
            if functype == FunctionType.DL:
                t = deepcopy(Output_DL_Timing_Default)
            else:
                t = deepcopy(Output_SL_Timing_Default)
            t['deviceId'] = device_id
            t['outputId'] = out['outputId']
            rs = output_write_timing_original(device_id, out['outputId'], t)
            if rs['status'] == 0 and functype == FunctionType.SL:
                t = deepcopy(Output_SL_Timing_Default)
                t['deviceId'] = device_id
                t['outputId'] = out['outputId'] - 1
                rs = output_write_timing_original(device_id, out['outputId'] - 1, t)
            if rs['status'] != 0:
                logger.error("write output slot function reset timing err")
                return jsonify(get_result_model(ResInfo.Middle_Data_Err))
        funcs['id' + str(func_id)] = {'functionType': functype}
    return funcs


@device_bp.route('/readIp', methods=['GET', 'POST'])
@api_log
def device_read_ip():
    """
    设备IP地址
    :return:参考前端接口文档readIp
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    return jsonify(read_ip(device_id))


@device_bp.route('/writeIp', methods=['GET', 'POST'])
@api_log
# @permission_required(PermissionName.COMMUNICATION_SETTING)
def device_write_ip():
    """
    设置设备IP地址
    :return:参考前端接口文档writeIP
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    push_data_org(SyncDataName.Device_Write_IP, params)
    opt = 0  # 重启
    Thread(target=_write_ip_shutdown, args=(device_id, params, opt, g.token, g.user)).start()
    add_opt_log(LogModule.Device, LogFunction.IP, LogOperation.Edit, params)
    return jsonify(get_result_model())


def _write_ip_shutdown(device_id, ip, opt, token, user):
    time.sleep(2)
    from H_9U import app
    with app.app_context():
        # 获取token
        g.token = token
        g.user = user
        rs = device_svc.device_ip_write(device_id, ip)
        if rs['status'] == 0:
            clear_push_data()
        else:
            push_data_org(SyncDataName.Device_Write_IP_Error, rs)


@device_bp.route('/readCom', methods=['GET', 'POST'])
@api_log
def device_read_com():
    """
    读取串口配置信息
    :return:参考前端接口文档readCom
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    return jsonify(device_svc.device_com_read(device_id))


@device_bp.route('/writeCom', methods=['GET', 'POST'])
@api_log
# @permission_required(PermissionName.COMMUNICATION_SETTING)
def device_write_com():
    """
    设置串口配置信息
    :return:参考前端接口文档writeCom
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = device_svc.device_com_write(device_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Device_Write_Com, params)
        add_opt_log(LogModule.Device, LogFunction.COM, LogOperation.Edit, params)
    return jsonify(device_svc.device_com_write(device_id, params))


@device_bp.route('/restore', methods=['GET', 'POST'])
@api_log
# @permission_required(PermissionName.RESET_SETTING)
def device_restore():
    """
    恢复出厂设置
    :return:s参考前端接口文档restore
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'type', 'password')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, opt_type, password = values
    password = des_decrypt(password)
    is_safe = g.safe
    flag = password_verifier(g.user['password'], password, is_safe)
    if not flag:
        return get_result_model(ResInfo.User_Password_Error)
    return jsonify(device_svc.device_restore_factory_write(device_id, opt_type))


@device_bp.route('/mandatorySave', methods=['GET', 'POST'])
@api_log
def device_save():
    """
    根设备保存当前信息
    :return:参考前端接口文档save
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    return jsonify(device_svc.device_save_write(device_id))


@device_bp.route('/shutdown', methods=['GET', 'POST'])
@api_log
# @permission_required(PermissionName.RESET_SETTING)
def device_shutdown():
    """
    设备关机
    :return:参考前端接口文档shutdown
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'type')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, opt_type = values
    rs = device_svc.device_shutdown_write(device_id, opt_type)
    if rs['status'] == 0:
        add_opt_log(LogModule.Device, LogFunction.Shutdown, LogOperation.Edit, params)
    return jsonify(rs)


@device_bp.route('/endProcess', methods=['GET', 'POST'])
@api_log
def device_end_process():
    """
    设备关机
    :return:参考前端接口文档shutdown
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'type')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, opt_type = values
    rs = device_svc.device_end_process(device_id, opt_type)
    return jsonify(rs)


@device_bp.route('/readFirewareVersion', methods=['GET', 'POST'])
@api_log
def device_fireware_version():
    """
    获取程序版本号
    :return:参考前端接口文档readFirewareVersion
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    return jsonify(read_fireware_version(device_id))


@device_bp.route('/Upgrade', methods=['GET', 'POST'])
def device_upgrade():
    """
    固件升级
    :return:参考前端接口文档upgrade
    """
    logger.info("API接口/device/Upgrade开始")
    # 判断是否升级，升级不允许操作
    if not device_svc.begin_upgrade():
        logger.info('设备正在升级中，不再次 发送升级命令')
        return get_result_model(ResInfo.Device_Is_Upgrading)

    # nginx 上传文件后重命名
    try:
        upload_file = request.form['file.path']
        unix_time = int(time.time() * 10000)
        dest_path = upload_file + str(unix_time) + '.zip'
        os.rename(upload_file, dest_path)
    except Exception as e:
        logger.error("重命名上传文件异常")
        logger.error(e)
        device_svc.end_upgrade()
        return jsonify(get_result_model(ResInfo.Params_Error))

    # 解析和校验参数
    try:
        device_id = request.form.get('deviceId')
        upgrade_type = request.form.get('upgradeType')
        parts_str = request.form.get('parts', None)

        if device_id is None or upgrade_type is None:
            device_svc.end_upgrade()
            return jsonify(get_result_model(ResInfo.Params_Error))

        if parts_str is not None and parts_str != "":
            parts = json.loads(parts_str)
        else:
            parts = None

        upgrade_type = int(upgrade_type)
        device_id = int(device_id)
        if upgrade_type not in UpgradeType.All_List or device_id < 0:
            device_svc.end_upgrade()
            return jsonify(get_result_model(ResInfo.Params_Error))
        if upgrade_type == UpgradeType.Part and parts is None:
            device_svc.end_upgrade()
            return jsonify(get_result_model(ResInfo.Params_Error))
        # 调用升级接口
        rs = device_svc.device_upgrade(device_id, upgrade_type, parts, dest_path)
        # rs = device_svc.device_upgrade(device_id, upgrade_type, parts)
        if rs['status'] != 0:
            device_svc.end_upgrade()
        else:
            add_opt_log(LogModule.Device, LogFunction.Upgrade, LogOperation.Edit, str(upgrade_type) + ',' + str(parts))
    except Exception as e:
        logger.error('固件升级失败：%s' % str(e))
        device_svc.end_upgrade()
        rs = get_result_model(ResInfo.Device_Upgrade_Fail)
    logger.info("API接口/device/Upgrade结束")
    return jsonify(rs)


@device_bp.route('/upgradeStatus', methods=['GET', 'POST'])
@api_log
def api_device_upgrade_status():
    """
    固件升级(二)：轮询读取固件升级状态
    :return: 参考前端接口文档upgradeStatus
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    return jsonify(device_svc.device_upgrade_sts_read(device_id))


@device_bp.route('/upgradeInfo', methods=['GET', 'POST'])
@api_log
def upgrade_info_read():
    """
    读取固件升级信息
    :return: 参考接口文档 upgradeInfo
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    rs = device_svc.device_upgrade_info_read(device_id)
    if rs['status'] == 0:
        rs['data']['versionType'] = SysSettings.Version_Type
    return jsonify(rs)


@device_bp.route('/writeSlot', methods=['GET', 'POST'])
@api_log
def write_slot():
    """
    设置slot属性
    :return: 成功失败
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'slotId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, slot_id = values
    rs = device_svc.device_slot_write(device_id, slot_id, params)
    if rs['status'] == 0:
        add_opt_log(LogModule.Slot, LogFunction.Slot, LogOperation.Edit, params)
    return jsonify(rs)


@device_bp.route('/readMainControl', methods=['GET', 'POST'])
@api_log
def main_control_read():
    """
    读取主控信息
    :return:
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    slot_id = params['slotId'] if 'slotId' in params else None

    return jsonify(device_svc.device_main_control_read(device_id, slot_id))


@device_bp.route('/importSettings', methods=['GET', 'POST'])
@api_log
# @permission_required(PermissionName.BACKUP_MANAGEMENT)
def device_settings_import():
    """
    导入配置文件
    :return:
    """
    # 开始导入
    if device_svc.begin_settings():
        try:
            device_id = int(request.form.get('deviceId', 0))
            rs = device_svc.import_settings(device_id)
        except Exception as e:
            logger.exception(e)
            rs = get_result_model(ResInfo.Server_Err)
        finally:
            device_svc.end_settings()
    else:
        rs = get_result_model(ResInfo.Device_Is_Setting)
    if rs['status'] == 0:
        add_opt_log(LogModule.Device, LogFunction.Import_Settings, LogOperation.Edit, LogConfig.Success)
    return jsonify(rs)


@device_bp.route('/exportSettings', methods=['GET', 'POST'])
@api_log
# @permission_required(PermissionName.BACKUP_MANAGEMENT)
def device_settings_export():
    """
    导入配置文件
    :return:
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    # 开始导出
    if device_svc.begin_settings():
        try:
            rs = device_svc.export_settings(device_id)
            if rs['status'] == 0:
                file_path = rs['data']['filePath']
                domain = read_domain(device_id)
                file_url = file_path[file_path.index('download'):]
                rs['data'] = {
                    'deviceId': device_id,
                    'fileUrl': domain + file_url
                }
        except Exception as e:
            logger.exception(e)
            rs = get_result_model(ResInfo.Server_Err)
        finally:
            device_svc.end_settings()
    else:
        rs = get_result_model(ResInfo.Device_Is_Setting)
    if rs['status'] == 0:
        add_opt_log(LogModule.Device, LogFunction.Export_Settings, LogOperation.Edit, LogConfig.Success)
    return jsonify(rs)


@device_bp.route('/selfCheck', methods=['GET', 'POST'])
@api_log
def device_self_check():
    """
    设备自检
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    user = g.user
    role_id = user['roleId']
    # 增加接口校验
    role_rs = get_code_list(PermissionCode.Device_Check, role_id)
    if role_rs['status'] != 0:
        return jsonify(role_rs)

    recheck = False
    if device_svc.read_self_check_status():  # 设备正在自检查
        recheck = True
    # 开启自检
    device_svc.begin_check()
    rs = device_svc.device_self_check(device_id)
    if rs['status'] != 0 and not recheck:
        # 失败则关闭自检
        device_svc.end_check()
    else:
        add_opt_log(LogModule.Device, LogFunction.Self_Check, LogOperation.Edit, LogConfig.Success)
    return jsonify(rs)


@device_bp.route('/selfCheckStatus', methods=['GET', 'POST'])
@api_log
def api_device_self_check_status():
    """
    设备自检状态
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    return jsonify(device_svc.device_self_check_status(device_id))


@device_bp.route('/effectiveTime', methods=['GET', 'POST'])
@api_log
def effective_time():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'licenseKey', 'Isrestricted')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, license_key, is_restricted = values

    license_key = params.pop('licenseKey')
    new_key = params.pop('newLicenseKey', '')

    local_time = params.pop('localTime')
    local_time['deviceId'] = device_id
    if local_time['automatic'] == 1:
        if 'time' not in local_time:
            return jsonify(get_result_model(ResInfo.Params_Error))

        detester = local_time.pop('time')
        try:
            date = datetime.datetime.strptime(detester, DateConfig.FormatDate)
        except Exception as e:
            logger.error(e)
            return jsonify(get_result_model(ResInfo.Params_Error))

        local_time['time'] = {
            "year": date.year,
            "month": date.month,
            "day": date.day,
            "hour": date.hour,
            "minute": date.minute,
            "second": date.second
        }

    if is_restricted == 1:
        if 'effectivetime' not in params:
            return jsonify(get_result_model(ResInfo.Params_Error))

        detester = params.pop('effectivetime')
        try:
            date = datetime.datetime.strptime(detester, DateConfig.FormatDate)
        except Exception as e:
            return jsonify(get_result_model(ResInfo.Params_Error))

        params['effectivetime'] = {
            "year": date.year,
            "month": date.month,
            "day": date.day,
            "hour": date.hour,
            "minute": date.minute,
            "second": date.second
        }
    rs = valid_license_user(license_key, new_key)
    if rs['status'] == 0:
        rs = device_svc.effective_time_write(device_id, params)
        if rs['status'] == 0:
            rs = device_svc.local_time_write(device_id, local_time)
            add_opt_log(LogModule.Device, LogFunction.Effective_Time, LogOperation.Edit, params)
    return jsonify(rs)


@device_bp.route('/refresh', methods=['GET', 'POST'])
@api_log
def refresh():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    from H_9U.util.cache import cacher
    cacher.clear_all()
    cacher.UserCache.clear()

    rs = device_detail(device_id)
    return jsonify(rs)


@device_bp.route('/readEffectiveTime', methods=['GET', 'POST'])
@api_log
def effective_time_read():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = inited_license_user()
    if rs['status'] == 0:
        inited = rs['data']['inited']
        rs = device_svc.effective_time_read(device_id)
        if rs['status'] == 0:
            rs['data']['initedLicenseKey'] = inited
    return jsonify(rs)


@device_bp.route('/readLocalTime', methods=['GET', 'POST'])
@api_log
def read_local_time():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    rs = device_svc.local_time_read(device_id)
    return jsonify(rs)


@device_bp.route('/localTime', methods=['GET', 'POST'])
@api_log
def write_local_time():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    if params['automatic'] == 1:
        if 'time' not in params:
            return jsonify(get_result_model(ResInfo.Params_Error))

        detester = params.pop('time')
        try:
            date = datetime.datetime.strptime(detester, DateConfig.FormatDate)
        except Exception as e:
            logger.error(e)
            return jsonify(get_result_model(ResInfo.Params_Error))

        params['time'] = {
            "year": date.year,
            "month": date.month,
            "day": date.day,
            "hour": date.hour,
            "minute": date.minute,
            "second": date.second
        }
    rs = device_svc.local_time_write(device_id, params)
    return jsonify(rs)


@device_bp.route('/writeIPCMode', methods=['GET', 'POST'])
@api_log
def write_ipc_mode():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'slotId', 'videoMode')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, slot_id, mode = values
    rs = slot_detail(device_id, slot_id)
    has_sources = False
    if rs['status'] == 0:
        if rs['data']['modelId'] != ModelId.Input_IPC_Card:
            rs = get_result_model(ResInfo.Device_Subcard_Model_Id_Err)
        else:
            from H_9U.api.ipc import ipc_slot_source_count
            rs = ipc_slot_source_count(device_id, slot_id)
            if rs['status'] == 0:
                has_sources = True
            rs = device_svc.ipc_mode_write(device_id, slot_id, params)
    if rs['status'] == 0:
        add_opt_log(LogModule.Slot, LogFunction.IPC_Mode, LogOperation.Edit, params)
        push_data_org(SyncDataName.Input_IPC_Mode, params)
        if has_sources:
            push_data_org(SyncDataName.Input_IPC_Source_Refresh, {'deviceId': 0})
        from H_9U.api.ipc import ipc_montage_list
        push_data(ipc_montage_list, SyncDataName.Input_IPC_Montage_List, device_id)
        from H_9U.api.screen import screen_list
        push_data(screen_list, SyncDataName.Screen_Read_All_List, device_id)
    return jsonify(rs)


@device_bp.route('/readResourceMode', methods=['GET', 'POST'])
@api_log
def read_resource_mode():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    rs = device_svc.resource_mode_read(device_id)
    return jsonify(rs)


@device_bp.route('/writeResourceMode', methods=['GET', 'POST'])
@api_log
def write_resource_mode():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'type')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, mode = values
    rs = device_svc.resource_mode_write(device_id, params)
    if rs['status'] == 0:
        # opt_type: 0：设备重启；1：设备关机；2：开机
        device_svc.device_shutdown_write(device_id, 0)
    return jsonify(rs)


@device_bp.route('/readBackupMode', methods=['GET', 'POST'])
@api_log
def read_backup_mode():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    rs = device_svc.backup_mode_read(device_id)
    return jsonify(rs)


@device_bp.route('/writeBackupMode', methods=['GET', 'POST'])
@api_log
def write_backup_mode():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'mode')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    user = g.user
    role_id = user['roleId']
    # 增加接口校验
    role_rs = get_code_list(PermissionCode.Device_Other, role_id)
    if role_rs['status'] != 0:
        return jsonify(role_rs)
    # 增加接口校验
    role_rs = get_code_list(PermissionCode.Device_Backups, role_id)
    if role_rs['status'] != 0:
        return jsonify(role_rs)


    device_id, mode = values
    rs = device_svc.backup_mode_write(device_id, params)
    return jsonify(rs)


@device_bp.route('/readDeviceType', methods=['GET', 'POST'])
@api_log
def read_device_type():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = device_svc.device_type_read(device_id)
    return jsonify(rs)


@device_bp.route('/hk/cardInfo', methods=['GET', 'POST'])
@api_log
def read_device_hk_card_info():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'cardType')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, card_type = values
    rs = device_svc.device_read_card_info(device_id, card_type)
    return jsonify(rs)


@device_bp.route('/hk/firewareVersion', methods=['GET', 'POST'])
@api_log
def read_device_hk_fire_ware_version():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = device_svc.device_read_device_hk_fire_ware_version(device_id)
    return jsonify(rs)


@device_bp.route('/readDeviceBackgroundColor', methods=['GET', 'POST'])
@api_log
def read_device_background_color():
    """
    读取设备屏幕和图层的背景颜色
    :param device_id: 设备Id
    :return: 结果对象，参考DeviceDetail
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = device_svc.read_device_background_color(device_id)
    return jsonify(rs)


@device_bp.route('/writeDeviceBackgroundColor', methods=['GET', 'POST'])
@api_log
def write_device_background_color():
    """
    设置设备屏幕和图层的背景颜色
    :param device_id: 设备Id
    :return: 结果对象，参考DeviceDetail
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = device_svc.write_device_background_color(device_id, params)
    if rs['status'] == 0 and rs['data']:
        add_opt_log(LogModule.Device, LogFunction.Background_Color, LogOperation.Edit, params)
    return jsonify(rs)


###### osd接口 #####
@device_bp.route('/fontList', methods=['GET', 'POST'])
@api_log
def api_font_list():
    """
    设置设备屏幕和图层的背景颜色
    :param device_id: 设备Id
    :return: 结果对象，参考DeviceDetail
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = device_font_list(device_id)
    return jsonify(rs)


@device_bp.route('/fontDelete', methods=['GET', 'POST'])
@api_log
def api_font_delete():
    """
    设置设备屏幕和图层的背景颜色
    :param device_id: 设备Id
    :return: 结果对象，参考DeviceDetail
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = device_svc.write_device_font_delete(device_id, params)
    if rs['status'] == 0:
        cacher.LayerCache.clear()
        push_data(device_font_list, SyncDataName.Device_Font_List, device_id)
    return jsonify(rs)


@device_bp.route('/fontAdd', methods=['GET', 'POST'])
def api_font_add():
    """
    创建BKG
    :return: bkg Id数据
    """
    params = request.form.get('json')
    params = valid_json(params)
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    #
    values = valid_params(params, 'deviceId')
    device_id = values
    rs = device_font_list(device_id)
    if rs['status'] != 0:
        return jsonify(get_result_model(ResInfo.Middle_Data_Err))
    else:
        if len(rs['data']['fontList']) > DeviceConfig.Device_Font_Count:
            return jsonify(get_result_model(ResInfo.Device_Font_Exceeds_Limit))
    rs = device_svc.device_font_create(device_id)
    if rs['status'] == 0:
        rs['data']['path'] = font_manager.get_font_path_info(rs['data']['path'])
        push_data(device_font_list, SyncDataName.Device_Font_List, device_id)
    return jsonify(rs)


@device_bp.route('/readVersionComparison', methods=['GET', 'POST'])
@api_log
def api_version_comparison_read():
    """
    设备子卡版本对比
    :param device_id: 设备Id
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = device_svc.device_version_comparison_data_read(device_id)
    return jsonify(rs)


@device_bp.route('/writeFiberMode', methods=['GET', 'POST'])
@api_log
def api_fiber_mode_write():
    """
    4光纤卡模式设置
    :param device_id: 设备Id
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = device_svc.device_fiber_mode_write(device_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Device_Write_FiberMode, params)
        from H_9U.api.screen import screen_list
        push_data(screen_list, SyncDataName.Screen_Read_All_List, device_id)
        from H_9U.api.output import output_read_all_list
        push_data(output_read_all_list, SyncDataName.Output_Read_List, device_id)
    return jsonify(rs)


@device_bp.route('/readInputBackupList', methods=['GET', 'POST'])
@api_log
def api_input_backup_list_read():
    """
    获取输入源备份列表
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = device_svc.input_backup_list_read(device_id)
    return jsonify(rs)


@device_bp.route('/writeInputBackup', methods=['GET', 'POST'])
@api_log
def api_input_backup_write():
    """
    开启/关闭输入源备份
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'enable', 'type')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, enable, type_info = values
    rs = device_svc.input_backup_write(device_id, enable, type_info, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Input_Backup_Switch, params)
        from H_9U.api.screen import screen_list
        push_data(screen_list, SyncDataName.Screen_Read_All_List, device_id)
    return jsonify(rs)


@device_bp.route('/readInputBackup', methods=['GET', 'POST'])
@api_log
def api_input_backup_read():
    """
    读取开启/关闭输入源备份状态
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = device_svc.input_backup_read(device_id)
    return jsonify(rs)


@device_bp.route('/inputBackupUpdate', methods=['GET', 'POST'])
@api_log
def api_input_backup_update():
    """
    添加/更新输入源备份
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = device_svc.input_backup_update(device_id, params)
    if rs['status'] == 0:
        list_rs = device_svc.input_backup_list_read(device_id)
        if list_rs['status'] == 0:
            rs['data'] = list_rs['data']
            push_data_org(SyncDataName.Input_Backup_Update, rs['data'])
        from H_9U.api.screen import screen_list
        push_data(screen_list, SyncDataName.Screen_Read_All_List, device_id)
    return jsonify(rs)


@device_bp.route('/inputBackupDelete', methods=['GET', 'POST'])
@api_log
def api_input_backup_delete():
    """
    删除输入源备份
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = device_svc.input_backup_delete(device_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Input_Backup_Delete, params)
    return jsonify(rs)


@device_bp.route('/test', methods=['GET', 'POST'])
@api_log
def api_test():
    """
    删除输入源备份
    :return: 结果对象
    """
    rs = device_svc.test()
    return jsonify(rs)


@device_bp.route('/checkAuthCode', methods=['GET', 'POST'])
@api_log
def api_check_auth_code():
    """
    忘记密码校验设备授权码
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'deviceId', 'authCode')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, auth_code = values
    return jsonify(device_svc.check_auth_code(device_id, auth_code))


@device_bp.route('/readHttp', methods=['GET', 'POST'])
@api_log
def api_switch_http_read():
    """
    忘记密码校验设备授权码
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    return jsonify(device_svc.switch_http_read(device_id, HttpConfigType.DeviceHttpConfig))


@device_bp.route('/writeHttp', methods=['GET', 'POST'])
@api_log
def api_switch_http_write():
    """
    忘记密码校验设备授权码
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'deviceId', 'enable')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, enable = values
    rs = device_svc.switch_http_write(device_id, HttpConfigType.DeviceHttpConfig, enable)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Device_Write_Http, params)
        if enable == 1:
            # 执行切换nginx配置脚本
            Timer(3, _nginx_reload_event, args =[' https']).start()
        else:
            # 执行切换nginx配置脚本
            Timer(3, _nginx_reload_event, args =[' http']).start()
    return jsonify(rs)


@device_bp.route('/centralCmdList', methods=['GET', 'POST'])
@api_log
def api_central_cmd_list():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = device_svc.central_cmd_list_read(device_id)
    return jsonify(rs)


@device_bp.route('/writeOutputInterfaceMode', methods=['GET', 'POST'])
@api_log
def api_write_output_interface_mode():
    """
    设置设备输出接口的模式
    :param device_id: 设备Id
    :return: 结果对象，参考DeviceDetail
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'slotId', 'interfaceId', 'outputMode')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, slot_id, interface_id, output_mode = values
    data = {
         "deviceId": device_id,
         "interfaceId": interface_id,
         "slotId": slot_id,
         "outputMode": output_mode
     }
    rs = device_svc.write_device_output_interface_mode(device_id, data)
    if rs['status'] == 0:
        push_data_org( SyncDataName.Device_Mode_Update, params)
    return jsonify(rs)


@device_bp.route('/centralCmdDetail', methods=['GET', 'POST'])
@api_log
def api_central_cmd_detail():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'id')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, cmd_id = values
    return jsonify(device_svc.central_cmd_detail_read(device_id, cmd_id))


@device_bp.route('/centralCmdCreate', methods=['GET', 'POST'])
@api_log
def api_central_cmd_create():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = device_svc.central_cmd_list_read(device_id)
    if rs['status'] != 0:
        return jsonify(rs)
    if len(rs['data']['cmdList']) >= 50:
        return jsonify(get_result_model(ResInfo.Device_Central_Cmd_Exceeds_Limit))
    params['createTime'] = str(time.time())
    rs = device_svc.central_cmd_create_write(device_id, params)
    if rs['status'] == 0:
        cmd_id = rs['data']['id']
        params['id'] = cmd_id
        push_data_org(SyncDataName.Device_Central_Cmd_Create, params)
    return jsonify(rs)


@device_bp.route('/centralCmdModify', methods=['GET', 'POST'])
@api_log
def api_central_cmd_modify():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'id', 'name', 'cmd')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, cmd_id, name, cmd = values
    rs = device_svc.central_cmd_modify_write(device_id, {'deviceId': device_id, 'id': cmd_id, 'name': name, 'cmd': cmd})
    if rs['status'] == 0:
        push_data_org(SyncDataName.Device_Central_Cmd_Modify, params)
    return jsonify(rs)


@device_bp.route('/centralCmdDeleteBatch', methods=['GET', 'POST'])
@api_log
def api_central_cmd_delete():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    if not params['cmdList']:
        return jsonify(get_result_model())
    rs = device_svc.central_cmd_delete_write(device_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Device_Central_Cmd_DeleteBatch, params)
    return jsonify(rs)


@device_bp.route('/centralCmdApply', methods=['GET', 'POST'])
@api_log
def api_central_cmd_apply():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'id')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, id = values
    rs = device_svc.central_cmd_apply_write(device_id, params)
    return jsonify(rs)


@device_bp.route('/writeAudioCardAttribute', methods=['GET', 'POST'])
@api_log
def api_audio_card_attribute():
    """
    设置音频卡的属性
    1. 通道设置
    2.接口延时设置
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'slotId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, slot_id = values
    rs = device_svc.audio_card_attribute_write(device_id, slot_id, params)
    if rs['status'] == 0:
        interface_id = params.get('interfaceId')
        push_data(slot_detail, SyncDataName.Device_Read_Audio_Detail, device_id, slot_id)
        # 音频卡设置单双通道
        if interface_id == 255:
            push_data_org(SyncDataName.Device_Audio_Attribute, params)
            from H_9U.api.screen import screen_list
            push_data(screen_list, SyncDataName.Screen_Read_All_List, device_id)
            push_data(audio_card_list, SyncDataName.Device_Audio_List, device_id)
    return jsonify(rs)


@device_bp.route('/readAudioCardList', methods=['GET', 'POST'])
@api_log
def api_audio_card_list():
    """
    读取所有音频卡的属性
    通道与源的属性
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = audio_card_list(device_id)
    return jsonify(rs)


@device_bp.route('/readGenLockWorkStatus', methods=['GET', 'POST'])
@api_log
def api_genLock_status():
    """
    读取所有genLock工作状态列表信息
    通道与源的属性
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = genLock_status_read(device_id)
    return jsonify(rs)


@device_bp.route('/writeOutputEthDistance', methods=['GET', 'POST'])
@api_log
def api_write_slot_eth_distance():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'slotId', 'ethdistance')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, slot_id, distance = values
    rs = slot_detail(device_id, slot_id)
    if rs['status'] == 0:
        if rs['data']['modelId'] not in [ModelId.Output_4_RJ45_Card, ModelId.Pd_Output_4_RJ45_Card]:
            rs = get_result_model(ResInfo.Device_Subcard_Model_Id_Err)
        else:
            params = {
                'deviceId': device_id,
                'slotId': slot_id,
                'ethdistance': distance
            }
            rs = device_svc.device_slot_eth_distance(device_id, slot_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Output_EthDistance, params)
    return jsonify(rs)


@device_bp.route('/writePortMode', methods=['GET', 'POST'])
@api_log
def api_write_port_mode():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = device_svc.write_port_mode(device_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Device_Write_Port_Mode, params)
        from H_9U.api.screen import screen_list
        push_data(screen_list, SyncDataName.Screen_Read_All_List, device_id)
        from H_9U.api.input import input_read_all_list
        push_data(input_read_all_list, SyncDataName.Input_Read_List, device_id)
    return jsonify(rs)

@device_bp.route('/sdpCreate', methods=['GET', 'POST'])
@api_log
def api_sdp_create():
    params = request.form.get('json')
    params = valid_json(params)
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'type', 'name')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, type, name = values
    try:
        # 从表单的file字段获取文件，file为该表单的name值
        f = request.files["file"]
        if f.filename.rsplit('.', 1)[1].lower() != "sdp":
            return get_result_model(ResInfo.Upload_File_Fail)
    except Exception as e:
        print(e)
        return get_result_model(ResInfo.Upload_File_Fail)
    return jsonify(device_svc.import_sdp(device_id, params))



@device_bp.route('/ipPing', methods=['GET', 'POST'])
@api_log
def api_ip_ping():
    """
    删除输入源备份
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'deviceId','ip')
    device_id, ip = values
    if ping(ip):
        rs = get_result_model(ResInfo.Device_Ip_Conflict)
    else:
        rs = get_result_model()
    return jsonify(rs)


@device_bp.route('/isDisplayed', methods=['GET', 'POST'])
@api_log
def api_ipc_ip_displayed():
    """
    删除输入源备份
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'deviceId','isDisplayed')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, is_displayed = values
    rs = device_svc.ip_displayed(device_id, is_displayed)
    return jsonify(rs)

@device_bp.route('/isDisplayedRead', methods=['GET', 'POST'])
@api_log
def api_ipc_displayed_read():
    """
    删除输入源备份
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = device_svc.ip_displayed_read(device_id)
    return jsonify(rs)


@device_bp.route('/writeLcdConfig', methods=['GET', 'POST'])
@api_log
def api_write_lcd_config():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'webLockEnable', 'lcdOffTimeOut','lcdScreenDisplay' ,'password')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, web_lock_enable, lcd_off_time_out, lcd_screen_display, password = values
    password = des_decrypt(password)
    flag = password_verifier(g.user['password'], password)
    if not flag:
        return jsonify(get_result_model(ResInfo.User_Password_Error))
    rs = device_svc.write_lcd_config(device_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Device_Write_Lcd_config, params)
    return jsonify(rs)


@device_bp.route('/readLcdConfig', methods=['GET', 'POST'])
@api_log
def api_read_lcd_config():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = device_svc.read_lcd_config(device_id)
    return jsonify(rs)


def lcd_detail(device_id):
    return device_svc.read_lcd_config(device_id)




@device_bp.route('/writeBrightMode', methods=['GET', 'POST'])
@api_log
def bright_mode_write():
    """
    设置亮度调节模式
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'globalBrightAdjustMode')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, mode = values
    rs = device_svc.bright_mode_write(device_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Device_Write_Bright, params)
    return jsonify(rs)


@device_bp.route('/readBrightMode', methods=['GET', 'POST'])
@api_log
def bright_mode_read():
    """
    读取亮度调节模式
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = device_svc.bright_mode_read(device_id)
    return jsonify(rs)




def genLock_status_read(device_id):
    return device_svc.genLock_status_read(device_id)


def audio_card_list(device_id):
    """
    读取所有音频卡的属性
    通道与源的属性
    :return: 结果对象
    """
    return device_svc.audio_input_audio_card_list(device_id)


def _nginx_reload_event(param):
    logger.info("....Nginx启动......")
    os.system('chmod +x ' + SysSettings.Safe_Config_SH_Path)
    os.system('sh ' + SysSettings.Safe_Config_SH_Path + param)


@device_bp.route('/testWrite', methods=['GET', 'POST'])
@api_log
def api_test_write():
    """
    删除输入源备份
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'deviceId','count')
    device_id, count = values
    rs = device_svc.test_write(count)
    return jsonify(rs)


def device_font_list(device_id):
    from H_9U.api.screen import screen_list
    screen_rs = screen_list(device_id)
    font_id_list = []
    if screen_rs['status'] == 0 and screen_rs['data']:
        for screen_messs in screen_rs['data']:
            font_id = screen_messs['Osd']['words']['font']
            font_id_list.append(font_id)
    rs = device_svc.read_device_font_list(device_id)
    if rs['status'] != 0 or not rs['data']['fontList']:
        return rs
    des = rs
    for x in des['data']['fontList']:
        x['path'] = font_manager.get_font_path_info(x['path'])
        x['name'] = x['name'].replace(" ", "")
        if len(font_id_list) == 0:
            x['is_used'] = 0
        else:
           if x['id'] in font_id_list:
               x['is_used'] = 1
           else:
               x['is_used'] = 0
    return des
###### osd 结束 #####

def slot_detail(device_id, slot_id):
    """
    读取设备详情信息， 补充输入输出信息
    :param device_id:  设备Id
    :param slot_id:  槽Id
    :return:  参考接口文档readSlot
    """
    # 读取设备槽信息
    rs = device_svc.device_slot_read(device_id, slot_id)

    if rs['status'] == 0 and rs['data']:
        interface_list = []
        model_id = rs['data']['modelId']
        card_type = rs['data']['cardType']
        # 判断卡类别
        card_category = CardCategory.get_card_category(model_id)
        rs['data']['cardCategory'] = card_category
        interface_list = rs['data']['interfaces']
        # 读取版本信息
        v_rs = device_svc.device_firware_version_read(device_id)
        if v_rs['status'] == 0 and v_rs['data']:
            v_rs = v_rs['data']
        else:
            v_rs = None
        # 卡类型 0: 不在线 1:IN槽; 2:OUT槽; 3:MVR槽
        # 根据卡类型补充输入信息和输出信息
        # 输入卡
        if card_type == CardType.In:
            signal = 0
            # 版本信息
            if v_rs:
                v_input = [x for x in v_rs['inputCard'] if x['slotId'] == slot_id][0]
                _build_slot_version_data(rs['data'], v_input)
            if card_category == CardCategory.H_2xHDMI20_2xDP12_I:
                interface_list = [s for s in rs['data']['interfaces'] if s['functionType'] != 255]
                for interface in interface_list:
                    if interface['interfaceId'] == 0 or interface['interfaceId'] == 1:
                        interface['interfaceId'] = 0
                    if interface['interfaceId'] == 2 or interface['interfaceId'] == 3:
                        interface['interfaceId'] = 1
            # 新 4选2卡中存在数据为2个接口，针对2接口数据不做处理，只处理返回4接口数据
            if card_category == CardCategory.H2xHDMIDPICard and len(rs['data']['interfaces']) == 4:
                interface_list = [s for s in rs['data']['interfaces'] if s['functionType'] != 255]
                for interface in interface_list:
                    if interface['interfaceId'] == 0 or interface['interfaceId'] == 1:
                        interface['interfaceId'] = 0
                    if interface['interfaceId'] == 2 or interface['interfaceId'] == 3:
                        interface['interfaceId'] = 1
            # if card_category == CardCategory.H1xST2110Card:
            #     interface_list = [s for s in rs['data']['interfaces'] if s['functionType'] != 255]
            #     for interface in interface_list:
            #        if interface['iSignal'] != 0:
            #            signal = interface['iSignal']
            #     for interface in interface_list:
            #         interface['iSignal'] = signal
            input_info = []
            # 两路光口卡返回的数据当SL时 为8个input, 不适合通用逻辑的判断处理
            if card_category == CardCategory.H2xFIBERCard:
                des = device_svc.read_fiber_detail(device_id, slot_id)
                if des['status'] == 0 and des['data']:
                    general = des['data']['general']
                    rs['data']['inputs'] = general
                return rs
            # 非IPC卡读取输入源信息
            if card_category != CardCategory.IPCCard:
                # 补充输入信息
                input_rs = input_read_all_list(device_id)
                if input_rs['status'] == 0 and input_rs['data']:
                    # 槽中所有输入
                    inputs = [i for i in input_rs['data'] if i['slotId'] == slot_id and i['online'] == 1]
                    if inputs:
                        is_signal = 0
                        for i in inputs:
                            # 如果是4选2 卡
                            if card_category in [CardCategory.H_2xHDMI20_2xDP12_I, CardCategory.H2xHDMIDPICard, CardCategory.H_2xHDMI21DP14AUDIOCard]:
                                new_list = [s for s in interface_list if s['interfaceId'] == i['interfaceId']]
                                if new_list:
                                    is_signal = new_list[0]['iSignal']
                            elif card_category == CardCategory.K4_HDMI_DP_Card:
                                is_signal = rs['data']['interfaces'][0]['iSignal']
                            elif card_category == CardCategory.H1xST2110Card:
                                interface_list = [s for s in rs['data']['interfaces'] if s['functionType'] != 255]
                                for interface in interface_list:
                                    if interface['iSignal'] != 0:
                                        signal = interface['iSignal']
                                    if interface['iSignal'] == 1:
                                        signal = interface['iSignal']
                                        break
                                is_signal = signal
                                # 刷新率为0  要显示信号
                                if i['resolution']['refresh'] == 0:
                                    signal = 0
                            else:
                                is_signal = [s for s in rs['data']['interfaces']if s['interfaceId'] == i['interfaceId']][0]['iSignal']
                                # 补充输入信息
                            new_in = {
                                    'id': i['inputId'],
                                    'name': i['general']['name'],
                                    'resolution': i['resolution'],
                                    'colorSpace': i['general']['colorSpace'],
                                    'sampleRate': i['general']['sampleRate'],
                                    'colorDepth': i['general']['colorDepth'],
                                    'iSignal': is_signal
                                }
                            if card_category == CardCategory.H1xST2110Card:
                                new_in['colorRange'] = i['general']['colorRange']
                                new_in['colorGamut'] = i['general']['colorGamut']
                                new_in['dynamicRange'] = i['general']['dynamicRange']
                            if 'cscMode' in i['general']:
                                new_in['cscMode'] = i['general']['cscMode']
                            input_info.append(new_in)

            if card_category ==  CardCategory.AudioCard:
                detail_rs = device_svc.audio_card_detail_read(device_id, slot_id)
                if detail_rs['status'] == 0 and detail_rs['data']:
                    rs['data']['audio'] = detail_rs['data']
                    rs['data']['functions'] = []
            rs['data']['inputs'] = input_info
            # 母子卡，补充slotFunction信息
            if card_category in [CardCategory.MotherChildCard, CardCategory.Input4xHDBaseTCard, CardCategory.H4xHDMIAudioCard, CardCategory.H_Output4LinkCard]:
                # 根据 interfaceType 判断是否切换功能
                used_rs = _get_used_inputs(device_id)
                func_rs = device_svc.slot_function_read(device_id, slot_id)
                functions = []
                if func_rs['status'] == 0 and func_rs['data'] and used_rs['status'] == 0:
                    # 当前对应功能接口
                    for i_id in [0, 1]:
                        interfaces = [x for x in func_rs['data']['interfaces'] if x['interfaceId'] // 2 == i_id]

                        slot_inters = [t for t in interfaces if t['interfaceType'] not in InterfaceType._Allow_Function_Change_List]
                        if slot_inters:
                            continue

                        if interfaces:
                            functions.append({
                                'id': i_id,
                                'functionType': FunctionType.DL
                                if len([x for x in interfaces if x['functionType'] != FunctionType.SL]) > 0
                                else FunctionType.SL,
                                'isUsed': 1 if slot_id in used_rs['data'] and len(
                                    [x for x in [i['interfaceId'] for i in interfaces] if x in used_rs['data'][slot_id]]) > 0
                                else 0
                            })
                if functions:
                    rs['data']['functions'] = functions
            elif card_category == CardCategory.K4_HDMI_DP_Card:
                used_rs = _get_used_inputs(device_id)
                functions = []
                if used_rs['status'] == 0:
                    for x in rs['data']['interfaces']:
                        if x['functionType'] != 255:
                            functions.append({
                                'id': x['interfaceId'],
                                'functionType': x['functionType'],
                                'isUsed': 1 if slot_id in used_rs['data'] else 0
                            })
                if functions:
                    rs['data']['functions'] = functions
            # 4选2
            elif card_category in [CardCategory.H_2xHDMI20_2xDP12_I, CardCategory.H2xHDMIDPICard]:
                used_rs = _get_used_inputs(device_id)
                functions = []
                if used_rs['status'] == 0:
                    if rs['data']['interfaces'][0]['functionType'] == 255:
                        functions.append({
                        'id': 0,
                        'functionType': 3,
                        'isUsed': 1 if slot_id in used_rs['data'] else 0
                    })
                    if len(rs['data']['interfaces']) == 4:
                        if rs['data']['interfaces'][2]['functionType'] == 255:
                            functions.append({
                            'id': 1,
                            'functionType': 3,
                            'isUsed': 1 if slot_id in used_rs['data'] else 0
                        })
                        if rs['data']['interfaces'][2]['functionType'] == 2:
                            functions.append({
                                'id': 1,
                                'functionType': 2,
                                'isUsed': 1 if slot_id in used_rs['data'] else 0
                            })
                    if rs['data']['interfaces'][0]['functionType'] == 2:
                        functions.append({
                            'id': 0,
                            'functionType': 2,
                            'isUsed': 1 if slot_id in used_rs['data'] else 0
                        })

                if functions:
                    rs['data']['functions'] = functions
            elif card_category in[CardCategory.H_2xHDMI21DP14AUDIOCard]:
                used_rs = _get_used_inputs(device_id)
                functions = []
                if used_rs['status'] == 0:
                    if rs['data']['interfaces'][0]['functionType'] == 255:
                        functions.append({
                            'id': 0,
                            'functionType': 5,
                            'isUsed': 1 if slot_id in used_rs['data'] else 0
                        })
                    if rs['data']['interfaces'][0]['functionType'] == 4:
                        functions.append({
                            'id': 0,
                            'functionType': 4,
                            'isUsed': 1 if slot_id in used_rs['data'] else 0
                        })
                if functions:
                    rs['data']['functions'] = functions
        if card_type == CardType.STD_IO_Card:
            if v_rs:
                v_input = [x for x in v_rs['inputCard'] if x['slotId'] == slot_id][0]
                _build_slot_version_data(rs['data'], v_input)
        # 输出卡
        elif card_type == CardType.Out or card_type == CardType.Sender:
            # 版本信息
            if v_rs:
                v_output = [x for x in v_rs['outputCard'] if x['slotId'] == slot_id][0]
                _build_slot_version_data(rs['data'], v_output)

            # 读取所有输出
            from H_9U.api.screen import screen_freezed_output
            freeze_output = screen_freezed_output(device_id)['data']['freeze']
            output_rs = output_read_all_list(device_id)
            output_info = []
            if output_rs['status'] == 0 and output_rs['data']:
                # 找到当前槽中所有输出
                outputs = [i for i in output_rs['data'] if i['slotId'] == slot_id]
                if outputs:
                    for i in outputs:
                        # 补充输出信息
                        output_info.append({
                            'id': i['outputId'],
                            'name': i['general']['name'],
                            'resolution': i['resolution'],
                            'iSignal': i['iSignal'],
                            'colorSpace': i['general']['colorSpace'],
                            'colorDepth': i['general']['colorDepth'],
                            'sampleRate': i['general']['sampleRate'],
                            'freeze': 1 if i['outputId'] in freeze_output else 0,
                            'outputMode':[x['outputMode'] for x in interface_list if x['interfaceId'] == i['interfaceId']][0],
                            'interfaceType': i['interfaceType'],
                            'interfaceId': i['interfaceId']

                        })
            rs['data']['outputs'] = output_info
            interface_id = "interfaceId"
            if card_category == CardCategory.TwoInOneCard:
                rs['data']['interfaces'] = [
                    {
                        interface_id: 0,
                        "interfaceType": InterfaceType.RJ45,
                        "iSignal": 0,
                        "functionType": FunctionType.SL
                    },
                    {
                        interface_id: 1,
                        "interfaceType": InterfaceType.OpticalFiber,
                        "iSignal": 0,
                        "functionType": FunctionType.SL
                    }
                ]
            elif card_category == CardCategory.TwoInOneCard_20:
                rs['data']['interfaces'] = [
                    {
                        interface_id: 0,
                        "interfaceType": InterfaceType.RJ45,
                        "iSignal": 0,
                        "functionType": FunctionType.SL
                    }
                ]
            elif card_category == CardCategory.Output4FiberCard:
                rs['data']['interfaces'] = [
                    {
                        interface_id: 0,
                        "interfaceType": InterfaceType.OpticalFiber,
                        "iSignal": 0,
                        "functionType": FunctionType.SL
                    }
                ]
        # 预监卡
        elif card_type == CardType.Mvr:
            # 版本信息
            if v_rs:
                _build_slot_version_data(rs['data'], v_rs['mvrCard'])

    return rs


def _build_slot_version_data(slot, version):
    """
    补充slot版本信息
    :param slot: slotdata
    :param version: 版本信息
    :return: 版本信息
    """
    if not slot or not version:
        slot['sn'] = ''
        slot['hardwareVersion'] = ''
        slot['mcuVersion'] = ''
        slot['fpgaVersion'] = ['']
        return
    slot['sn'] = version['sn']
    slot['hardwareVersion'] = version['hardver']
    slot['mcuVersion'] = version['MCU']['internalver']
    slot['fpgaVersion'] = []

    if 'FPGA' in version:
        slot['fpgaVersion'].append(version['FPGA']['internalver'])
    if 'FPGA_A' in version:
        slot['fpgaVersion'].append(version['FPGA_A']['internalver'])
    if 'FPGA_B' in version:
        slot['fpgaVersion'].append(version['FPGA_B']['internalver'])


def mvr_is_online(device_id):
    """
    预监卡是否在线
    :param device_id: 设备Id
    :return: 1 0 是否在线
    """
    return device_svc.mvr_is_online(device_id)


def device_upgrade_status():
    """
    查询是否在升级
    :return: true 是
    """
    return device_svc.read_upgrade_status()


def device_begin_upgrade():
    return device_svc.begin_upgrade()


def device_upgrade_from_usb():
    device_svc.set_upgrade_from_usb()


def device_self_check_status():
    return device_svc.read_self_check_status()


def device_setting_status():
    return device_svc.read_setting_file_operate_status()


def ipc_card_list(device_id):
    """
    获取ipc卡列表
    :param device_id:
    :return: ipc卡列表
    """
    rs = device_detail(device_id)
    if rs['status'] == 0:
        if rs['data']:
            ipc_list = [x['slotId'] for x in rs['data']['slotList'] if x['cardCategory'] == CardCategory.IPCCard]
            rs['data'] = ipc_list
        else:
            rs['data'] = []

    return rs


def _get_used_inputs(device_id):
    """
    读取被使用的输入源dict
    :param device_id: dictId
    :return: {slotId:[interfaceId1, interfaceId2, ...]}
    """
    return get_used_inputs_dict(device_id)


def read_ip(device_id):
    return device_svc.device_ip_read(device_id)


def read_domain(device_id):
    ip_rs = read_ip(device_id)
    domain = ''
    if ip_rs['status'] == 0:
        domain = 'http://{}.{}.{}.{}/'.format(
            ip_rs['data']['ip']['ip0'],
            ip_rs['data']['ip']['ip1'],
            ip_rs['data']['ip']['ip2'],
            ip_rs['data']['ip']['ip3']
        )
    return domain


def read_export_device_language(device_id):
    return device_svc.read_device_language(device_id)


def read_fireware_version(device_id):
    return device_svc.device_firware_version_read(device_id)


def ipc_out_mode_read(device_id, slot_id):
    return device_svc.ipc_mode_read(device_id, slot_id)
