"""
Created on 2019年6月28日
@author: 刘益宗
input接口.
对应前端接口input
返回值统一为 {data:JSON结构体, msg:"", status:0}
"""
from io import BytesIO
import base64
import os

from flask import Blueprint, g
from flask.globals import request
from flask.json import jsonify
from flask import send_file, send_from_directory

from H_9U.api.filter.filter import _filter_input_data, _filter_api_input_group_data, \
    role_filter_data_func
from H_9U.service.input import input_svc
from H_9U.service.role import rolesvc
from H_9U.util.common import valid_json, valid_params
from H_9U.models.result import get_result_model, ResInfo
from H_9U.models.sysconst import InterfaceType, CardCategory, FunctionType, CardType, DataType, \
    PermissionCode
from H_9U.util.logwrapper import api_log
from H_9U.util.log import logger
from H_9U.api.websender import push_data, push_data_org
from H_9U.models.syncdataname import SyncDataName
from H_9U.util.permission_valid import valid_function_permission


input_bp = Blueprint('input', __name__)


@input_bp.route('/readList', methods=['GET', 'POST'])
@api_log
def input_read_list():
    """
    输入列表
    :return: 无分组信息
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    user = g.user
    role_id = user['roleId']
    rs = input_read_all_list(device_id)
    if rs['status'] == 0:
        role_rs, role_type = role_filter_data_func(role_id)
        rs['data'] = _filter_input_data(role_type, role_rs['data']['inputList'], rs['data'])
    return jsonify(rs)


@input_bp.route('/readDetail', methods=['GET', 'POST'])
@api_log
def input_read_detail():
    """
    读取input详细信息
    :return:参考文档 readDetail
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'inputId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, input_id = values

    return jsonify(input_detail(device_id, input_id))


@input_bp.route('/readEDID', methods=['GET', 'POST'])
@api_log
def input_read_edid():
    """
    读取指定输入的EDID
    :return:参考文档readEDID
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'inputId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, input_id = values

    return jsonify(input_svc.input_edid_read(device_id, input_id))


@input_bp.route('/readOSD', methods=['GET', 'POST'])
@api_log
def input_read_osd():
    """
    读取指定输入的OSD
    :return:参考文档readOSD
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'inputId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, input_id = values

    return jsonify(input_svc.read_osd(device_id, input_id))


@input_bp.route('/writeOSD', methods=['GET', 'POST'])
@api_log
def input_write_osd():
    """
    设置指定OSD
    :return:参考文档writeOSD
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'inputId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, input_id = values

    # 接口权限校验 设置Screen信息  code = 2  data_type = 2
    rs = valid_function_permission(PermissionCode.Screen_Edit, DataType.Input_Type, input_id)
    if rs['status'] != 0:
        return jsonify(rs)

    rs = input_svc.write_osd(device_id, input_id, params)
    return jsonify(rs)


@input_bp.route('/writeImageOSD', methods=['GET', 'POST'])
def input_write_image_osd():
    """
    设置图片OSD
    :return:参考文档writeImgageOSD
    """
    params = request.form.get('json')
    params = valid_json(params)
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'inputId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, input_id = values

    # 接口校验
    rs = valid_function_permission(PermissionCode.Screen_Edit, DataType.Input_Type, input_id)
    if rs['status'] != 0:
        return jsonify(rs)

    has_file = False
    if 'file' in request.files:
        has_file = True

    rs = input_svc.write_osd(device_id, input_id, params, has_file)
    return jsonify(rs)


@input_bp.route('/deleteCrop', methods=['GET', 'POST'])
@api_log
def input_crop_delete():
    """
    删除指定input的截取信息
    :return: 参考文档deleteCrop
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    # 接口校验
    input_id = params.get('inputId')
    rs = valid_function_permission(PermissionCode.Screen_Edit, DataType.Input_Type, input_id)
    if rs['status'] != 0:
        return jsonify(rs)

    rs = input_svc.crop_delete(params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Input_Crop_Delete, params)
        from H_9U.api.screen import screen_list
        push_data(screen_list, SyncDataName.Screen_Read_All_List, device_id)

    return jsonify(rs)


@input_bp.route('/writeCrop', methods=['GET', 'POST'])
@api_log
def input_crop_create():
    """
    创建输入截取
    :return: 参考文档writeCrop
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    # 接口校验
    input_id = params.get('inputId')
    rs = valid_function_permission(PermissionCode.Screen_Edit, DataType.Input_Type, input_id)
    if rs['status'] != 0:
        return jsonify(rs)

    rs = input_svc.crop_create(params)
    if rs['status'] == 0:
        crop_rs = input_svc.read_crop_detail(device_id, params.get('inputId'), rs['data']['id'])
        if crop_rs['status'] == 0:
            crop_rs['data']['inputId'] = input_id
            push_data_org(SyncDataName.Input_Crop_Create,  crop_rs['data'])
    return jsonify(rs)


@input_bp.route('/updateCrop', methods=['GET', 'POST'])
@api_log
def input_crop_update():
    """
    创建输入截取
    :return: 参考文档writeCrop
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'inputId', 'cropId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, input_id, crop_id = values

    # 接口校验
    rs = valid_function_permission(PermissionCode.Screen_Edit, DataType.Input_Type, input_id)
    if rs['status'] != 0:
        return jsonify(rs)

    if crop_id < 0:
        return jsonify(get_result_model(ResInfo.Params_Error))

    rs = input_svc.crop_update(device_id, input_id, crop_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Input_Crop_Update, params)
        from H_9U.api.screen import screen_list
        push_data(screen_list, SyncDataName.Screen_Read_All_List, device_id)

    return jsonify(rs)


@input_bp.route('/writeGeneral', methods=['GET', 'POST'])
@api_log
def input_write_general():
    """
    设置常规信息
    :return: 参考接口文档writGeneral
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'inputId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, input_id = values

    # 接口校验
    rs = valid_function_permission(PermissionCode.Screen_Edit, DataType.Input_Type, input_id)
    if rs['status'] != 0:
        return jsonify(rs)

    rs = input_svc.write_general(device_id, input_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Input_Write_General, params)
        from H_9U.api.screen import screen_list
        push_data(screen_list, SyncDataName.Screen_Read_All_List, device_id)

    return jsonify(rs)


@input_bp.route('/readTiming', methods=['GET', 'POST'])
@api_log
def input_read_timing():
    """
    读取timing
    :return: 参考文档readTiming
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'inputId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, input_id = values

    return jsonify(input_svc.read_timing(device_id, input_id))


@input_bp.route('/writeTiming', methods=['GET', 'POST'])
@api_log
def input_write_timing():
    """
    设置Timing
    :return: 参考文档writeTiming
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'inputIdList')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, input_list = values

    # 接口校验
    for input_id in input_list:
        rs = valid_function_permission(PermissionCode.EDID_Edit, DataType.Input_Type, input_id)
        if rs['status'] != 0:
            return jsonify(rs)
    rs = input_svc.write_timing(device_id, input_list, params)
    if rs['data'] and len([x for x in rs['data'] if x['success'] == 1]) > 0:
        push_data_org(SyncDataName.Input_Read_List, device_id)
    return jsonify(rs)


@input_bp.route('/readInputOnlineList', methods=['GET', 'POST'])
@api_log
def input_read_input_online_list():
    """
    获取在线的输入列表数据
    :return: 无分组信息
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = input_read_all_list(device_id)
    if rs['status'] == 0 and rs['data']:
        input_list = []
        for message in rs['data']:
            if message['online'] == 1:

                input_message = {
                    "inputId": message['inputId'],
                    "online": message['online'],
                    "isSignal": message['iSignal'],
                    "slotId": message['slotId'],
                    "modelId": message['modelId'],
                    "interfaceType": message['interfaceType'],
                    "interfaceId": message['interfaceId'],
                    "crops": message['crops'],
                    "resolution": message['resolution']
                }
                if 'mosaicMode' in message:
                    mosaic_mode = message['mosaicMode']
                    input_message['mosaicMode'] = mosaic_mode
                input_list.append(input_message)
        rs['data'] = input_list
    return jsonify(rs)


@input_bp.route('/download', methods=['GET', 'POST'])
def input_edid_download():
    """
    导出edid
    :return: edid文件
    """
    input_ids = request.values.get('inputIds')
    if not input_ids:
        return jsonify(get_result_model(ResInfo.Params_Error))
    input_id = input_ids[0]
    if input_id <= 0:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = request.values.get('deviceId')

    # 接口校验
    rs = valid_function_permission(PermissionCode.EDID_Edit, DataType.Input_Type, input_id)
    if rs['status'] != 0:
        return jsonify(rs)

    rs = input_svc.input_edid_read(device_id, input_id)
    if rs['status'] == 0 and rs['data']:
        file_name = rs['data']['file']['fileName']
        data = base64.standard_b64decode(rs['data']['file']['fileData'])
        memory_file = BytesIO()
        with open(memory_file, "w", encoding='utf-8') as f:
            f.write(data)
        memory_file.seek(0)
        return send_file(memory_file, attachment_filename=file_name, as_attachment=True)


@input_bp.route('/importEDID', methods=['GET', 'POST'])
def input_edid_import():
    """
    导入edie文件
    :return: 结果对象
    """
    device_id = request.form.get('deviceId')
    input_ids = request.form.get('inputIdList')

    # 接口校验
    for input_id in input_ids:
        rs = valid_function_permission(PermissionCode.EDID_Edit, DataType.Input_Type, input_id)
        if rs['status'] != 0:
            return jsonify(rs)

    if device_id is None or not input_ids:
        return jsonify(get_result_model(ResInfo.Params_Error))
    return jsonify(input_svc.input_edid_import(int(device_id), input_ids.split(',')))


@input_bp.route('/exportEDID', methods=['GET', 'POST'])
def input_edid_export():
    """
    导入edie文件
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'inputIdList')
    device_id, input_ids = values
    if device_id is None or not input_ids:
        return jsonify(get_result_model(ResInfo.Params_Error))

    input_id = input_ids[0]
    rs = valid_function_permission(PermissionCode.EDID_Edit, DataType.Input_Type, input_id)
    if rs['status'] != 0:
        return jsonify(rs)

    rs = input_svc.input_edid_export(int(device_id), input_id)
    if rs['status'] == 0 and rs['data']:
        file_name = rs['data']['file']['fileName']
        if not os.path.exists(file_name):
            return jsonify(get_result_model(ResInfo.File_Not_Exists))
        path, name = os.path.split(file_name)
        return send_from_directory(path, name, as_attachment=True)
    return jsonify(rs)


@input_bp.route('/timingRestore', methods=['GET', 'POST'])
@api_log
def input_timing_restore():
    """
    导入edie文件
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'inputIdList')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, input_list = values
    # 接口校验
    for input_id in input_list:
        rs = valid_function_permission(PermissionCode.EDID_Edit, DataType.Input_Type, input_id)
        if rs['status'] != 0:
            return jsonify(rs)

    params['templateOperationType'] = 0
    rs = input_svc.write_timing(device_id, input_list, params)
    if rs['data'] and len([x for x in rs['data'] if x['success'] == 1]) > 0:
        push_data_org(SyncDataName.Input_Read_List, device_id)
    return jsonify(rs)


@input_bp.route('/groupList', methods=['GET', 'POST'])
@api_log
def input_group_list():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    # 获取当前登录用户的角色
    user = g.user
    role_id = user['roleId']
    rs = input_svc.input_group_list(device_id)
    if rs['status'] == 0:
        role_rs, role_type= role_filter_data_func(role_id)
        rs['data'] = _filter_api_input_group_data(role_type, role_rs['data']['inputList'], rs['data'])
    return jsonify(rs)


@input_bp.route('/groupCreate', methods=['GET', 'POST'])
@api_log
def input_group_create():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    # 获取当前登录用户的角色
    user = g.user
    role_id = user['roleId']
    role_rs = rolesvc.get_role_by_id(role_id)
    if role_rs['status'] == 0 and role_rs['data']:
        if role_rs['data']['role_type'] != 1:
            return jsonify(get_result_model(ResInfo.Role_Jurisdiction_Code))
    rs = input_svc.input_group_create(device_id, params)
    if rs['status'] == 0:
        g_id = rs['data']['id']
        detail = input_svc.input_group_detail(device_id, g_id)
        if detail['status'] == 0:
            rs['data'] = detail['data']
            push_data_org(SyncDataName.Input_Group_Data, detail['data'])
    return jsonify(rs)


@input_bp.route('/groupUpdate', methods=['GET', 'POST'])
@api_log
def input_group_update():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'inputGroupId', 'inputGroupName')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, group_id, group_name = values
    # 获取当前登录用户的角色
    user = g.user
    role_id = user['roleId']
    role_rs = rolesvc.get_role_by_id(role_id)
    if role_rs['status'] == 0 and role_rs['data']:
        if role_rs['data']['role_type'] != 1:
            return jsonify(get_result_model(ResInfo.Role_Jurisdiction_Code))
    group_data = {
        "deviceId": device_id,
        'groups': [
            {
                "inputGroupId": group_id,
                "inputGroupName": group_name
            }
        ]
    }
    rs = input_svc.input_group_list_update(device_id, group_data)
    if rs['status'] == 0:
        rs = input_svc.input_group_node_update(device_id, group_id, params)
        if rs['status'] == 0:
            detail = input_svc.input_group_detail(device_id, group_id)
            if detail['status'] == 0:
                push_data_org(SyncDataName.Input_Group_Data, detail['data'])
    return jsonify(rs)


@input_bp.route('/groupOrder', methods=['GET', 'POST'])
@api_log
def input_group_order():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'groups')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, groups = values
    rs = input_svc.input_group_list_update(device_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Input_Group_List, {'deviceId': device_id})
    return jsonify(rs)


@input_bp.route('/groupDelete', methods=['GET', 'POST'])
@api_log
def input_group_delete():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'inputGroupId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    # 获取当前登录用户的角色
    user = g.user
    role_id = user['roleId']
    role_rs = rolesvc.get_role_by_id(role_id)
    if role_rs['status'] == 0 and role_rs['data']:
        if role_rs['data']['role_type'] != 1:
            return jsonify(get_result_model(ResInfo.Role_Jurisdiction_Code))
    device_id, group_id = values
    rs = input_svc.input_group_delete(device_id, group_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Input_Group_Delete, params)
    return jsonify(rs)


@input_bp.route('/groupNodeDelete', methods=['GET', 'POST'])
@api_log
def input_group_node_delete():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'inputGroupId', 'inputId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, group_id, input_id = values
    # 获取当前登录用户的角色
    user = g.user
    role_id = user['roleId']
    role_rs = rolesvc.get_role_by_id(role_id)
    if role_rs['status'] == 0 and role_rs['data']:
        if role_rs['data']['role_type'] != 1:
            return jsonify(get_result_model(ResInfo.Role_Jurisdiction_Code))
    rs = input_svc.input_group_node_delete(device_id, group_id, input_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Input_Group_Node_Delete, params)
        push_data_org(SyncDataName.Input_Group_List, {'deviceId': device_id})
    return jsonify(rs)


@input_bp.route('/regulate', methods=['GET', 'POST'])
@api_log
def input_regulate():
    """
    输入源校正，只针对VGA
    :return:
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'inputId', 'regulate')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, input_id, regulate = values

    # 接口校验
    rs = valid_function_permission(PermissionCode.Screen_Edit, DataType.Input_Type, input_id)
    if rs['status'] != 0:
        return jsonify(rs)

    rs = input_svc.input_regulate(device_id, input_id, params)
    return jsonify(rs)


@input_bp.route('/reverseControl', methods=['GET', 'POST'])
@api_log
def api_input_reverse_control_write():
    """
    输入源反控设置
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'inputId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, input_id = values
    rs = input_svc.reverse_control_info_write(device_id, input_id, params)
    if rs['status'] == 0:
        push_data(input_read_all_list, SyncDataName.Input_Read_List, device_id)
        from H_9U.api.screen import screen_layers_list
        push_data(screen_layers_list, SyncDataName.Screen_Layer_List, device_id)
    return jsonify(rs)


@input_bp.route('/mosaicMontage', methods=['GET', 'POST'])
@api_log
def api_input_mosaic_montage_read():
    """
    读取光口拼接源
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'inputId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, input_id = values
    rs = input_svc.mosaic_montage_read(device_id, input_id)
    return jsonify(rs)


@input_bp.route('/writeMosaicMontage', methods=['GET', 'POST'])
@api_log
def api_input_mosaic_montage_write():
    """
    设置光口拼接源
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'inputId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, input_id = values
    rs = input_svc.mosaic_montage_write(device_id, params)
    if rs['status'] == 0:
        from H_9U.api.screen import screen_layers_list
        push_data(screen_layers_list, SyncDataName.Screen_Layer_List, device_id)
        push_data(input_read_all_list, SyncDataName.Input_Read_List, device_id)
        mosaic_montage_rs = input_svc.mosaic_montage_read(device_id, input_id)
        if mosaic_montage_rs['status'] == 0 and mosaic_montage_rs['data']:
            params['mosaicList'] = mosaic_montage_rs['data']['mosaicList']
            push_data_org(SyncDataName.Input_Write_MosaicMontage, params)
    return jsonify(rs)

@input_bp.route('/mosaicRename', methods=['GET', 'POST'])
@api_log
def api_input_mosaic_rename():
    """
    光口拼接源下的源重命名
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'inputId', 'mosaicId', 'mosaicName')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, input_id, mosaic_id, mosaic_name= values
    rs = input_svc.mosaic_rename(device_id, params)
    if rs['status'] == 0:
        from H_9U.api.screen import screen_layers_list
        push_data(screen_layers_list, SyncDataName.Screen_Layer_List, device_id)
        push_data_org(SyncDataName.Input_Mosaic_Rename, params)
    return jsonify(rs)

def input_detail(device_id, input_id):
    """
    读取input详细信息
    :param device_id:  设备Id
    :param input_id:  输入Id
    :return:  参考接口readDetail
    """
    rs = input_read_all_list(device_id)
    detail = {}
    if rs['status'] == 0:
        detail = [i for i in rs['data'] if i['inputId'] == input_id]
        if detail:
            detail = detail[0]
    rs['data'] = detail
    return rs


def input_original_detail(device_id, input_id):
    """
    读取输入原始详细信息
    :param device_id:  设备Id
    :param input_id:  输入Id
    :return:  设备详细信息
    """
    return input_svc.read_detail(device_id, input_id)


def input_get_connect_capacity(device_id, input_id):
    """
    读取输入接口连接容量
    :param device_id: 设备Id
    :param input_id:  输入Id
    :return:  连接容量
    """
    return input_svc.input_get_connect_capacity(device_id, input_id)


def input_read_all_list(device_id):
    """
    读取列表
    :param device_id: 设备Id
    :return: 结果对象
    """
    # 读取全部输入
    rs = input_svc.read_list(device_id)
    from H_9U.api.device import device_detail
    device_rs = device_detail(device_id)
    used_rs = get_used_inputs_dict(device_id)
    if rs["status"] == 0 and device_rs['status'] == 0 and used_rs['status'] == 0:
        crops = []
        input_dict = used_rs['data']
        if rs['data']['inputs']:
            # 循环输入信息
            for item in [x for x in rs['data']['inputs']]:
                if item['online'] == 1:
                    crop_rs = input_svc.read_crop_list(device_id, item['inputId'])
                    if crop_rs['status'] == 0 and crop_rs['data']:
                        crops = crop_rs['data']['crops']
                item['crops'] = crops
                item['iSignal'] = 1 if item['resolution']['refresh'] > 0 else 0
                item['isUsed'] = 1 if item['slotId'] in input_dict and item['interfaceId'] in input_dict[
                    item['slotId']] else 0
                # 12G SDI 输入卡 不支持EDID设置
                if item['interfaceType'] == InterfaceType.SDI_12G:
                    item['isEDIDSetting'] = InterfaceType.allow_edid_setting(item['interfaceType'], 1)
                else:
                    item['isEDIDSetting'] = InterfaceType.allow_edid_setting(item['interfaceType'])
                item['modelId'] =  [x for x in device_rs['data']['slotList'] if x['slotId'] == item['slotId']][0]['modelId']
        k4_cards = [x for x in device_rs['data']['slotList'] if x['cardCategory'] in [CardCategory.K4_HDMI_DP_Card, CardCategory.H_2xHDMI20_2xDP12_I, CardCategory.H2xHDMIDPICard, CardCategory.H_2xHDMI21DP14AUDIOCard]]
        k4_card_slots = [s['slotId'] for s in k4_cards]
        # 含有4k卡
        if k4_cards:
            for i in rs['data']['inputs']:
                try:
                    if i['online'] == 1 and i['slotId'] in k4_card_slots:
                        slot = [x for x in k4_cards if x['slotId'] == i['slotId']][0]
                        functions = [x for x in slot['interfaces'] if x['functionType'] != FunctionType.Disable]
                        if len(functions) == 1:
                            func_type = functions[0]['functionType'] if functions else FunctionType.Disable
                            if func_type == FunctionType.HDMI20:
                                i['interfaceType'] = InterfaceType.HDMI20
                            elif func_type == FunctionType.HDMI21:
                                i['interfaceType'] = InterfaceType.HDMI21
                            elif func_type == FunctionType.DP12:
                                i['interfaceType'] = InterfaceType.DP12
                            elif func_type == FunctionType.DP14:
                                i['interfaceType'] = InterfaceType.DP14
                            i['isUsed'] = 1 if i['slotId'] in input_dict else 0
                except Exception as e:
                    logger.error("4k输入源列表异常：" + str(e))
        k4_chose_2 = [x for x in device_rs['data']['slotList'] if x['cardCategory'] in [CardCategory.H_2xHDMI20_2xDP12_I, CardCategory.H2xHDMIDPICard] and len(x['interfaces']) == 4]
        k4_chose_2_slots = [s['slotId'] for s in k4_chose_2]
        if k4_chose_2:
            for i in rs['data']['inputs']:
                # 判断在线的且槽位号在4选2数组中
                i_slot_id = i['slotId']
                i_interface_id = i['interfaceId']
                if i['online'] == 1 and i_slot_id in k4_chose_2_slots:
                    # 遍历 4选2的槽位数组列表
                    for k4_chose_2_slot in k4_chose_2_slots:
                        # 匹配输入列表的槽位id 与 4选2 中的槽位id 一致
                        if i_slot_id == k4_chose_2_slot:
                            # 取出来每一个正确的设备详细数据
                            k4 = [x for x in k4_chose_2 if x['slotId'] == i_slot_id][0]
                            # for k4_chose in k4:
                            for device_info in k4['interfaces'][0:2]:
                                func_type = device_info['functionType']
                                if func_type == 255:
                                    continue
                                if func_type == FunctionType.HDMI20:
                                    func_type = InterfaceType.HDMI20
                                if func_type == FunctionType.DP12:
                                    func_type = InterfaceType.DP12
                                if i_interface_id == 0:
                                    i['interfaceType'] = func_type
                            for device_info in k4['interfaces'][2:4]:
                                function_type = device_info['functionType']
                                if function_type == 255:
                                    continue
                                if function_type == FunctionType.HDMI20:
                                    function_type = InterfaceType.HDMI20
                                if function_type == FunctionType.DP12:
                                    function_type = InterfaceType.DP12
                                if i_interface_id == 1:
                                    i['interfaceType'] = function_type
                        i['isUsed'] = 1 if i['slotId'] in input_dict else 0
    else:
        rs = get_result_model(ResInfo.Middle_Data_Err)
    rs['data'] = rs['data']['inputs']
    return rs


def input_general_list(device_id):
    return input_svc.read_list(device_id)


def input_group_detail(device_id, g_id):
    return input_svc.input_group_detail(device_id, g_id)


def get_used_inputs_dict(device_id):
    """
    读取被使用的输入源dict
    :param device_id: dictId
    :return: {slotId:[interfaceId1, interfaceId2, ...]}
    """
    from H_9U.api.device import device_detail_without_cache
    rs = device_detail_without_cache(device_id)
    if rs['status'] != 0:
        return rs
    used_dict = {}
    input_slots = [x for x in rs['data']['slotList'] if x['cardType'] == CardType.In and x['interfaces']]
    for slot in input_slots:
        for interface in slot['interfaces']:
            if interface['isUsed'] == 1:
                if slot['slotId'] in used_dict:
                    used_dict[slot['slotId']].append(interface['interfaceId'])
                else:
                    used_dict[slot['slotId']] = [interface['interfaceId']]
    rs['data'] = used_dict
    return rs


def get_slot_by_input_id(input_id):
    """
    根据输入源id获取槽位号
    :return:
    """
    slot_list = []
    detail_rs = input_svc.read_detail_by_input_id(input_id)
    if detail_rs['status'] == 0 and detail_rs['data']:
        detail_slot_id = detail_rs['data']['slotId']
        slot_list = [detail_slot_id]
    return slot_list
