"""
Created on 2019年6月28日
@author: 刘益宗
input接口.
对应前端接口input
返回值统一为 {data:JSON结构体, msg:"", status:0}
"""
import os
import time
from copy import deepcopy

from flask import Blueprint, g
from flask.globals import request
from flask.json import jsonify
from H_9U.util.log import logger
from H_9U.api.device import read_domain
from H_9U.conf.syssettings import SysSettings

from H_9U.api.filter.filter import role_filter_data_func, _filter_input_data, _filter_ipc_data
from H_9U.service.ipc import ipc_svc
from H_9U.service.role import rolesvc
from H_9U.util.common import valid_json, valid_params
from H_9U.models.result import get_result_model, ResInfo
from H_9U.models.sysconst import IPCDefaultValue, PermissionCode, IPCImportStatus, IPCTemplate
from H_9U.util.logwrapper import api_log
from H_9U.api.websender import push_data, push_data_org
from H_9U.models.syncdataname import SyncDataName
from H_9U.util.permission_valid import get_code_list

ipc_bp = Blueprint('ipc', __name__)


# IPC 接口 开始
@ipc_bp.route('/IPCSourceList', methods=['GET', 'POST'])
@api_log
def ipc_source_list():
    """
    分页获取ipc source list
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    params['seqPageIndex'] -= 1
    return jsonify(ipc_svc.ipc_source_list(device_id, params))


@ipc_bp.route('/IPCSourceFreeList', methods=['GET', 'POST'])
@api_log
def ipc_source_free_list():
    """
    获取ipc源总数量并且过滤掉了分组内占用的源
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    return jsonify(ipc_svc.ipc_source_free_list(device_id, params))


@ipc_bp.route('/IPCSourceChannelList', methods=['GET', 'POST'])
@api_log
def ipc_source_channel_list():
    """
    分页获取ipc源信息带通道码流详情
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    params['seqPageIndex'] -= 1
    return jsonify(ipc_svc.ipc_source_channel_list(device_id, params))


@ipc_bp.route('/IPCSourceLayOutList', methods=['GET', 'POST'])
@api_log
def ipc_source_layout_list():
    """
    分页获取ipc source layout list
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'deviceId', 'channelNum')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, channel_num = values
    params['seqPageIndex'] -= 1
    # 获取当前登录用户的角色
    return jsonify(ipc_svc.ipc_source_layout_list(device_id, channel_num, params))


@ipc_bp.route('/IPCSourceListBySlotId', methods=['GET', 'POST'])
@api_log
def ipc_source_list_by_slot_id():
    """
    根据slotId 查询源列表数据
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'slotId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, slot_id = values
    rs = ipc_svc.ipc_source_list_by_slot_id(device_id, slot_id)
    return jsonify(rs)


@ipc_bp.route('/IPCCameraList', methods=['GET', 'POST'])
@api_log
def ipc_camera_list():
    """
    ipc扫描摄像头列表
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'decodeCapacity')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, decode_capacity = values
    return jsonify(ipc_svc.ipc_camera_list(device_id, decode_capacity))


@ipc_bp.route('/IPCChannelList', methods=['GET', 'POST'])
@api_log
def ipc_channel_list():
    """
    ipc源通道列表
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'sourceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, source_id = values
    rs = ipc_svc.ipc_channel_detail_list(device_id, source_id)
    return jsonify(rs)

def ipc_top_channel(device_id, source_id):
    return ipc_svc.ipc_top_channel(device_id, source_id)


@ipc_bp.route('/IPCChannelFullDetail', methods=['GET', 'POST'])
@api_log
def ipc_channel_detail_api():
    """
    ipc源通道详情·
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'sourceId', "channelId")
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, source_id, channel_id = values
    rs = ipc_channel_detail(device_id, source_id, channel_id)
    return jsonify(rs)


@ipc_bp.route('/IPCSourceCreate', methods=['GET', 'POST'])
@api_log
def ipc_source_create():
    """
    创建IPC信号源，不反返回错误
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    user = g.user
    role_id = user['roleId']
    # 增加接口校验
    role_rs = get_code_list(PermissionCode.IPC_Administration, role_id)
    if role_rs['status'] != 0:
        return jsonify(role_rs)
    # 增加接口校验
    role_rs = get_code_list(PermissionCode.IPC_Edit, role_id)
    if role_rs['status'] != 0:
        return jsonify(role_rs)
    rs = ipc_svc.ipc_source_create(device_id, params)
    if rs['status'] == 0:
        push_data_org(
            SyncDataName.Input_IPC_Source_Create, {
                'deviceId': device_id})
    return jsonify(rs)


@ipc_bp.route('/IPCSourceUpdate', methods=['GET', 'POST'])
@api_log
def ipc_source_update():
    """
    修改IPC信号源，不反返回错误
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'sourceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, source_id = values

    user = g.user
    role_id = user['roleId']
    # 增加接口校验
    role_rs = get_code_list(PermissionCode.IPC_Administration, role_id)
    if role_rs['status'] != 0:
        return jsonify(role_rs)
    # 增加接口校验
    role_rs = get_code_list(PermissionCode.IPC_Edit, role_id)
    if role_rs['status'] != 0:
        return jsonify(role_rs)
    rs = ipc_svc.ipc_source_update(device_id, params)
    if rs['status'] == 0:
        data_info = {
            "deviceId": device_id,
            "sourceId": source_id,
            "name": params.get('sourceName')
        }
        push_data_org(SyncDataName.Input_IPC_Source_Rename, data_info)
        from H_9U.api.screen import screen_list
        push_data(
            screen_list,
            SyncDataName.Screen_Read_All_List,
            device_id)
        push_data_org(SyncDataName.Input_IPC_Source_Edit, params)
    return jsonify(rs)


@ipc_bp.route('/IPCSourceDetail', methods=['GET', 'POST'])
@api_log
def ipc_source_detail_info():
    """
    IPC信号源，不反返回错误
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'sourceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, source_id = values
    rs = ipc_svc.ipc_source_detail_info(device_id, source_id)
    return jsonify(rs)


@ipc_bp.route('/IPCSourceDelete', methods=['GET', 'POST'])
@api_log
def ipc_source_delete():
    """
    删除IPC信号源，返回删除成功的id
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = ipc_svc.ipc_source_delete(device_id, params)
    if rs['status'] == 0:
        from H_9U.api.screen import screen_list
        push_data(
            screen_list,
            SyncDataName.Screen_Read_All_List,
            device_id)
        push_data_org(SyncDataName.Input_IPC_Source_Delete, {'deviceId': device_id, 'sourceList': [
                      {'sourceId': x['sourceId']} for x in rs['data']['sourceList'] if x['result'] == 0]})
    return jsonify(rs)


@ipc_bp.route('/IPCSourceRename', methods=['GET', 'POST'])
@api_log
def ipc_source_rename():
    """
    ipc信号源重命名
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'sourceId', 'name')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, source_id, name = values
    rs = ipc_svc.ipc_source_general(
        device_id, source_id, {
            'deviceId': device_id, 'sourceId': source_id, 'sourceName': name})
    if rs['status'] == 0:
        push_data_org(SyncDataName.Input_IPC_Source_Rename, params)
        from H_9U.api.screen import screen_list
        push_data(
            screen_list,
            SyncDataName.Screen_Read_All_List,
            device_id)
    return jsonify(rs)


@ipc_bp.route('/IPCChannelEdit', methods=['GET', 'POST'])
@api_log
def ipc_channel_edit():
    """
    ipc信号源通道修改
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(
        params,
        'deviceId',
        'slotId',
        'sourceId',
        'channelId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, slot_id, source_id, channel_id = values

    user = g.user
    role_id = user['roleId']
    # 增加接口校验
    role_rs = get_code_list(PermissionCode.IPC_Edit, role_id)
    if role_rs['status'] != 0:
        return jsonify(role_rs)
    rs = ipc_svc.ipc_channel_edit(device_id, params)
    if rs['status'] == 0:
        push_data(
            ipc_svc.ipc_channel_detail,
            SyncDataName.Input_IPC_Channel_Edit,
            device_id,
            source_id, channel_id)
        from H_9U.api.screen import screen_list
        push_data(
            screen_list,
            SyncDataName.Screen_Read_All_List,
            device_id)
    return jsonify(rs)


@ipc_bp.route('/IPCChannelAdd', methods=['GET', 'POST'])
@api_log
def ipc_channel_add():
    """
    ipc信号源通道添加
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'deviceId', 'slotId', 'sourceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, slot_id, source_id = values
    user = g.user
    role_id = user['roleId']
    # 增加接口校验
    role_rs = get_code_list(PermissionCode.IPC_Edit, role_id)
    if role_rs['status'] != 0:
        return jsonify(role_rs)

    rs = ipc_svc.ipc_channel_add(device_id, params)
    if rs['status'] == 0:
        push_data(
            ipc_svc.ipc_channel_detail_list,
            SyncDataName.Input_IPC_Channel_Add,
            device_id,
            source_id)
    return jsonify(rs)


@ipc_bp.route('/IPCChannelDelete', methods=['GET', 'POST'])
@api_log
def ipc_channel_delete():
    """
    ipc信号源通道删除
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'deviceId', 'sourceList')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, slot_list = values
    rs = ipc_svc.ipc_channel_delete(device_id, params)
    if rs['status'] == 0:
        channel_id_list = []
        source_id = 0
        for x in rs['data']['sourceList']:
            source_id = x['sourceId']
            for channel in x['channelList']:
                if channel['result'] == 0:
                    channel_id_list.append(channel['channelId'])
        push_data_org(
            SyncDataName.Input_IPC_Channel_Delete, {
                'deviceId': device_id, 'sourceId': source_id, 'channelList': channel_id_list})
        from H_9U.api.screen import screen_list
        push_data(
            screen_list,
            SyncDataName.Screen_Read_All_List,
            device_id)
    return jsonify(rs)


@ipc_bp.route('/IPCSlotList', methods=['GET', 'POST'])
@api_log
def ipc_slot_count_list():
    """
    ipc slot源列表
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    # 获取当前登录用户的角色
    user = g.user
    role_id = user['roleId']
    rs = get_role_ipc_list_data(_ipc_slot_source_count_list(device_id), role_id)
    return jsonify(rs)


@ipc_bp.route('/IPCChannelFreeList', methods=['GET', 'POST'])
@api_log
def ipc_channel_free_list():
    """
    ipc 空闲的通道数据
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'deviceId', 'slotId', 'montageId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, slot_id, montage_id = values
    params['seqPageIndex'] -= 1
    rs = ipc_svc.ipc_source_channel_list(device_id, params)
    ms = ipc_montage_list(device_id)
    if ms['status'] != 0:
        return jsonify(ms)
    # 获取当前拼接源使用的streamId 集合
    stream_ids = _get_stream_id(ms, slot_id, montage_id)
    des = get_result_model()
    if rs['status'] == 0:
        des = _get_channel_free_data(rs, stream_ids)
    return jsonify(des)



@ipc_bp.route('/IPCSlotSourceCount', methods=['GET', 'POST'])
@api_log
def ipc_slot_count():
    """
    ipc 源个数以及通道个数
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'deviceId', 'slotId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, slot_id = values
    rs = ipc_svc.ipc_slot_source_count(device_id, slot_id)
    return jsonify(rs)


@ipc_bp.route('/IPCCheckSourceName', methods=['GET', 'POST'])
def ipc_check_source_name():
    """
    查询导入文件处理状态
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'deviceId', 'sourceId', 'name')
    device_id, source_id, name = values
    rs = ipc_svc.ipc_search_source_list(
        device_id, {'deviceId': device_id, 'name': name})
    repeat = 0
    if rs['status'] == 0:
        for x in rs['data']['sourceList']:
            if x['sourceName'] == name and x['sourceId'] != source_id:
                repeat = 1
                break
    else:
        repeat = 1

    rs = get_result_model()
    rs['data'] = {
        'deviceId': device_id,
        'sourceId': source_id,
        'name': name,
        'repeat': repeat
    }
    return jsonify(rs)


# ipc源分组 开始
@ipc_bp.route('/IPCGroupList', methods=['GET', 'POST'])
@api_log
def ipc_group_list():
    """
    ipc分组列表
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    rs = ipc_svc.ipc_group_list(device_id)
    return jsonify(rs)


@ipc_bp.route('/IPCGroupSourceList', methods=['GET', 'POST'])
@api_log
def ipc_group_source_list():
    """
    ipc分组列表
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    rs = ipc_svc.ipc_group_source_list(device_id)
    return jsonify(rs)


@ipc_bp.route('/IPCGroupNodeList', methods=['GET', 'POST'])
@api_log
def ipc_group_node_list():
    """ipc 分组下源列表"""
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'groupId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, group_id = values

    return jsonify(ipc_svc.ipc_group_node_list_without_page(device_id, group_id))


@ipc_bp.route('/IPCGroupCreate', methods=['GET', 'POST'])
@api_log
def ipc_group_create():
    """
    创建ipc源分组
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    # 获取当前登录用户的角色
    user = g.user
    role_id = user['roleId']
    role_rs = rolesvc.get_role_by_id(role_id)
    if role_rs['status'] == 0 and role_rs['data']:
        if role_rs['data']['role_type'] != 1:
            return jsonify(get_result_model(ResInfo.Role_Jurisdiction_Code))
    rs = ipc_svc.ipc_group_create(device_id, params)
    if rs['status'] == 0:
        data = {
            "deviceId": device_id,
            "groupId": rs['data']['id'],
            "name": params['name'],
            "parentId": params['parentId'],
            "level": params['level'],
            "sourceList": params['sourceList']
        }
        rs['data'] = data
        # 推送
        push_data_org(SyncDataName.Input_IPC_GROUP_CREATE, data)
    return jsonify(rs)


@ipc_bp.route('/IPCGroupDelete', methods=['GET', 'POST'])
@api_log
def ipc_group_delete():
    """
    删除ipc源分组
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    # 获取当前登录用户的角色
    user = g.user
    role_id = user['roleId']
    role_rs = rolesvc.get_role_by_id(role_id)
    if role_rs['status'] == 0 and role_rs['data']:
        if role_rs['data']['role_type'] != 1:
            return jsonify(get_result_model(ResInfo.Role_Jurisdiction_Code))

    rs = ipc_svc.ipc_group_delete(device_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Input_IPC_GROUP_DELETE, params)
    return jsonify(rs)


@ipc_bp.route('/IPCGroupUpdate', methods=['GET', 'POST'])
@api_log
def ipc_group_update():
    """
    更新ipc源分组
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'groupId', 'name')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, group_id, name = values

    # 获取当前登录用户的角色
    user = g.user
    role_id = user['roleId']
    role_rs = rolesvc.get_role_by_id(role_id)
    if role_rs['status'] == 0 and role_rs['data']:
        if role_rs['data']['role_type'] != 1:
            return jsonify(get_result_model(ResInfo.Role_Jurisdiction_Code))

    rs = ipc_svc.ipc_group_update(device_id, group_id, params)
    if rs['status'] == 0:
        data = {
            "deviceId": device_id,
            "groupId": group_id,
            "name": params['name'],
            "parentId": params['parentId'],
            "level": params['level'],
            "sourceList": params['sourceList'],
            "fromEditParentId": params['fromEditParentId']
        }
        push_data_org(
            SyncDataName.Input_IPC_GROUP_UPDATE, data)
    return jsonify(rs)


@ipc_bp.route('/IPCGroupNodeDelete', methods=['GET', 'POST'])
@api_log
def ipc_group_node_delete():
    """
    ipc源分组删除节点
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'groupId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, group_id = values

    # 获取当前登录用户的角色
    user = g.user
    role_id = user['roleId']
    role_rs = rolesvc.get_role_by_id(role_id)
    if role_rs['status'] == 0 and role_rs['data']:
        if role_rs['data']['role_type'] != 1:
            return jsonify(get_result_model(ResInfo.Role_Jurisdiction_Code))

    rs = ipc_svc.ipc_group_node_delete(device_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Input_IPC_GROUP_NODE_DELETE, params)
    return jsonify(rs)


# 拼接源 开始
@ipc_bp.route('/IPCMontageList', methods=['GET', 'POST'])
def ipc_montage_list_api():
    """ipc拼接源列表"""
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    user = g.user
    role_id = user['roleId']
    rs = ipc_montage_list(device_id)
    if rs['status'] == 0:
        role_rs, role_type = role_filter_data_func(role_id)
        rs['data'] = _filter_ipc_data(role_type, role_rs['data']['inputList'], rs['data'])
    return jsonify(rs)


@ipc_bp.route('/IPCTemplateDetail', methods=['GET', 'POST'])
def ipc_montage_template_detail():
    """ipc模板详情"""
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(
        params,
        'deviceId',
        'templateId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, template_id = values
    rs = ipc_svc.ipc_montage_template_detail(device_id, template_id)
    return jsonify(rs)


@ipc_bp.route('/IPCMontageTemplateCreate', methods=['GET', 'POST'])
def ipc_montage_template_create():
    """
    ipc拼接源创建模板
    :return:
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'deviceId', 'slotId', 'montageId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))

    device_id, slot_id, montage_id = values
    params['createTime'] = str(time.time())
    rs = ipc_svc.ipc_montage_template_create(device_id, params)

    if rs['status'] == 0:
        drs = ipc_svc.ipc_montage_list(device_id, slot_id)
        if drs['status'] == 0:
            data = [x for x in drs['data']['montageList']
                    if x['montageId'] == montage_id][0]
            push_data_org(SyncDataName.Input_IPC_Montage_Save, data)
        push_data_org(SyncDataName.Input_IPC_Montage_Edit, {'deviceId': device_id})
    return jsonify(rs)


@ipc_bp.route('/IPCMontageTemplateUpdate', methods=['GET', 'POST'])
def ipc_montage_template_update():
    """
    ipc 拼接源 模板修改
    :return:
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(
        params,
        'deviceId',
        'slotId',
        'templateId',
        'montageId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, slot_id, template_id, montage_id = values
    rs = ipc_svc.ipc_montage_list(device_id, slot_id)
    if rs['status'] == 0 and rs['data'] and rs['data']['montageList']:
        montages = [x for x in rs['data']['montageList'] if x['montageId'] == montage_id]
        if not montages:
            return jsonify(get_result_model(ResInfo.Middle_Data_Err))
        templates = [x for x in montages[0]['templateList'] if x['templateId'] == template_id]
        if not templates:
            return jsonify(get_result_model(ResInfo.Middle_Data_Err))
        params['createTime'] = templates[0]['createTime'] if 'createTime' in templates[0] else str(time.time())
    else:
        return jsonify(get_result_model(ResInfo.Middle_Data_Err))

    rs = ipc_svc.ipc_montage_template_update(device_id, params)
    if rs['status'] == 0:
        drs = ipc_svc.ipc_montage_list(device_id, slot_id)
        if drs['status'] == 0:
            data = [x for x in drs['data']['montageList']
                    if x['montageId'] == montage_id][0]
            push_data_org(SyncDataName.Input_IPC_Montage_Save, data)
        push_data_org(SyncDataName.Input_IPC_Montage_Edit, {'deviceId': device_id})
    return jsonify(rs)


@ipc_bp.route('/IPCMontageTemplateDelete', methods=['GET', 'POST'])
def ipc_montage_template_delete():
    """
       ipc 拼接源 模板删除
       :return:
       """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(
        params,
        'deviceId',
        'slotId',
        'templateId',
        'montageId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, _, _, _ = values
    rs = ipc_svc.ipc_montage_template_delete(device_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Input_IPC_Montage_Template_Delete, params)
        push_data_org(SyncDataName.Input_IPC_Montage_Edit, {'deviceId': device_id})
    return jsonify(rs)


@ipc_bp.route('/IPCMontageTemplateApply', methods=['GET', 'POST'])
def ipc_montage_template_apply():
    """
    ipc 拼接源 模板应用
    :return:
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(
        params,
        'deviceId',
        'slotId',
        'templateId',
        'montageId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, _, _, _ = values
    rs = ipc_svc.ipc_montage_template_apply(device_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Input_IPC_Montage_Template_Apply, params)
        from H_9U.api.screen import screen_list
        push_data(
            screen_list,
            SyncDataName.Screen_Read_All_List,
            device_id)
    return jsonify(rs)


@ipc_bp.route('/IPCMontageRename', methods=['GET', 'POST'])
def ipc_montage_rename():
    """
    ipc拼接源重命名
    :return:
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = ipc_svc.ipc_montage_rename(device_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Input_IPC_Montage_Rename, params)
        from H_9U.api.screen import screen_list
        push_data(
            screen_list,
            SyncDataName.Screen_Read_All_List,
            device_id)
    return jsonify(rs)


@ipc_bp.route("/IPCStreamStatus", methods=['GET', 'POST'])
def ipc_stream_status():
    """
    读取ipc源拉流状态
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'deviceId', 'sourceType', 'id')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, source_type, biz_id = values
    stream_list = []
    # sourceType = 1  普通输入源  3.ipc 源  5.ipc 拼接源
    if source_type == 5:  # 模式1
        temp_rs = ipc_svc.ipc_montage_template_detail(device_id, biz_id)
        if temp_rs['status'] == 0 and temp_rs['data']:
            stream_list = [{'streamId': x['streamId']} for x in temp_rs['data']['streamList']]
    if source_type == 3:
        stream_list = [{"streamId": biz_id}]

    if stream_list:
        rs = ipc_svc.ipc_stream_detail(device_id, {'deviceId': device_id, 'streamList': stream_list})
        if rs['status'] == 0:
            rs['data']['sourceType'] = source_type
    else:
        rs = get_result_model()
        rs['data'] = {"deviceId": device_id, "sourceType": source_type, "streamList": []}
    return jsonify(rs)


@ipc_bp.route('/IPCStreamList', methods=['GET', 'POST'])
@api_log
def ipc_stream_list():
    """
    IPC数据源详情列表
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = ipc_svc.ipc_slot_stream_list(device_id)
    return jsonify(rs)



@ipc_bp.route('/IPCSlotSourceList', methods=['GET', 'POST'])
@api_log
def ipc_slot_source_list():
    """
    获取ipc下全部信号源
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'slotId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, slot_id = values
    rs = ipc_svc.ipc_slot_source_list(device_id, slot_id)
    return jsonify(rs)


@ipc_bp.route('/import', methods=['GET', 'POST'])
@api_log
def ipc_import_source():
    """
    导入ipc源
    """
    device_id = request.values['deviceId']
    try:
        # 从表单的file字段获取文件，file为该表单的name值
        f = request.files["file"]
        print("当前文件名称为=", f.filename)
        print("当前文件名称后缀为=", f.filename.rsplit('.', 1)[1].lower())
        if f.filename.rsplit('.', 1)[1].lower() != "csv":
            return get_result_model(ResInfo.Input_IPC_Import_File_Fmt_Err)
    except Exception as e:
        print(e)
        return get_result_model(ResInfo.Upload_File_Fail)
    rs = ipc_svc.ipc_import_source(int(device_id))
    return jsonify(rs)


@ipc_bp.route('/importProgress', methods=['GET', 'POST'])
@api_log
def ipc_import_progress():
    """
    查询导入进度
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'deviceId', 'importId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, import_id = values
    rs = ipc_svc.ipc_import_progress(import_id)
    if rs["status"] == 0:
        if rs["data"]["failFilePath"] != "":
            from H_9U.api.device import read_domain
            domain = read_domain(device_id)
            rs["data"]["failFilePath"] = domain + rs["data"]["failFilePath"]
        if rs["data"]["importStatus"] != IPCImportStatus.Importing:
            push_data_org(
                SyncDataName.Input_IPC_Source_Create, {
                    'deviceId': device_id})
    return jsonify(rs)


@ipc_bp.route('/cancelImport', methods=['GET', 'POST'])
@api_log
def ipc_cancel_import():
    """
    取消导入
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'deviceId', 'importId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, import_id = values
    rs = ipc_svc.ipc_cancel_import(import_id)
    push_data_org(
        SyncDataName.Input_IPC_Source_Create, {
            'deviceId': device_id})
    return jsonify(rs)


@ipc_bp.route('/export', methods=['GET', 'POST'])
@api_log
def ipc_export_source():
    """
    导出ipc源
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    from H_9U.api.device import read_domain, read_export_device_language
    language_rs = read_export_device_language(device_id)
    data = {
        "seqPageSize": IPCDefaultValue.Export_Source_Count,
        "seqPageIndex": 0,
        "deviceId": device_id,
        "language": language_rs['data']['languageMode']
    }
    rs = ipc_svc.ipc_export_source(device_id, data)
    if rs['status'] == 0:
        domain = read_domain(device_id)
        rs['data']['filePath'] = domain + rs['data']['filePath']
    return jsonify(rs)


@ipc_bp.route('/downloadTemplate', methods=['GET', 'POST'])
@api_log
def ipc_download_template():
    """
    下载ipc源
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = get_result_model()
    # 构建导出路径+文件名
    file_name = 'template.csv'
    target_file = os.path.join(SysSettings.Download_IPC_File_Path, file_name)
    if not os.path.exists(SysSettings.Download_IPC_File_Path):
        os.makedirs(SysSettings.Download_IPC_File_Path, mode=0o777, exist_ok=True)
    if not os.path.exists(target_file):
        try:
            with open(target_file, 'w', encoding='utf-8_sig', newline='') as f:
                f.writelines(IPCTemplate.template)
        except Exception as e:
            logger.exception('创建导出文件异常：%s' % e)
            return jsonify(get_result_model(ResInfo.Input_IPC_Create_Template_Err))
    file_url = target_file[target_file.index('download'):]
    domain = read_domain(device_id)
    rs['data']['filePath'] = domain + file_url
    return jsonify(rs)



@ipc_bp.route('/IPCSourceStatus', methods=['GET', 'POST'])
@api_log
def ipc_source_status_list():
    """
    获取ipc下全部信号源
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'groupId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, group_id = values
    rs = ipc_svc.ipc_source_status_list(device_id, group_id, params)
    return jsonify(rs)



def ipc_montage_list(device_id):
    """
    ipc 拼接源列表
    :param device_id:
    :return:
    """
    card_list = []
    slot_list = _get_ipc_card_list(device_id)
    if slot_list:
        from H_9U.api.device import ipc_out_mode_read
        for slot_id in slot_list:
            mode_rs = ipc_out_mode_read(device_id, slot_id)
            if mode_rs['status'] == 0 and mode_rs['data']['videoMode'] == 0:
                card_list.append(mode_rs['data'])
    montage_list = []
    for x in card_list:
        rs = ipc_svc.ipc_montage_list(device_id, x['slotId'])
        if rs['status'] == 0 and rs['data']:
            montage_list.append(rs['data'])
    rs = get_result_model()
    rs['data'] = {"deviceId": device_id, "slotList": montage_list}
    return rs


def ipc_slot_montage_list(device_id, slot_id):
    """
    ipc卡拼接源列表
    :param device_id:设备Id
    :param slot_id:槽位id
    :return:拼接源列表
    """
    rs = ipc_svc.ipc_montage_list(device_id, slot_id)
    return rs


def ipc_source_detail(device_id, source_id):
    """ipc拼接源详情"""
    rs = ipc_svc.ipc_source_detail(device_id, source_id)
    return rs


def ipc_channel_detail(device_id, source_id, channel_id):
    return ipc_svc.ipc_channel_detail(device_id, source_id, channel_id)


def ipc_slot_source_count(device_id, slot_id):
    """
    ipc slot 信号源梳理
    :param device_id:设备Id
    :param slot_id: slotId
    :return: 结果对象
    """
    return ipc_svc.ipc_slot_source_count(device_id, slot_id)


def _get_ipc_card_list(device_id):
    """
    获取所有ipc卡输入源
    :param device_id: 设备Id
    :return: upc输入源id数组
    """
    from H_9U.api.device import ipc_card_list
    rs = ipc_card_list(device_id)
    ipc_list = []
    if rs['status'] == 0:
        ipc_list = rs['data']
    return ipc_list


def _ipc_slot_source_count_list(device_id):
    slot_list = _get_ipc_card_list(device_id)
    rs = get_result_model()
    rs['data'] = {
        'deviceId': device_id,
        'ipcSlotList': []
    }
    if slot_list:
        from H_9U.api.device import ipc_out_mode_read
        for slot_id in slot_list:

            mode_rs = ipc_out_mode_read(device_id, slot_id)
            ipc_rs = ipc_slot_source_count(device_id, slot_id)
            if ipc_rs['status'] == 0 and mode_rs['status'] == 0:
                rs['data']['ipcSlotList'].append({
                    'slotId': slot_id,
                    'sourceCount': ipc_rs['data']['sourceCount'],
                    'videoMode': mode_rs['data']['videoMode']})
    return rs


def _get_stream_id(ms, slot_id, montage_id):
    stream_ids = []
    if not ms['data']['slotList']:
        return stream_ids
    ms1 = deepcopy(ms)
    montage_slot = {}
    for slot in ms1['data']['slotList']:
        if slot['slotId'] == slot_id:
            montage_slot = slot
            break
    for montage in montage_slot['montageList']:
        m_id = montage['montageId']
        template_list = montage['templateList']
        for template in template_list:
            for stream in template['streamList']:
                if montage_id != m_id:
                    stream_ids.append(stream['streamId'])
    print("被使用的 stream_ids =", stream_ids)
    stream_ids = set(stream_ids)
    return stream_ids


def _get_channel_free_data(rs, stream_ids):
    des = deepcopy(rs)
    for source in des['data']['sourceList']:
        channel_list = source['channelList']
        if not channel_list:
            continue
        for channel in channel_list:
            stream_list = channel['streamList']
            if not stream_list:
                continue
            for stream in stream_list:
                stream_id = stream['streamId']
                stream['isUsed'] = 0
                if stream_id in stream_ids:
                    stream['isUsed'] = 1
    return des


def get_role_montage_list_data(ipc_rs, role_id):
    """
    获取当前登录用户的输入列表数据
    :return: 无分组信息
    """
    slot_ids = []
    role_rs = rolesvc.get_role_by_id(role_id)
    if role_rs['status'] != 0 or not role_rs['data']:
        return ipc_rs
    role_type = role_rs['data']['role_type']
    if role_type == 1:
        return ipc_rs
    # 查询当前角色拥有的输入源权限
    rs = rolesvc.get_input_by_role_id(role_id)
    if rs['status'] == 0 and rs['data']:
        slot_ids = [x['inputId'] for x in rs['data']]
    if ipc_rs['status'] == 0 and ipc_rs['data']:
        all_message_list = []
        for all_message in ipc_rs['data']['slotList']:
            slot_id = all_message['slotId']
            if slot_id in slot_ids:
                all_message_list.append(all_message)
        ipc_rs['data']['slotList'] = all_message_list
    return ipc_rs


def get_role_ipc_list_data(ipc_rs, role_id):
    """
    获取当前登录用户的输入列表数据
    :return: 无分组信息
    """
    slot_ids = []
    role_rs = rolesvc.get_role_by_id(role_id)
    if role_rs['status'] != 0 or not  role_rs['data']:
        return ipc_rs
    role_type = role_rs['data']['role_type']
    if role_type == 1:
        return ipc_rs
    # 查询当前角色拥有的输入源权限
    rs = rolesvc.get_input_by_role_id(role_id)
    if rs['status'] == 0 and rs['data']:
        slot_ids = [x['inputId'] for x in rs['data']]
    if ipc_rs['status'] == 0 and ipc_rs['data']:
        all_message_list = []
        for all_message in ipc_rs['data']['ipcSlotList']:
            slot_id = all_message['slotId']
            if slot_id in slot_ids:
                all_message_list.append(all_message)
        ipc_rs['data']['ipcSlotList'] = all_message_list
    return ipc_rs

