"""
Created on 2019年6月28日
@author: 刘益宗
layer api
参考前端接口文档layer部分
返回值统一为 {data:JSON结构体, msg:"", status:0}
"""

from flask import Blueprint
from flask.globals import request
from flask.json import jsonify
from copy import deepcopy

from H_9U.service.layer import layer_svc
from H_9U.util.common import valid_json, valid_params
from H_9U.models.result import get_result_model, ResInfo
from H_9U.models.sysconst import LayerType, ConnectCapacity, SourceType
from H_9U.api.input import input_get_connect_capacity
from H_9U.util.logwrapper import api_log
from H_9U.api.websender import push_data, push_data_org
from H_9U.models.syncdataname import SyncDataName
layer_bp = Blueprint('layer', __name__)


@layer_bp.route('/detailList', methods=['GET', 'POST'])
@api_log
def layer_read_list():
    """
    获取指定screen的layer的枚举
    :return:参考前端接口文档readList
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values

    rs = screen_layer_detail_list(device_id, screen_id)
    return jsonify(rs)


@layer_bp.route('/readDetail', methods=['GET', 'POST'])
@api_log
def read_detail():
    """
    读取详细信息
    :return: 参考前端接口文档readDetail
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId', 'layerId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, layer_id = values

    rs = layer_read_detail(device_id, screen_id, layer_id)
    return jsonify(rs)


def layer_read_detail(device_id, screen_id, layer_id):
    """
    读取屏幕详细信息
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :param layer_id: 图层Id
    :return: 参考readDetail文档    """
    rs = screen_layer_detail_list(device_id, screen_id)
    if rs['status'] == 0 and rs['data']:
        detail = [x for x in rs['data'] if x['layerId'] == layer_id]
        if detail:
            rs['data'] = detail[0]
        else:
            rs['data'] = {}
    return rs


@layer_bp.route('/create', methods=['GET', 'POST'])
@api_log
def layer_create():
    """
    添加Layer
    :return:参考前端接口文档create
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values

    if not params['general']:
        return jsonify(get_result_model(ResInfo.Params_Error))

    input_id = params['source']['inputId']

    # 图层类型
    params['general']['type'] = LayerType.Default

    if params['source']['sourceType'] == SourceType.InputSource:
        # 输入接口容量：
        rs = input_get_connect_capacity(device_id, input_id)
        # 获取不到连接容量，直接返回错误
        if rs['status'] != 0:
            return jsonify(get_result_model(ResInfo.Middle_Data_Err))
    elif params['source']['sourceType'] == SourceType.IPC:  # 255 表示使用IPC源
        __build_ipc_data(device_id, params)

    params['general']['sizeType'] = ConnectCapacity.SL  # conn

    # 添加图层
    rs = layer_svc.create(device_id, screen_id, params)
    if rs['status'] != 0 or not rs['data']:
        return jsonify(rs)
    layer_id = rs['data']['id']
    nontime = _screen_nontime(device_id, screen_id)
    if nontime == 0:
        __build_nontime_data(device_id, layer_id, screen_id)
    elif nontime == 1:
        __build_time_data(device_id, layer_id, screen_id)
    return jsonify(rs)


def __build_time_data(device_id, layer_id, screen_id):
    from H_9U.api.screen import screen_non_time_detail
    details = screen_non_time_detail(device_id, screen_id)
    if details['status'] == 0:
        data = {}
        data['nonTime'] = details['data']['en_non_time']
        data['audio'] = details['data']['noTimeAudio']
        data['layerId'] = layer_id
        data['screenId'] = screen_id
        data['screenLayers'] = details['data']['layers']
        push_data_org(SyncDataName.Layer_Create, data)


def __build_nontime_data(device_id, layer_id, screen_id):
    from H_9U.api.screen import screen_detail
    details = screen_detail(device_id, screen_id)
    data = {}
    if details['status'] == 0 and details['data']:
        data['nonTime'] = details['data']['nonTime']
        data['audio'] = details['data']['audio']
        data['layerId'] = layer_id
        data['screenId'] = screen_id
        data['screenLayers'] = details['data']['screenLayers']
    push_data_org(SyncDataName.Layer_Create, data)


def __build_ipc_data(device_id, params):
    source_id = params['source']['ipcSourceId']
    from H_9U.api.ipc import ipc_top_channel
    rs = ipc_top_channel(device_id, source_id)
    if rs['status'] == 0 and rs['data']:
        name = '{0}_{1}_main'.format(rs['data']['sourceName'], rs['data']['channelName'])
        stream_id = [x for x in rs['data']['streamList'] if x['streamIndex'] == 0][0]['streamId']
        params['source']['sourceType'] = SourceType.IPCSource
        params['source']['streamId'] = stream_id
        params['source']['name'] = name


@layer_bp.route('/delete', methods=['GET', 'POST'])
@api_log
def layer_delete():
    """
    从屏幕中删除指定的layer
    :return:参考前端接口文档delete
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId', 'layerId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, layer_id = values
    nontime = _screen_nontime(device_id, screen_id)
    if nontime is None:
        rs = get_result_model(ResInfo.Middle_Data_Err)
    else:
        # 删除图层
        rs = layer_svc.delete(device_id, screen_id, layer_id)
    # 推送图层
    if rs['status'] == 0:
        push_data_org(SyncDataName.Layer_Delete, {
            "deviceId": device_id,
            "screenId": screen_id,
            "layerId": layer_id
        })

    return jsonify(rs)


@layer_bp.route('/deleteBatch', methods=['GET', 'POST'])
@api_log
def layer_delete_batch():
    """
    批量删除窗口
    :return: 参考前端接口文档deleteBatch
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId', 'model')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, model = values
    from H_9U.api.screen import screen_clear_layers
    rs = screen_clear_layers(device_id, screen_id)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Layer_Delete_All, {'deviceId': device_id, 'screenId': screen_id, 'layers': []})
    return jsonify(rs)


@layer_bp.route('/writeGeneral', methods=['GET', 'POST'])
@api_log
def layer_write_general():
    """
    设置Layer基础信息
    :return: 参考文档writeGeneral
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId', 'layerId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, layer_id = values
    rs = layer_svc.general_write(device_id, screen_id, layer_id, params)
    # 推送
    if rs['status'] == 0:
        nontime = _screen_nontime(device_id, screen_id)
        if nontime == 0:
            push_data(layer_read_detail, SyncDataName.Layer_Write_General, device_id, screen_id, layer_id)
        elif nontime == 1:
            from H_9U.api.screen import screen_non_time_detail
            rs = screen_non_time_detail(device_id, screen_id)
            if rs['status'] == 0:
                layers = [x for x in rs['data']['layers'] if x['layerId'] == layer_id]
                if layers:
                    layer = layers[0]
                    layer['screenId'] = screen_id
                    layer['deviceId'] = device_id
                    push_data_org(SyncDataName.Layer_Write_General, layer)
    return jsonify(rs)


@layer_bp.route('/writeWindow', methods=['GET', 'POST'])
@api_log
def layer_write_layout_window():
    """
    设置Layer基础信息
    :return: 参考文档writeWindow
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId', 'layerId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, layer_id = values

    # 判断图层Id<0返回参数错误
    if layer_id < 0:
        return jsonify(get_result_model())
    rs = layer_svc.layout_window_write(device_id, screen_id, layer_id, params)
    # 推送窗口信息
    if rs['status'] == 0:
        push_data_org(SyncDataName.Layer_Write_Window, params)
    return jsonify(rs)


@layer_bp.route('/writeSource', methods=['GET', 'POST'])
@api_log
def layer_write_source():
    """
    设置Layer源
    :return:
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId', 'layerId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, layer_id = values

    name = params['name']
    if params['sourceType'] == SourceType.IPC:
        source_id = params['ipcSourceId']
        from H_9U.api.ipc import ipc_top_channel
        rs = ipc_top_channel(device_id, source_id)
        if rs['status'] == 0 and rs['data']:
            name = '{0}_{1}_main'.format(rs['data']['sourceName'], rs['data']['channelName'])
            stream_id = [x for x in rs['data']['streamList'] if x['streamIndex'] == 0][0]['streamId']
            params['sourceType'] = SourceType.IPCSource
            params['streamId'] = stream_id
            params['name'] = name

    rs = layer_svc.source_write(device_id, screen_id, layer_id, params)
    if rs['status'] != 0:
        return jsonify(rs)
    rs['data'] = {
        'deviceId': device_id,
        'screenId': screen_id,
        'layerId': layer_id,
        'sourceType': params['sourceType'],
        'name': name
    }
    nontime = _screen_nontime(device_id, screen_id)
    if nontime == 0:
        push_data(layer_read_detail, SyncDataName.Layer_Write_Source, device_id, screen_id, layer_id)
    elif nontime == 1:
        from H_9U.api.screen import screen_non_time_detail
        rs = screen_non_time_detail(device_id, screen_id)
        if rs['status'] == 0:
            layers = [x for x in rs['data']['layers'] if x['layerId'] == layer_id]
            if layers:
                layer = layers[0]
                layer['screenId'] = screen_id
                layer['deviceId'] = device_id
                push_data_org(SyncDataName.Layer_Write_Source, layer)
    if rs['status'] == 0:
        rs['data'] = {
            'deviceId': device_id,
            'screenId': screen_id,
            'layerId': layer_id,
            'sourceType': params['sourceType'],
            'name': name
        }
        nontime = _screen_nontime(device_id, screen_id)
        if nontime == 0:
            push_data(layer_read_detail, SyncDataName.Layer_Write_Source, device_id, screen_id, layer_id)
        elif nontime == 1:
            from H_9U.api.screen import screen_non_time_detail
            rs = screen_non_time_detail(device_id, screen_id)
            if rs['status'] == 0:
                layers = [x for x in rs['data']['layers'] if x['layerId'] == layer_id]
                if layers:
                    layer = layers[0]
                    layer['screenId'] = screen_id
                    layer['deviceId'] = device_id
                    push_data_org(SyncDataName.Layer_Write_Source, layer)
    return jsonify(rs)


@layer_bp.route('/readZIndex', methods=['GET', 'POST'])
@api_log
def layer_read_z_index():
    """
    读取屏幕图层Z index
    :return:  参考文档 readZIndex
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values

    return jsonify(layer_svc.z_order_read(device_id, screen_id))


@layer_bp.route('/writeZIndex', methods=['GET', 'POST'])
@api_log
def layer_write_z_index():
    """
    设置图层Z index
    :return: 参考文档 writeZIndex
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId', 'layerId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, layer_id = values
    rs = layer_svc.z_order_write(device_id, screen_id, params)
    if rs['status'] != 0:
        return jsonify(rs)
    nontime = _screen_nontime(device_id, screen_id)
    z_rs = None
    if nontime == 0:
        z_rs = layer_svc.z_order_read(device_id, screen_id)
    elif nontime == 1:
        from H_9U.api.screen import screen_non_time_detail
        rs = screen_non_time_detail(device_id, screen_id)
        if rs['status'] == 0:
            layers = [x for x in rs['data']['layers']]
            data = {"screenId": screen_id, "deviceId": device_id, "layersZOrder": []}
            for layer in layers:
                data["layersZOrder"].append({'layerId': layer['layerId'], 'zOrder': layer['general']['zorder']})
            z_rs = get_result_model()
            z_rs['data'] = data
    if z_rs and z_rs['status'] == 0:
        rs = z_rs
        # 推送
        push_data_org(SyncDataName.Layer_Write_Z_Index, rs['data'])
    return jsonify(rs)


def screen_zorder_read(device_id, screen_id):
    return layer_svc.z_order_read(device_id, screen_id)


@layer_bp.route('/writeLock', methods=['GET', 'POST'])
@api_log
def write_lock():
    """
    图层锁定，0 解锁，1 锁定
    :return: 参考接口文档writeLock
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId', 'layerId', 'lock')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, layer_id, is_lock = values
    nontime = _screen_nontime(device_id, screen_id)
    if nontime is not None:
        rs = layer_svc.write_lock(device_id, screen_id, layer_id, params)
    else:
        rs = get_result_model(ResInfo.Middle_Data_Err)
    if rs['status'] == 0:
        # 推送
        if nontime == 0:
            push_data(layer_read_detail, SyncDataName.Layer_Write_Lock, device_id, screen_id, layer_id)
        elif nontime == 1:
            from H_9U.api.screen import screen_non_time_detail
            rs = screen_non_time_detail(device_id, screen_id)
            if rs['status'] == 0:
                layers = [x for x in rs['data']['layers'] if x['layerId'] == layer_id]
                if layers:
                    layer = layers[0]
                    layer['screenId'] = screen_id
                    layer['deviceId'] = device_id
                    push_data_org(SyncDataName.Layer_Write_Lock, layer)
    return jsonify(rs)


@layer_bp.route('/screenLayerTake', methods=['GET', 'POST'])
@api_log
def screen_layer_take():
    """
    屏幕上屏参考接口文档screenLayerTake
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    from H_9U.api.screen import screen_notime_play
    rs = screen_notime_play(device_id, screen_id, params)
    if rs['status'] == 0:
        # 推送
        push_data_org(SyncDataName.Layer_Screen_Take, params)
    return jsonify(rs)


@layer_bp.route('/freeze', methods=['GET', 'POST'])
@api_log
def layer_freeze():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId', 'layerId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, layer_id = values
    rs = layer_svc.freeze(device_id, screen_id, layer_id, params)
    if rs['status'] == 0:
        nontime = _screen_nontime(device_id, screen_id)
        if nontime == 0:
            push_data(layer_read_detail, SyncDataName.Layer_Freeze, device_id, screen_id, layer_id)
        elif nontime == 1:
            from H_9U.api.screen import screen_non_time_detail
            rs = screen_non_time_detail(device_id, screen_id)
            if rs['status'] == 0:
                layers = [x for x in rs['data']['layers'] if x['layerId'] == layer_id]
                if layers:
                    layer = layers[0]
                    layer['screenId'] = screen_id
                    layer['deviceId'] = device_id
                    push_data_org(SyncDataName.Layer_Freeze, layer)
    return jsonify(rs)


@layer_bp.route('/writeLayerLockRatio', methods=['GET', 'POST'])
@api_log
def write_layer_lock_ratio():
    """
    图层锁定纵横比设置
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId', 'layerId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, layer_id = values
    rs = layer_svc.write_layer_lock_ratio(device_id, screen_id, layer_id, params)
    if rs['status'] == 0:
        nontime = _screen_nontime(device_id, screen_id)
        if nontime == 0:
            push_data(layer_read_detail, SyncDataName.Layer_Write_Lock_Ratio, device_id, screen_id, layer_id)
        elif nontime == 1:
            from H_9U.api.screen import screen_non_time_detail
            rs = screen_non_time_detail(device_id, screen_id)
            if rs['status'] == 0:
                layers = [x for x in rs['data']['layers'] if x['layerId'] == layer_id]
                if layers:
                    layer = layers[0]
                    layer['screenId'] = screen_id
                    layer['deviceId'] = device_id
                    push_data_org(SyncDataName.Layer_Write_Lock_Ratio, layer)
    return jsonify(rs)

def screen_layer_detail_list(device_id, screen_id):
    """
    屏幕图层列表
    :param device_id: 设备Id
    :param screen_id:  屏幕Id
    :return:  屏幕下所有图层
    """
    rs = layer_svc.layer_full_detail_list(device_id, screen_id)
    if rs['status'] == 0 and rs['data']:
        rs['data'] = rs['data']['screenLayers']
    return rs


@layer_bp.route('/screenLayerLayout', methods=['GET', 'POST'])
@api_log
def screen_layer_layout():
    """
    屏幕上屏参考接口文档screenLayerTake
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    layers = params.get('layers', None)
    if layers is None:
        return jsonify(get_result_model(ResInfo.Params_Error))

    rs = layer_svc.screen_layer_layout(device_id, screen_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Layer_Screen_Layout, params)
    return jsonify(rs)


@layer_bp.route('/writeAudio', methods=['GET', 'POST'])
@api_log
def api_layer_audio_status_write():
    """
    图层音频设置
    :return: 参考接口文档writeFrequency
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(
        params,
        'deviceId',
        'screenId',
        'layerId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, layer_id = values
    rs = layer_svc.screen_audio_status(device_id, screen_id, layer_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Layer_Write_Audio_Status, params)
    return jsonify(rs)



def screen_layers(device_id, screen_id):
    return layer_svc.read_list(device_id, screen_id)


def _screen_nontime(device_id, screen_id):
    from H_9U.api.screen import screen_detail
    rs = screen_detail(device_id, screen_id)
    if rs['status'] == 0:
        return rs['data']['nonTime']
