"""
Created on 2019年6月28日
@author: 刘益宗
main接口.
对应前端接口main
返回值统一为 {data:JSON结构体, msg:"", status:0}
"""
import json

from flask.globals import request, g
from flask.json import jsonify
from flask import Blueprint

from H_9U.models.sysconst import PermissionCode, HttpConfigType, PlanarConst, DeviceType
from H_9U.service.main import main_svc
from H_9U.util.common import valid_json, valid_params, device_safe_config_read
from H_9U.models.result import get_result_model, ResInfo
from H_9U.conf.syssettings import SysSettings
from H_9U import controller
from H_9U.util.logwrapper import api_log
from H_9U.service.device import device_svc
from H_9U.util.cache import cacher
from H_9U.api.device import device_detail, device_init_status
from H_9U.models.syncdataname import SyncDataName
from H_9U.api.websender import push_data_org
from H_9U.util.permission_valid import get_code_list

main_bp = Blueprint('main', __name__)


@main_bp.route('/writeLanguageMode', methods=['GET', 'POST'])
@api_log
def main_write_language_mode():
    """
    设置语言模式
    :return:
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'languageMode')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    language = values
    device_id = params.get('deviceId', 0)
    # 语言,0:英文；1：中文
    rs = device_svc.write_device_language(device_id, {'deviceId': device_id, 'lang': 'zh' if language == 1 else 'en'})
    return jsonify(rs)


@main_bp.route('/readLanguageMode', methods=['GET', 'POST'])
@api_log
def main_read_language_mode():
    """
    设置语言模式
    :return:
    """
    params = valid_json(request.get_data().decode())
    device_id = params.get('deviceId', 0)
    return jsonify(device_svc.read_device_language(device_id))


@main_bp.route('/readVideoServerInfo', methods=['GET', 'POST'])
@api_log
def main_read_video_server_info():
    """
    读取视频服务器地址
    :return:
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    mvr_url = SysSettings.MVR_URL
    echo_url = SysSettings.Echo_URL
    # 预监卡是否在线
    rs = device_svc.mvr_is_online(device_id)
    if rs['status'] == 0 and rs['data'] and rs['data']['online'] == 1:
        # 读取预监卡详细信息
        rs = device_svc.device_slot_read(device_id, rs['data']['slotId'])
        if rs['status'] == 0 and rs['data']:
            # 读取encoding信息
            mvr_url = rs['data']['encoding']['mvrUrl']
            echo_url = rs['data']['encoding']['echoUrl']
    rs = main_svc.read_video_status(device_id)
    video_status = 1
    if rs['status'] == 0 and rs['data']:
        video_status = rs['data']['videoStatus']
    rs = device_svc.switch_http_read(device_id, HttpConfigType.DeviceHttpConfig)
    if rs['status'] != 0 or not  rs['data']:
        return rs
    is_open_https = rs['data']['is_open']
    if is_open_https == 1:
        rs = device_svc.device_ip_read(device_id)
        if rs['status'] == 0 and rs['data']:
            ips = rs['data']['ip']
            ip = "{0}.{1}.{2}.{3}".format(ips['ip0'], ips['ip1'], ips['ip2'], ips['ip3'])
            mvr_url = "ws://" + ip + ":8082"
            echo_url = "ws://" + ip + ":8083"
    rs = get_result_model()
    rs['data'] = {
        'mvrUrl': mvr_url,
        'echoUrl': echo_url,
        'videoStatus': video_status
    }
    return jsonify(rs)


@main_bp.route('/writeVideoStatus', methods=['GET', 'POST'])
@api_log
# @permission_required(PermissionName.OTHER_SETTING)
def main_write_video_status():
    """
    设置语言模式
    :return:
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'videoStatus')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, video_status = values

    user = g.user
    role_id = user['roleId']
    # 增加接口校验
    role_rs = get_code_list(PermissionCode.Device_Other, role_id)
    if role_rs['status'] != 0:
        return jsonify(role_rs)
    # 增加接口校验
    role_rs = get_code_list(PermissionCode.Edit_Stream, role_id)
    if role_rs['status'] != 0:
        return jsonify(role_rs)

    rs = main_svc.write_video_status(device_id, video_status)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Video_Status, params)
    return jsonify(rs)


@main_bp.route('/initStatus', methods=['GET', 'POST'], endpoint='init')
@api_log
def main_init_status():
    """
    读取系统初始化状态
    :return: 初始化状态和语言模式
    """
    params = valid_json(request.get_data().decode())
    device_id = SysSettings.Device_Id
    if params:
        values = valid_params(params, 'deviceId')
        if values is not None:
            device_id = values
    status = controller.getInitStatus()
    if not status:
        return jsonify(get_result_model(ResInfo.Device_Is_Initing))
    # socket初始化成功
    rs = get_result_model()
    rs['data']['initStatus'] = 1
    if SysSettings.Local_Device_Type == 1:  # H系列
        # 读取initstatus接口，失败直接返回
        init_rs = device_init_status(SysSettings.Device_Id)
        if init_rs['status'] != 0:
            return jsonify(get_result_model(ResInfo.Device_Is_Initing))
        rs['data']['initDetail'] = init_rs['data']
        if init_rs['data']['rate'] < 100:
            rs['data']['initStatus'] = 0
            return jsonify(rs)
    # 其他数据
    language_rs = device_svc.read_device_language(device_id)
    if language_rs['status'] == 0 and language_rs['data']:
        rs['data']['languageMode'] = language_rs['data']['languageMode']
    # 处理主控id
    device_rs = device_detail(device_id)
    if device_rs['status'] == 0 and device_rs['data']:
        model_id = device_rs['data']['modelId']
        rs['data']['mainModelId'] = model_id
        if model_id in DeviceType.P_Main_All:
            rs['data']['bkgLimit'] = PlanarConst.Bkg_And_Osd_Max_Count[model_id]
            rs['data']['osdLimit'] = PlanarConst.Bkg_And_Osd_Max_Count[model_id]
            rs['data']['versionType'] = SysSettings.Version_Type
    # 处理中性模式
    rs['data']['nomarkMode'] = {}
    nomark_rs = _nomark_mode_read(device_id)
    if nomark_rs['status'] == 0:
        rs['data']['nomarkMode'] = nomark_rs['data']
    rs['data']['softwareVersion'] = SysSettings.SoftwareVersion
    # 读取安全模式配置文件 SysSettings.Middleware_Safe_Config_Path
    rs['data']['safe'] = device_safe_config_read(SysSettings.Middleware_Safe_Config_Path)
    rs_http = device_svc.switch_http_read(device_id, HttpConfigType.DeviceHttpConfig)
    web_socket_port = SysSettings.WebSocket_Port
    if rs_http['status'] == 0 and rs_http['data'] and rs_http['data']['is_open'] == 1:
        web_socket_port = SysSettings.HTTPSWebSocketPort
    rs['data']['port'] = web_socket_port
    return jsonify(rs)


@main_bp.route('/softwareVersion', methods=['GET', 'POST'])
@api_log
def main_software_version():
    """
    读取软件版本号
    :return: 版本号
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    version = SysSettings.SoftwareVersion
    rs = get_result_model()
    rs['data'] = {
        'deviceId': device_id,
        'version': version
    }
    return jsonify(rs)


@main_bp.route('/cacheClear', methods=['GET', 'POST'])
@api_log
def main_cache_clear():
    try:
        cacher.clear_all()
    except Exception as e:
        print(e)
    return jsonify(get_result_model())


@main_bp.route('/readNomarkMode', methods=['GET', 'POST'])
@api_log
def main_read_nomark_mode():
    """
    设置语言模式
    :return:
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    rs = _nomark_mode_read(device_id)
    return jsonify(rs)


@main_bp.route('/writeNomarkMode', methods=['GET', 'POST'])
@api_log
def main_write_nomark_mode():
    """
    设置语言模式
    :return:
    """
    device_id = request.form.get('deviceId')
    nomark_mode = request.form.get('nomarkMode')

    if device_id is None or nomark_mode is None:
        return jsonify(get_result_model(ResInfo.Params_Error))

    device_id, nomark_mode = int(device_id), int(nomark_mode)

    custom_list = None
    lang_list = None
    if nomark_mode == 2:  # 定制商标
        custom_list = request.form.get('customList')
        if custom_list:
            custom_list = json.loads(custom_list)
        lang_list = request.form.get('langList')
        if lang_list:
            lang_list = json.loads(lang_list)

    rs = device_svc.nomark_mode_write(device_id, nomark_mode, custom_list, lang_list)
    if rs['status'] == 0:
        result = _nomark_mode_read(device_id)
        if result['status'] == 0:
            push_data_org(SyncDataName.Nomark_Data, result['data'])

        # 设置成功则设备重启
        # opt_type: 0：设备重启；1：设备关机；2：开机
        device_svc.device_shutdown_write(device_id, 0)
    return jsonify(rs)


@main_bp.route('/writePackagePwd', methods=['GET', 'POST'])
@api_log
def main_write_package_pwd():
    """
    更新升级包密码
    :return:
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'packagePwd')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, package_pwd = values
    rs = main_svc.write_package_pwd(package_pwd)
    return jsonify(rs)


def _nomark_mode_read(device_id):
    return device_svc.nomark_mode(device_id)


def init_status(device_id=0):
    """
    socket通信controller状态
    :return:True-初始化完毕、False-初始化进行中
    """
    status = controller.getInitStatus()
    if status and SysSettings.Local_Device_Type == 1:  # H系列
        rs = device_init_status(device_id)
        if rs['status'] != 0 or rs['data']['rate'] < 100:
            status = False
    return status


