"""
Created on 2019年6月28日
@author: 刘益宗
mvrwindow接口.
对应前端接口mvrwindow
返回值统一为 {data:JSON结构体, msg:"", status:0}
"""
from flask import Blueprint, g
from flask import jsonify
from flask.globals import request

from H_9U.api.device import mvr_is_online
from H_9U.service.mvr import mvr_svc
from H_9U.service.mvrwindow import mvr_window_svc
from H_9U.util.common import valid_json, valid_params
from H_9U.models.result import get_result_model, ResInfo
from H_9U.api.screen import screen_general_list
from H_9U.api.input import input_read_all_list
from H_9U.api.websender import push_data, push_data_org
from H_9U.models.syncdataname import SyncDataName
from H_9U.util.logwrapper import api_log
from H_9U.models.sysconst import DataType, PermissionCode
from H_9U.util.permission_valid import valid_function_permission, get_code_list

mvr_window_bp = Blueprint('mvr_window', __name__)


@mvr_window_bp.route('/readList', methods=['GET', 'POST'])
@api_log
def mvr_window_read_list():
    """
    读取mvr window 列表
    :return: 参考前端接口文档readList
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    mvr_id = params.get('MVRId', 0)

    return jsonify(mvr_window_svc.mvr_window_list_read(device_id, mvr_id))


@mvr_window_bp.route('/readAll', methods=['GET', 'POST'])
@api_log
def mvr_window_read_all():
    """
    读取mvr window 列表
    :return: 参考前端接口文档readAll
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    mvr_id = params.get('MVRId', 0)

    return jsonify(mvr_read_all_window(device_id, mvr_id))


@mvr_window_bp.route('/readAllInfo', methods=['GET', 'POST'])
@api_log
def mvr_window_read_all_info():
    """
    读取mvr window 列表
    :return: 参考前端接口文档readAllInfo
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    mvr_id = params.get('MVRId', 0)
    return jsonify(mvr_window_all_info(device_id, mvr_id))


@mvr_window_bp.route('/readInfo', methods=['GET', 'POST'])
@api_log
def mvr_window_read_info():
    """
    读取mvr window 列表
    :return: 参考前端接口文档readAllInfo
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    mvr_id = params.get('MVRId', 0)
    rs = mvr_window_read_info(device_id, mvr_id)
    return jsonify(rs)


def mvr_window_read_info(device_id, mvr_id):
    """
    读取mvrWindow信息
    :param device_id: 设备Id
    :param mvr_id:  mvrId
    :return:  mvrWindow信息
    """
    rs = mvr_read_detail(device_id, mvr_id)
    if rs['status'] == 0 and rs['data']:
        # mvr下window信息
        windows = []
        window_rs = mvr_read_all_window(device_id, mvr_id)
        if window_rs['status'] == 0 and window_rs['data']:
            windows = window_rs['data']
        rs['data']['windows'] = windows

        # 全部输出入信息
        inputs = []
        input_rs = input_read_all_list(device_id)
        if input_rs['status'] == 0 and input_rs['data']:
            inputs = input_rs['data']
        rs['data']['inputs'] = inputs

        # 读取全部屏幕信息
        screens = []
        screen_rs = screen_general_list(device_id)
        if screen_rs['status'] == 0 and screen_rs['data']:
            screens = screen_rs['data']
        rs['data']['screens'] = screens
    return rs

def mvr_read_detail(device_id, mvr_id):
    """
    读取mvr详情
    :param device_id: 设备Id
    :param mvr_id: MVRId
    :return: mvr详细信息
    """
    rs = mvr_svc.mvr_detail_read(device_id, mvr_id)
    # 补充mvr是否在线信息
    if rs['status'] == 0 and rs['data']:
        rs['data']['online'] = mvr_is_online(device_id)['data']['online']
    return rs


def mvr_window_all_info(device_id, mvr_id=0):
    """
    读取mvrWindow全部信息
    :param device_id:  设备Id
    :param mvr_id: mvrid
    :return:  mvrWindow全部信息
    """
    rs = mvr_read_detail(device_id, mvr_id)
    if rs['status'] == 0 and rs['data']:
        # mvr下window信息
        windows = []
        window_rs = mvr_read_all_window(device_id, mvr_id)
        if window_rs['status'] == 0 and window_rs['data']:
            windows = window_rs['data']
        rs['data']['windows'] = windows
    return rs


@mvr_window_bp.route('/readDetail', methods=['GET', 'POST'])
@api_log
def read_detail():
    """
    读取窗口详细信息
    :return: 参考前端接口文档readDetail
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'windowId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, window_id = values

    mvr_id = params.get('MVRId', 0)
    rs = mvr_window_detail(device_id, mvr_id, window_id)
    return jsonify(rs)


def mvr_window_detail(device_id, mvr_id, window_id):
    """
    窗口详情信息
    :param device_id: 设备id
    :param mvr_id:  mvrId
    :param window_id:  windowId
    :return: 窗口详细信息
    """
    return mvr_window_svc.mvr_window_detail_read(device_id, mvr_id, window_id)


@mvr_window_bp.route('/create', methods=['GET', 'POST'])
@api_log
def mvr_window_create():
    """
    创建窗口
    :return: 返回结果Id
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    mvr_id = params.get('MVRId', 0)

    user = g.user
    role_id = user['roleId']
    # 接口权限校验 设置Screen信息  code = 3  DataType = 5
    role_rs = get_code_list(PermissionCode.Pre_Supervision, role_id)
    if role_rs['status'] != 0:
        return jsonify(role_rs)

    window_type = params.get('windowType')
    if window_type == 1:
        screen_id = params.get('srcId')
        # 接口权限校验 删除Screen  code = 17  DataType = 4
        screen_id_list = [screen_id]
        rs = valid_function_permission(PermissionCode.Pre_Supervision, DataType.Screen_Type, screen_id_list)
        if rs['status'] != 0:
            return jsonify(rs)
    if window_type == 0:
        slot_id =  params.get('slotId')
        slot_id_list = [slot_id]
        rs = valid_function_permission(PermissionCode.Pre_Supervision, DataType.Input_Type, slot_id_list)
        if rs['status'] != 0:
            return jsonify(rs)

    rs = mvr_window_svc.mvr_window_create(device_id, mvr_id, params)
    if rs['status'] == 0:
        window_id = rs['data']['id']
        w_rs = mvr_window_detail(device_id, mvr_id, window_id)
        if w_rs['status'] == 0 and w_rs['data']:
            rs['data']['window'] = w_rs['data']
            push_data_org(SyncDataName.MVR_Window_Create, w_rs['data'])

    return jsonify(rs)


# 删除窗口
@mvr_window_bp.route('/delete', methods=['GET', 'POST'])
@api_log
def mvr_window_delete():
    """
    删除mvr窗口
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'windowId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, window_id = values

    mvr_id = params.get('MVRId', 0)
    user = g.user
    role_id = user['roleId']
    # 接口权限校验 设置Screen信息  code = 3  DataType = 5
    role_rs = get_code_list(PermissionCode.Pre_Supervision, role_id)
    if role_rs['status'] != 0:
        return jsonify(role_rs)

    rs = mvr_window_svc.mvr_window_delete(device_id, mvr_id, window_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.MVR_Window_Delete, {'deviceId': device_id, 'MVRId': mvr_id, 'windowId': window_id})

    return jsonify(rs)


@mvr_window_bp.route('/deleteBatch', methods=['GET', 'POST'])
@api_log
def mvr_window_delete_batch():
    """
    批量删除窗口
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'model')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id,  model = values
    mvr_id = params.get('MVRId', 0)
    window_id_list = params.get('windowIdList')
    user = g.user
    role_id = user['roleId']
    # 接口权限校验 设置Screen信息  code = 3  DataType = 5
    role_rs = get_code_list(PermissionCode.Pre_Supervision, role_id)
    if role_rs['status'] != 0:
        return jsonify(role_rs)

    if model not in [0, 1]:
        return get_result_model(ResInfo.Params_Error)
    # 删除Id校验
    if model == 1 and not window_id_list:
        # 不指定Id返回参数错误
        return get_result_model(ResInfo.Params_Error)
    rs = mvr_window_svc.mvr_window_delete_batch(device_id, mvr_id, model, window_id_list)
    if rs['status'] == 0 or rs['status'] == ResInfo.MVR_Window_Batch_Delete_Partial_Success.value:
        data = [x for x in rs['data'] if x['status'] == 0]
        if data:
            push_data_org(SyncDataName.MVR_Window_Delete_Batch, data)
    return jsonify(rs)


@mvr_window_bp.route('/readEnable', methods=['GET', 'POST'])
@api_log
def mvr_window_read_general():
    """
    读取基本信息
    :return: 参考前端接口文档readGeneral
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'windowId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, window_id = values
    mvr_id = params.get('MVRId', 0)

    return jsonify(mvr_window_svc.mvr_window_general_read(device_id, mvr_id, window_id))


@mvr_window_bp.route('/writeGeneral', methods=['GET', 'POST'])
@api_log
def mvr_window_write_general():
    """
    设置基本信息
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'windowId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, window_id = values
    mvr_id = params.get('MVRId', 0)
    user = g.user
    role_id = user['roleId']
    role_rs = get_code_list(PermissionCode.Pre_Supervision, role_id)
    if role_rs['status'] != 0:
        return jsonify(role_rs)
    rs = mvr_window_svc.mvr_window_general_write(device_id, mvr_id, window_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.MVR_Window_Write_General, params)
    return jsonify(rs)


@mvr_window_bp.route('/writeUmdEnable', methods=['GET', 'POST'])
@api_log
def mvr_window_write_umd_general():
    """
    设置umd信息
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'windowId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, window_id = values
    mvr_id = params.get('MVRId', 0)

    user = g.user
    role_id = user['roleId']
    role_rs = get_code_list(PermissionCode.Pre_Supervision, role_id)
    if role_rs['status'] != 0:
        return jsonify(role_rs)

    return jsonify(mvr_window_svc.mvr_window_umd_enable_write(device_id, mvr_id, window_id, params))


@mvr_window_bp.route('/writeSource', methods=['GET', 'POST'])
@api_log
def mvr_window_write_source():
    """
    预监切源，writeSource
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'windowId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, window_id = values
    mvr_id = params.get('MVRId', 0)
    window_type = params.get('windowType')
    src_id = params.get('srcId')
    # 0：input 3:IPC
    if window_type == 0 or window_type == 3:
        from H_9U.api.input import get_slot_by_input_id
        slot_list = get_slot_by_input_id(src_id)
        rs = valid_function_permission(PermissionCode.Pre_Supervision, DataType.Input_Type, slot_list)
        if rs['status'] != 0:
            return jsonify(rs)
    # 1: 屏幕
    if window_type == 1:
        screen_id_list = [src_id]
        rs = valid_function_permission(PermissionCode.Pre_Supervision, DataType.Screen_Type, screen_id_list)
        if rs['status'] != 0:
            return jsonify(rs)
    rs = mvr_window_svc.mvr_window_write_source(device_id, mvr_id, window_id, params)
    if rs['status'] == 0:
        push_data(mvr_window_detail, SyncDataName.MVR_Window_Write_Source, device_id, mvr_id, window_id)
    return jsonify(rs)


@mvr_window_bp.route('/writeInfo', methods=['GET', 'POST'])
@api_log
def mvr_window_write_info():
    """
    预监切源，writeInfo
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'windowId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, window_id = values
    mvr_id = params.get('MVRId', 0)

    user = g.user
    role_id = user['roleId']
    # 接口权限校验 设置Screen信息  code = 3  DataType = 5
    role_rs = get_code_list(PermissionCode.Pre_Supervision, role_id)
    if role_rs['status'] != 0:
        return jsonify(role_rs)
    rs = mvr_window_svc.mvr_window_write_info(device_id, mvr_id, window_id, params)
    return jsonify(rs)


@mvr_window_bp.route('/createBatch', methods=['GET', 'POST'])
@api_log
def mvr_window_create_batch():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    mvr_id = params.get('MVRId', 0)
    windows = params.get('windows', 0)
    user = g.user
    role_id = user['roleId']
    # 接口权限校验 设置Screen信息  code = 3  DataType = 5
    role_rs = get_code_list(PermissionCode.Pre_Supervision, role_id)
    if role_rs['status'] != 0:
        return jsonify(role_rs)
    for window in windows:
        window_type = window['windowType']
        if window_type == 1:
            screen_id = window['srcId']
            # 接口权限校验 删除Screen  code = 17  DataType = 4
            screen_id_list = [screen_id]
            rs = valid_function_permission(PermissionCode.Pre_Supervision, DataType.Screen_Type, screen_id_list)
            if rs['status'] != 0:
                return jsonify(rs)
        if window_type == 0:
            windows = params.get('windows')
            for window in windows:
                input_id = window['srcId']
                from H_9U.api.input import  get_slot_by_input_id
                slot_list = get_slot_by_input_id(input_id)
                rs = valid_function_permission(PermissionCode.Pre_Supervision, DataType.Input_Type, slot_list)
                if rs['status'] != 0:
                    return jsonify(rs)
    rs = mvr_window_svc.mvr_window_create_batch(device_id, mvr_id, params)
    push_data(mvr_read_all_window, SyncDataName.MVR_Window_Read_All, device_id, mvr_id)

    return jsonify(rs)


def mvr_read_all_window(device_id, mvr_id=0):
    rs = mvr_window_svc.mvr_window_all_read(device_id, mvr_id)
    return rs

