"""
Created on 2021年9月07日
@author: 程声清
Input接口.
对应前端接口Input
返回值统一为 {body:加密串, msg:"", sign:"", status:0}
"""
from flask import Blueprint, g
from flask.json import jsonify
from H_9U.models.result import get_result_model, ResInfo
from H_9U.service import open_input_svc
from H_9U.util.common import get_params
from H_9U.util.logwrapper import open_log
from H_9U.util.request_handle import response_parameter_resolution
from H_9U.api.sync_queue import add_to_web
from H_9U.models.syncdataname import SyncDataName

open_input_bp = Blueprint('open_input', __name__)


@open_input_bp.route('/readList', methods=['GET', 'POST'])
@open_log
def input_read_list():
    """
    输入列表
    :return: 无分组信息
    """
    params = g.data
    values = get_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = open_input_svc.input_read_list(device_id)
    return jsonify(response_parameter_resolution(rs))


@open_input_bp.route('/readDetail', methods=['GET', 'POST'])
@open_log
def input_read_detail():
    """
    读取input详细信息
    :return:参考文档 readDetail
    """
    params = g.data
    values = get_params(params, 'deviceId', 'inputId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, input_id = values
    rs = open_input_svc.input_detail(device_id, input_id)
    return jsonify(response_parameter_resolution(rs))


@open_input_bp.route('/rename', methods=['GET', 'POST'])
@open_log
def input_rename():
    """
    重命名
    :return: 参考文档writeGeneral
    """
    params = g.data
    values = get_params(params, 'deviceId', 'inputId', 'name')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, input_id, name = values
    rs = open_input_svc.input_rename(device_id, input_id, params)
    # 修改输入源名称，推送输入源列表和屏幕列表
    if rs['status'] == 0:
        add_to_web(SyncDataName.Input_Read_List, None, device_id)
        add_to_web(SyncDataName.Screen_Read_All_List, None, device_id)
    return jsonify(response_parameter_resolution(rs))


@open_input_bp.route('/writeCrop', methods=['GET', 'POST'])
@open_log
def input_crop_create():
    """
    创建输入截取
    :return: 参考文档writeCrop
    """
    params = g.data
    values = get_params(params, 'deviceId', 'inputId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, input_id = values
    rs = open_input_svc.input_crop_create(device_id, input_id, params)
    # 新增截取，推送输入源列表
    if rs['status'] == 0:
        add_to_web(SyncDataName.Input_Read_List, None, device_id)
    return jsonify(response_parameter_resolution(rs))


@open_input_bp.route('/deleteCrop', methods=['GET', 'POST'])
@open_log
def input_crop_delete():
    """
    删除指定input的截取信息
    :return: 参考文档deleteCrop
    """
    params = g.data
    values = get_params(params, 'deviceId', 'inputId', 'cropId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, input_id, crop_id = values
    rs = open_input_svc.input_crop_delete(
        device_id, input_id, crop_id, params)
    # 删除截取，推送输入源列表和屏幕列表
    if rs['status'] == 0:
        add_to_web(SyncDataName.Input_Read_List, None, device_id)
        add_to_web(SyncDataName.Screen_Read_All_List, None, device_id)
    return jsonify(response_parameter_resolution(rs))


@open_input_bp.route('/updateCrop', methods=['GET', 'POST'])
@open_log
def input_crop_update():
    """
    创建输入截取
    :return: 参考文档writeCrop
    """
    params = g.data
    values = get_params(params, 'deviceId', 'inputId', 'cropId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, input_id, crop_id = values
    rs = open_input_svc.input_crop_update(
        device_id, input_id, crop_id, params)
    # 删除截取，推送输入源列表和屏幕列表
    if rs['status'] == 0:
        add_to_web(SyncDataName.Input_Read_List, None, device_id)
        add_to_web(SyncDataName.Screen_Read_All_List, None, device_id)
    return jsonify(response_parameter_resolution(rs))


@open_input_bp.route('/importEDID', methods=['GET', 'POST'])
@open_log
def input_ed_id_import():
    """
    导入edie文件
    :return: 结果对象
    """
    params = g.data
    values = get_params(params, 'deviceId', 'inputId', 'file', 'fileName')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, input_id, file_data, file_name = values
    rs = open_input_svc.input_ed_id_import(
        int(device_id), input_id, file_data, file_name)
    return jsonify(response_parameter_resolution(rs))


@open_input_bp.route('/writeTiming', methods=['GET', 'POST'])
@open_log
def input_write_timing():
    """
    设置Timing
    :return: 参考文档writeTiming
    """
    params = g.data
    values = get_params(params, 'deviceId', 'inputId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, input_id = values
    rs = open_input_svc.input_write_timing(device_id, input_id, params)
    # 修改timing，推送输入源列表和屏幕列表
    if rs['status'] == 0:
        add_to_web(SyncDataName.Input_Read_List, None, device_id)
    return jsonify(response_parameter_resolution(rs))


@open_input_bp.route('/writeResolution', methods=['GET', 'POST'])
@open_log
def input_write_resolution():
    """
    设置刷新率
    :return: 参考文档writeResolution
    """
    params = g.data
    values = get_params(params, 'deviceId', 'inputId', 'refreshRate', 'width', 'height')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, input_id, refresh, width, height = values
    rs = open_input_svc.input_detail(device_id, input_id)
    if rs['status'] == 0:
        conn = rs['data']["general"]['connectCapacity']
        rs = open_input_svc.input_write_resolution(device_id, input_id, refresh, width, height, conn)
        # 修改timing，推送输入源列表和屏幕列表
        if rs['status'] == 0:
            add_to_web(SyncDataName.Input_Read_List, None, device_id)
    return jsonify(response_parameter_resolution(rs))


@open_input_bp.route('/groupCreate', methods=['GET', 'POST'])
@open_log
def input_group_create():
    """
    创建输入分组
    :return: 参考文档groupCreate
    """
    params = g.data
    values = get_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = open_input_svc.input_group_create(device_id, params)
    # 创建分组，图送分组数据
    if rs['status'] == 0:
        g_id = rs['data']['id']
        add_to_web(SyncDataName.Input_Group_Data, None, device_id, g_id)
    return jsonify(response_parameter_resolution(rs))


@open_input_bp.route('/groupUpdate', methods=['GET', 'POST'])
@open_log
def input_group_update():
    """
    输入分组更新
    :return: 参考文档groupUpdate
    """
    params = g.data
    values = get_params(params, 'deviceId', 'inputGroupId', 'inputGroupName')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, input_group_id, input_group_name = values
    rs = open_input_svc.input_group_update(
        device_id, input_group_id, input_group_name, params)
    # 推送
    if rs['status'] == 0:
        add_to_web(SyncDataName.Input_Group_Data, None, device_id, input_group_id)
    return jsonify(response_parameter_resolution(rs))


@open_input_bp.route('/groupDelete', methods=['GET', 'POST'])
@open_log
def input_group_delete():
    """
    输入分组删除
    :return: 参考文档groupDelete
    """
    params = g.data
    values = get_params(params, 'deviceId', 'inputGroupId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, input_group_id = values
    rs = open_input_svc.input_group_delete(
        device_id, input_group_id, params)
    # 推送
    if rs['status'] == 0:
        add_to_web(SyncDataName.Input_Group_Delete, {"deviceId": device_id, "inputGroupId": input_group_id})
    return jsonify(response_parameter_resolution(rs))


@open_input_bp.route('/groupNodeDelete', methods=['GET', 'POST'])
@open_log
def input_group_node_delete():
    """
    分组删除输入源
    :return: 参考文档groupNodeDelete
    """
    params = g.data
    values = get_params(params, 'deviceId', 'inputGroupId', 'inputId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, input_group_id, input_id = values
    rs = open_input_svc.input_group_node_delete(
        device_id, input_group_id, input_id, params)
    # 推送
    if rs['status'] == 0:
        add_to_web(SyncDataName.Input_Group_Node_Delete,
                   {"deviceId": device_id, "inputGroupId": input_group_id, 'inputId': input_id})
    return jsonify(response_parameter_resolution(rs))
