"""
Created on 2021年11月01日
@author: 程声清
ipc api
参考前端接口文档ipc部分
返回值统一为 {body:加密串, msg:"", sign:"", status:0}
"""

from flask import Blueprint, g
from flask.json import jsonify

from H_9U.models.result import get_result_model, ResInfo
from H_9U.service import open_ipc_svc
from H_9U.service.network import network_svc
from H_9U.service.open_ipc_svc import ipc_channel_detail, ipc_channel_detail_list
from H_9U.util.common import get_params
from H_9U.util.logwrapper import open_log
from H_9U.util.request_handle import response_parameter_resolution
from H_9U.api.sync_queue import add_to_web
from H_9U.models.syncdataname import SyncDataName

open_ipc_bp = Blueprint('open_ipc', __name__)


@open_ipc_bp.route('/IPCSourceList', methods=['GET', 'POST'])
@open_log
def ipc_read_source_list():
    """
    读取指定IPCSource列表
    :return:
    """
    params = g.data
    values = get_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    params['seqPageIndex'] -= 1
    rs = open_ipc_svc.ipc_read_source_list(device_id, params)
    return jsonify(response_parameter_resolution(rs))


@open_ipc_bp.route('/IPCCameraList', methods=['GET', 'POST'])
@open_log
def ipc_read_camera_list():
    """
    读取摄像头列表
    :return:
    """
    params = g.data
    values = get_params(params, 'deviceId', 'slotId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, slot_id = values
    rs = open_ipc_svc.ipc_read_camera_list(device_id, slot_id)
    return jsonify(response_parameter_resolution(rs))


@open_ipc_bp.route('/IPCChannelList', methods=['GET', 'POST'])
@open_log
def ipc_channel_list():
    """
    ipc源通道列表
    """
    params = g.data
    values = get_params(params, 'deviceId', 'sourceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, source_id = values
    rs = open_ipc_svc.ipc_channel_detail_list(device_id, source_id)
    return jsonify(response_parameter_resolution(rs))


@open_ipc_bp.route('/IPCChannelFullDetail', methods=['GET', 'POST'])
@open_log
def ipc_channel_detail_api():
    """
    ipc源通道详情·
    """
    params = g.data
    values = get_params(params, 'deviceId', 'sourceId', 'channelId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, source_id, channel_id = values
    rs = open_ipc_svc.ipc_read_channel_detail(device_id, source_id, channel_id)
    return jsonify(response_parameter_resolution(rs))


@open_ipc_bp.route('/IPCChannelFreeList', methods=['GET', 'POST'])
@open_log
def ipc_channel_free_list():
    """
    ipc源通道详情·
    """
    params = g.data
    values = get_params(params, 'deviceId', 'sourceId', 'slotId', 'montageId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, source_id, slot_id,  montage_id = values
    des = open_ipc_svc.ipc_channel_list_data(device_id, source_id, slot_id, montage_id)
    return jsonify(response_parameter_resolution(des))


@open_ipc_bp.route('/IPCSourceCreate', methods=['GET', 'POST'])
@open_log
def ipc_source_create():
    """
    icc源创建
    :return:
    """
    params = g.data
    values = get_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = open_ipc_svc.ipc_source_create(device_id, params)
    if rs['status'] == 0:
        add_to_web(
            SyncDataName.Input_IPC_Source_Create, {
                'deviceId': device_id}, None)
    return jsonify(response_parameter_resolution(rs))


@open_ipc_bp.route('/IPCSourceDelete', methods=['GET', 'POST'])
@open_log
def ipc_source_delete():
    """
    icc源删除
    :return:
    """
    params = g.data
    values = get_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = open_ipc_svc.ipc_source_delete(device_id, params)
    if rs['status'] == 0:
        from H_9U.api.screen import screen_list
        add_to_web(SyncDataName.Screen_Read_All_List, None, screen_list, device_id)
        add_to_web(SyncDataName.Input_IPC_Source_Delete, None, {'deviceId': device_id, 'sourceList': [
                   {'sourceId': x['sourceId']} for x in rs['data']['sourceList'] if x['result'] == 0]})
    return jsonify(response_parameter_resolution(rs))


@open_ipc_bp.route('/IPCSourceRename', methods=['GET', 'POST'])
@open_log
def ipc_source_rename():
    """
    ipc信号源重命名
    :return:
    """
    params = g.data
    values = get_params(params, 'deviceId', 'sourceId', 'name')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, source_id, name = values
    rs = open_ipc_svc.ipc_source_rename(device_id, source_id, name)
    if rs['status'] == 0:
        add_to_web(SyncDataName.Input_IPC_Source_Rename, params, None)
    return jsonify(response_parameter_resolution(rs))


@open_ipc_bp.route('/IPCChannelEdit', methods=['GET', 'POST'])
@open_log
def ipc_channel_edit():
    """
     ipc信号源通道修改
     """
    params = g.data
    values = get_params(params, 'deviceId', 'slotId', 'sourceId', 'channelId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, slot_id, source_id, channel_id = values
    rs = open_ipc_svc.ipc_channel_edit(device_id, params)
    if rs['status'] == 0:
        from H_9U.api.screen import screen_list
        add_to_web(SyncDataName.Screen_Read_All_List, None, screen_list, device_id)
        add_to_web(SyncDataName.Input_IPC_Channel_Edit, None, ipc_channel_detail, device_id, source_id, channel_id)
    return jsonify(response_parameter_resolution(rs))


@open_ipc_bp.route('/IPCChannelAdd', methods=['GET', 'POST'])
@open_log
def ipc_channel_add():
    """
     ipc信号源通道添加
     """
    params = g.data
    values = get_params(params, 'deviceId', 'slotId', 'sourceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, slot_id, source_id = values
    rs = open_ipc_svc.ipc_channel_add(device_id, slot_id, source_id, params)
    if rs['status'] == 0:
        add_to_web(SyncDataName.Input_IPC_Channel_Add, None, ipc_channel_detail_list, device_id, source_id)
    return jsonify(response_parameter_resolution(rs))


@open_ipc_bp.route('/IPCChannelDelete', methods=['GET', 'POST'])
@open_log
def ipc_channel_delete():
    """
     ipc信号源通道删除
     """
    params = g.data
    values = get_params(params, 'deviceId', 'sourceList')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, source_list = values
    rs = open_ipc_svc.ipc_channel_delete(device_id, params)
    if rs['status'] == 0:
        channel_id_list = []
        source_id = 0
        for x in rs['data']['sourceList']:
            source_id = x['sourceId']
            for channel in x['channelList']:
                if channel['result'] == 0:
                    channel_id_list.append(channel['channelId'])
        add_to_web(SyncDataName.Input_IPC_Channel_Delete, {
                'deviceId': device_id, 'sourceId': source_id, 'channelList': channel_id_list})
        from H_9U.api.screen import screen_list
        add_to_web(SyncDataName.Screen_Read_All_List, None, screen_list, device_id)
    return jsonify(response_parameter_resolution(rs))


@open_ipc_bp.route('/IPCCheckSourceName', methods=['GET', 'POST'])
@open_log
def ipc_check_source_name():
    """
    检查源名称
    """
    params = g.data
    values = get_params(params, 'deviceId', 'sourceId', 'name')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, source_id, name = values
    rs = open_ipc_svc.ipc_search_source_list(device_id, source_id, name)
    return jsonify(response_parameter_resolution(rs))


@open_ipc_bp.route('/IPCGroupList', methods=['GET', 'POST'])
@open_log
def ipc_group_list():
    """
    ipc分组列表
    """
    params = g.data
    values = get_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = network_svc.network_source_group_list(device_id)
    return jsonify(response_parameter_resolution(rs))


@open_ipc_bp.route('/IPCGroupDetail', methods=['GET', 'POST'])
@open_log
def ipc_group_node_list():
    """
    ipc 分组下源列表
    """
    params = g.data
    values = get_params(params, 'deviceId', 'groupId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, group_id = values
    rs = network_svc.network_source_group_detail(device_id, group_id)
    return jsonify(response_parameter_resolution(rs))


@open_ipc_bp.route('/IPCGroupCreate', methods=['GET', 'POST'])
@open_log
def ipc_group_create():
    """
    创建ipc源分组
    """
    params = g.data
    values = get_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = network_svc.network_source_group_create(device_id, params)
    if rs['status'] == 0:
        params['groupId'] = rs['data']['id']
        add_to_web(SyncDataName.Input_IPC_GROUP_CREATE, params, None)
    return jsonify(response_parameter_resolution(rs))


@open_ipc_bp.route('/IPCGroupDelete', methods=['GET', 'POST'])
@open_log
def ipc_group_delete():
    """
    删除ipc源分组
    """
    params = g.data
    values = get_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = network_svc.network_source_group_delete(device_id, params)
    if rs['status'] == 0:
        add_to_web(SyncDataName.Input_IPC_GROUP_DELETE, params, None)
    return jsonify(response_parameter_resolution(rs))


@open_ipc_bp.route('/IPCGroupUpdate', methods=['GET', 'POST'])
@open_log
def ipc_group_update():
    """
    更新ipc源分组
    """
    params = g.data
    values = get_params(params, 'deviceId', 'groupId', 'name')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, group_id, name = values
    rs = network_svc.network_source_group_update(device_id, group_id, params)
    if rs['status'] == 0:
        add_to_web(SyncDataName.Input_IPC_GROUP_UPDATE,  {
                "deviceId": device_id, "groupId": group_id, "name": name}, None)
    return jsonify(response_parameter_resolution(rs))


@open_ipc_bp.route('/IPCGroupNodeDelete', methods=['GET', 'POST'])
@open_log
def ipc_group_node_delete():
    """
    ipc源分组删除节点
    """
    params = g.data
    values = get_params(params, 'deviceId', 'groupId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, group_id = values
    rs = network_svc.network_source_group_node_delete(device_id, group_id, params)
    if rs['status'] == 0:
        add_to_web(SyncDataName.Input_IPC_GROUP_UPDATE, params, None)
    return jsonify(response_parameter_resolution(rs))


@open_ipc_bp.route('/IPCMontageList', methods=['GET', 'POST'])
@open_log
def ipc_montage_list_api():
    """
    ipc拼接源列表
    """
    params = g.data
    values = get_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = open_ipc_svc.ipc_montage_list(device_id)
    return jsonify(response_parameter_resolution(rs))


@open_ipc_bp.route('/IPCMontageTemplateCreate', methods=['GET', 'POST'])
@open_log
def ipc_montage_template_create():
    """
    ipc拼接源创建模板
    """
    params = g.data
    values = get_params(params, 'deviceId', 'slotId', 'montageId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, slot_id, montage_id = values
    rs = open_ipc_svc.ipc_montage_template_create(device_id, params)
    if rs['status'] == 0:
        drs = open_ipc_svc.ipc_montage_list_get(device_id, slot_id)
        if drs['status'] == 0:
            data = [x for x in drs['data']['montageList']if x['montageId'] == montage_id][0]
            add_to_web(SyncDataName.Input_IPC_Montage_Save, data, None)
    return jsonify(response_parameter_resolution(rs))


@open_ipc_bp.route('/IPCMontageTemplateUpdate', methods=['GET', 'POST'])
@open_log
def ipc_montage_template_update():
    """
    ipc 拼接源 模板修改
    """
    params = g.data
    values = get_params(params, 'deviceId', 'slotId', 'templateId', 'montageId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, slot_id, template_id, montage_id = values
    rs = open_ipc_svc.ipc_montage_template_update(device_id, params)
    if rs['status'] == 0:
        drs = open_ipc_svc.ipc_montage_list_get(device_id, slot_id)
        if drs['status'] == 0:
            data = [x for x in drs['data']['montageList'] if x['montageId'] == montage_id][0]
            add_to_web(SyncDataName.Input_IPC_Montage_Save, data, None)
    return jsonify(response_parameter_resolution(rs))


@open_ipc_bp.route('/IPCMontageTemplateDelete', methods=['GET', 'POST'])
@open_log
def ipc_montage_template_delete():
    """
    ipc 拼接源 模板删除
    """
    params = g.data
    values = get_params(params, 'deviceId', 'slotId', 'templateId', 'montageId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, _, _, _ = values
    rs = open_ipc_svc.ipc_montage_template_delete(device_id, params)
    if rs['status'] == 0:
        add_to_web(SyncDataName.Input_IPC_Montage_Template_Delete, params, None)
    return jsonify(response_parameter_resolution(rs))


@open_ipc_bp.route('/IPCMontageTemplateApply', methods=['GET', 'POST'])
@open_log
def ipc_montage_template_apply():
    """
    ipc 拼接源 模板应用
    """
    params = g.data
    values = get_params(params, 'deviceId', 'slotId', 'templateId', 'montageId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, _, _, _ = values
    rs = open_ipc_svc.ipc_montage_template_apply(device_id, params)
    if rs['status'] == 0:
        add_to_web(SyncDataName.Input_IPC_Montage_Template_Apply, params, None)
    return jsonify(response_parameter_resolution(rs))


@open_ipc_bp.route('/IPCMontageRename', methods=['GET', 'POST'])
@open_log
def ipc_montage_rename():
    """
    ipc拼接源重命名
    """
    params = g.data
    values = get_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = open_ipc_svc.ipc_montage_rename(device_id, params)
    if rs['status'] == 0:
        add_to_web(SyncDataName.Input_IPC_Montage_Rename, params, None)
        from H_9U.api.screen import screen_list
        add_to_web(SyncDataName.Screen_Read_All_List, None, screen_list,device_id)
    return jsonify(response_parameter_resolution(rs))
