"""
Created on 2021年9月08日
@author: 程声清
layer api
参考前端接口文档layer部分
返回值统一为 {body:加密串, msg:"", sign:"", status:0}
"""
from flask import Blueprint, g
from flask.json import jsonify
from H_9U.api.input import input_get_connect_capacity
from H_9U.api.layer import __build_ipc_data
from H_9U.models.result import get_result_model, ResInfo
from H_9U.models.sysconst import SourceType, LayerType, ConnectCapacity
from H_9U.service import open_layer_svc, open_screen_svc
from H_9U.util.common import get_params
from H_9U.util.logwrapper import open_log
from H_9U.util.request_handle import response_parameter_resolution
from H_9U.api.sync_queue import add_to_web
from H_9U.models.syncdataname import SyncDataName

open_layer_bp = Blueprint('open_layer', __name__)


@open_layer_bp.route('/screenLayerTake', methods=['GET', 'POST'])
@open_log
def screen_layer_take():
    """
    屏幕上屏参考接口文档screenLayerTake
    :return: 结果对象
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    rs = open_layer_svc.layer_screen_no_time_play(
        device_id, screen_id, params)
    # 推送
    if rs['status'] == 0:
        add_to_web(SyncDataName.Layer_Screen_Take, params)
    return jsonify(response_parameter_resolution(rs))


@open_layer_bp.route('/create', methods=['GET', 'POST'])
@open_log
def layer_create():
    """
    添加Layer至screen
    :return: 结果对象
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    if not params['general']:
        return jsonify(get_result_model(ResInfo.Params_Error))

    input_id = params['source']['inputId']

    # 图层类型
    params['general']['type'] = LayerType.Default

    if params['source']['sourceType'] == SourceType.InputSource:
        # 输入接口容量：
        rs = input_get_connect_capacity(device_id, input_id)
        # 获取不到连接容量，直接返回错误
        if rs['status'] != 0:
            return jsonify(get_result_model(ResInfo.Middle_Data_Err))
    elif params['source']['sourceType'] == SourceType.IPC:  # 255 表示使用IPC源
        __build_ipc_data(device_id, params)

    params['general']['sizeType'] = ConnectCapacity.SL  # conn
    rs = open_layer_svc.layer_create(device_id, screen_id, params)
    if rs['status'] == 0:
        layer_id = rs['data']['id']
        add_to_web(SyncDataName.Layer_Create, None, device_id, layer_id)
    return jsonify(response_parameter_resolution(rs))


@open_layer_bp.route('/delete', methods=['GET', 'POST'])
@open_log
def layer_delete():
    """
     删除Layer
    :return: 结果对象
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId', 'layerId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, layer_id = values
    rs = open_layer_svc.layer_delete(device_id, screen_id, layer_id)
    if rs['status'] == 0:
        add_to_web(SyncDataName.Layer_Delete, {
            "deviceId": device_id,
            "screenId": screen_id,
            "layerId": layer_id
        })
    return jsonify(response_parameter_resolution(rs))


@open_layer_bp.route('/detailList', methods=['GET', 'POST'])
@open_log
def layer_read_list():
    """
    获取图层详细信息列表
    :return: 结果对象
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    rs = open_screen_svc.screen_layer_detail_list(device_id, screen_id)
    return jsonify(response_parameter_resolution(rs))


@open_layer_bp.route('/readDetail', methods=['GET', 'POST'])
@open_log
def layer_read_detail():
    """
     读取详细信息
     :return: 参考前端接口文档readDetail
     """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId', 'layerId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, layer_id = values
    rs = open_layer_svc.layer_read_detail(device_id, screen_id, layer_id)
    return jsonify(response_parameter_resolution(rs))


@open_layer_bp.route('/writeGeneral', methods=['GET', 'POST'])
@open_log
def layer_write_general():
    """
    设置Layer基础信息
    :return: 参考文档writeGeneral
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId', 'layerId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, layer_id = values
    rs = open_layer_svc.layer_write_general(
        device_id, screen_id, layer_id, params)
    if rs['status'] == 0:
        add_to_web(SyncDataName.Layer_Write_General, None, device_id, layer_id)
    return jsonify(response_parameter_resolution(rs))


@open_layer_bp.route('/readZIndex', methods=['GET', 'POST'])
@open_log
def layer_read_z_index():
    """
    读取屏幕图层Z index
    :return:  参考文档 readZIndex
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    rs = open_layer_svc.layer_read_z_index(device_id, screen_id)
    return jsonify(response_parameter_resolution(rs))


@open_layer_bp.route('/writeWindow', methods=['GET', 'POST'])
@open_log
def layer_write_layout_window():
    """
   设置指定图层的位置和大小信息
    :return:  参考文档 writeWindow
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId', 'layerId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, layer_id = values
    rs = open_layer_svc.layer_write_layout_window(
        device_id, screen_id, layer_id, params)
    if rs['status'] == 0:
        add_to_web(SyncDataName.Layer_Write_Window, params)
    return jsonify(response_parameter_resolution(rs))


@open_layer_bp.route('/clear', methods=['GET', 'POST'])
@open_log
def layer_clear():
    """
    屏幕图层清除
    :return:  参考文档 clear
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    rs = open_layer_svc.layer_clear(device_id, screen_id)
    if rs['status'] == 0:
        add_to_web(SyncDataName.Layer_Delete_All,
                   {'deviceId': device_id, 'screenId': screen_id, 'layers': []})
    return jsonify(response_parameter_resolution(rs))


@open_layer_bp.route('/writeZIndex', methods=['GET', 'POST'])
@open_log
def layer_write_z_index():
    """
    读取屏幕图层Z index
    :return:  参考文档 readZIndex
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    rs = open_layer_svc.layer_write_z_index(device_id, screen_id, params)
    if rs['status'] == 0:
        add_to_web(SyncDataName.Layer_Write_Z_Index, None, device_id, screen_id)
    return jsonify(response_parameter_resolution(rs))


@open_layer_bp.route('/writeLock', methods=['GET', 'POST'])
@open_log
def layer_write_z_lock():
    """
    锁定开启、关闭
    :return:  参考文档 writeLock
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId', 'layerId', 'lock')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, layer_id, is_lock = values
    rs = open_layer_svc.layer_write_z_lock(
        device_id, screen_id, layer_id, is_lock)
    if rs['status'] == 0:
        add_to_web(SyncDataName.Layer_Write_Lock, None, device_id, screen_id, layer_id)
    return jsonify(response_parameter_resolution(rs))


@open_layer_bp.route('/writeSource', methods=['GET', 'POST'])
@open_log
def layer_write_source():
    """
    更改图层源信息
    :return:  参考文档 writeSource
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId', 'layerId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, layer_id = values
    rs = open_layer_svc.layer_write_source(
        device_id, screen_id, layer_id, params)
    if rs['status'] == 0:
        add_to_web(SyncDataName.Layer_Write_Source, None, device_id, screen_id, layer_id)
    return jsonify(response_parameter_resolution(rs))


@open_layer_bp.route('/freeze', methods=['GET', 'POST'])
@open_log
def layer_freeze():
    """
    冻结图层
    :return:  参考文档 freeze
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId', 'layerId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, layer_id = values
    rs = open_layer_svc.layer_freeze(
        device_id, screen_id, layer_id, params)
    if rs['status'] == 0:
        add_to_web(SyncDataName.Layer_Freeze, None, device_id, screen_id, layer_id)
    return jsonify(response_parameter_resolution(rs))


@open_layer_bp.route('/screenLayerLayout', methods=['GET', 'POST'])
@open_log
def screen_layer_layout():
    """
    图层模板开窗
    :return:  参考文档 screenLayerLayout
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    rs = open_layer_svc.screen_layer_layout(device_id, screen_id, params)
    if rs['status'] == 0:
        add_to_web(SyncDataName.Layer_Screen_Layout, params)
    return jsonify(response_parameter_resolution(rs))
