"""
Created on 2019年6月28日
@author: 刘益宗
User接口.
对应前端接口User
返回值统一为 {data:JSON结构体, msg:"", status:0}
"""
from flask.globals import request
from flask.json import jsonify
from flask import Blueprint, g
from werkzeug.security import check_password_hash, generate_password_hash
from H_9U import SysSettings
from H_9U.models.sysconst import PermissionCode, DataType, SafeConfig, UserStatus
from H_9U.service import open_user_svc
from H_9U.service.role import rolesvc

from H_9U.service.user import usersvc
from H_9U.models.result import get_result_model, ResInfo
from H_9U.util import logutil
from H_9U.util.En_De_coryp import des_decrypt, des_encrypt
from H_9U.util.common import valid_json, valid_params, password_verifier
from H_9U.api.websender import push_data_org
from H_9U.models.syncdataname import SyncDataName
from H_9U.util.logwrapper import api_log
from H_9U.util.common import md5
from H_9U.util.password_generate import generate_password
from H_9U.util.permission_valid import get_code_list, valid_permission_role_user
from H_9U.util.regular_check import regular_check_username, regular_check_password

user_bp = Blueprint('user', __name__)


@user_bp.route('/login', endpoint='login', methods=['GET', 'POST'])
@api_log
def login():
    """
    用户登录
    :return: token， deviceid
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'username', 'password')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))

    username, password = values
    user_id = 0
    success = False
    g.param = params
    username = des_decrypt(username)
    password = des_decrypt(password)
    rs = usersvc.login(username, password, params)
    if rs['status'] == 0 and rs['data']:
        success = True
        user_id = rs['data']['userId']
        rs['data']['versionType'] = SysSettings.Version_Type
    logutil.add_login_log(username, user_id, success)
    return jsonify(rs)


@user_bp.route('/add', methods=['GET', 'POST'])
@api_log
def add():
    """
    添加用户
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'username')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    username = values
    user = g.user
    role_id = user['roleId']
    if not regular_check_username(username):
        return jsonify(get_result_model(ResInfo.User_Name_Invalid))
    password = params['password']
    is_safe = g.safe
    if is_safe == SafeConfig.IsSafe:
        password = params['password']
        is_valid = usersvc.valid_user_password(password)
        if not is_valid:
            return get_result_model(ResInfo.User_Password_Invalid)
    # 增加接口校验
    user_rs = get_code_list(PermissionCode.User_Operation, role_id)
    if user_rs['status'] != 0:
        return jsonify(user_rs)
    is_safe = g.safe
    rs = usersvc.add(params, is_safe)
    if rs['status'] == 0:
        user_id = rs['data']['id']
        rs = usersvc.get(user_id)
        user_des = rolesvc.get_role_by_user_id(user_id)
        if user_des['status'] == 0 and user_des['data']:
            role_id = user_des['data']['roleId']
            role_message = rolesvc.get_role_by_id(role_id)
            if role_message['status'] == 0 and role_message['data']:
                rs['data']['role'] = role_message['data']
        if rs['status'] == 0:
            push_data_org(SyncDataName.User_Add, rs['data'])
            user_id = rs['data']['id']
            message = 'add userId:{0}, username:{1}, roleId:{2} success'.format(
                user_id, username, params['roleId'])
            logutil.add_opt_log(
                logutil.LogModule.User,
                logutil.LogFunction.User,
                logutil.LogOperation.Add,
                message)
    return jsonify(rs)


@user_bp.route('/all', methods=['GET', 'POST'])
@api_log
def all_user():
    """
    获取全部用户
    :return: 全部用户列表
    """
    user = g.user
    role_id = user['roleId']
    # 超管角色
    from H_9U.service.role import rolesvc
    rs = rolesvc.get_role_by_id(role_id)
    if rs['status'] == 0 and rs['data']:
        if rs['data']['role_type'] == 1:
            rs = usersvc.get_all()
        else:
            rs = usersvc.get_user_by_role_id(role_id)
    return jsonify(rs)


@user_bp.route('/delete', methods=['GET', 'POST'])
@api_log
def delete():
    """
    删除用户
    :return: 全部用户列表
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'id', 'password')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    user_id, pwd = values
    if user_id <= 0:
        return get_result_model(ResInfo.Params_Error)
    # 接口校验
    user_id_list = [user_id]
    user_rs = valid_permission_role_user(PermissionCode.User_Operation, DataType.User_Type, user_id_list)
    if user_rs['status'] != 0:
        return jsonify(user_rs)
    pwd = des_decrypt(pwd)
    is_safe = g.safe
    flag = password_verifier(g.user['password'], pwd, is_safe)
    if not flag:
        return jsonify(get_result_model(ResInfo.User_Password_Error))

    rs = usersvc.delete(user_id)
    if rs['status'] == 0:
        push_data_org(SyncDataName.User_Delete, {'userId': user_id})
        user_rs = usersvc.detail(user_id)
        username = user_rs['data']['username'] if user_rs['data'] else ''
        message = "delete userId:{0}, username:{1} success".format(
            user_id, username)
        logutil.add_opt_log(
            logutil.LogModule.User,
            logutil.LogFunction.User,
            logutil.LogOperation.Delete,
            message)
    return jsonify(rs)


@user_bp.route('/update', methods=['GET', 'POST'])
@api_log
def update():
    """
    更新用户
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    user_id = params.get('id', 0)
    if user_id <= 0:
        return get_result_model(ResInfo.Params_Error)
    # 接口校验
    user_id_list = [user_id]
    user_rs = valid_permission_role_user(PermissionCode.User_Operation, DataType.User_Type, user_id_list)
    if user_rs['status'] != 0:
        return jsonify(user_rs)

    rs = usersvc.update(user_id, params)
    if rs['status'] == 0:
        usersvc.delete_user_token(user_id)
        rs = usersvc.get(user_id)
        if rs['status'] == 0:
            des = rolesvc.get_role_by_user_id(user_id)
            if des['status'] == 0 and des['data']:
                # 获取当前用户拥有的角色
                role_id = des['data']['roleId']
                role_rs = rolesvc.get_role_by_id(role_id)
                if role_rs['status'] == 0 and role_rs['data']:
                    rs['data']['role'] = role_rs['data']
                message = "update user, before:{0} , after: {1}".format(params, user_rs['data'])
                logutil.add_opt_log(
                    logutil.LogModule.User,
                    logutil.LogFunction.User,
                    logutil.LogOperation.Edit,
                    message)
            push_data_org(SyncDataName.User_Update, rs['data'])
    return jsonify(rs)


@user_bp.route('/logout', methods=['GET', 'POST'])
@api_log
def logout():
    """
    登出接口
    :return: 成功
    """
    token = g.token
    if token:
        usersvc.logout(token)
    return jsonify(get_result_model())


@user_bp.route('/allFuncs', methods=['GET', 'POST'])
@api_log
def all_funcs():
    """
    获取全部权限列表
    :return: 参考文档user/allFuncs
    """
    rs = usersvc.all_funcs()
    return jsonify(rs)


@user_bp.route('/detail', methods=['GET', 'POST'])
@api_log
def detail():
    """
    获取用户详情
    :return: 参考接口文档 user/detail
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'userId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    user_id = values
    rs = usersvc.detail(user_id)
    return jsonify(rs)


@user_bp.route('/resetPwd', methods=['GET', 'POST'])
@api_log
def reset_pwd():
    """
    获取用户详情
    :return: 参考接口文档 user/detail
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'userId', 'password')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    user_id, pwd = values

    # 接口校验
    user_id_list = [user_id]
    user_rs = valid_permission_role_user(PermissionCode.User_Operation, DataType.User_Type, user_id_list)
    if user_rs['status'] != 0:
        return jsonify(user_rs)
    pwd = des_decrypt(pwd)
    user_message = g.user
    login_password = user_message['password']
    is_safe = g.safe
    flag = password_verifier(login_password, pwd, is_safe)
    if not flag:
        return jsonify(get_result_model(ResInfo.User_Password_Error))
    if user_message['superadmin'] != 1:
        return get_result_model(ResInfo.User_Func_Err)
    if is_safe == SafeConfig.IsSafe:
        is_valid = usersvc.valid_user_password(pwd)
        if not is_valid:
            return jsonify(get_result_model(ResInfo.User_Password_Invalid))
    password = generate_password(8)
    rs = usersvc.update_password(user_id, password, UserStatus.Active, is_safe)
    password = des_encrypt(password)
    if rs['status'] == 0:
        usersvc.delete_user_token(user_id)
        rs['data'] = {
            "userId": user_id,
            "password": password
        }
        user_rs = usersvc.get(user_id, True)
        message = "reset password userId:{0} username:{1} success".format(
            user_id, user_rs['data']['username'] if rs['data'] else '')
        logutil.add_opt_log(
            logutil.LogModule.User,
            logutil.LogFunction.User,
            logutil.LogOperation.Edit,
            message)
    return jsonify(rs)


@user_bp.route('/updatePwd', methods=['GET', 'POST'])
@api_log
def update_pwd():
    """
    获取用户详情
    :return: 参考接口文档 user/detail
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'userId', 'newPwd')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    user_id, new_pwd = values
    user_id_list = [user_id]
    user_rs = valid_permission_role_user(PermissionCode.User_Operation, DataType.User_Type, user_id_list)
    if user_rs['status'] != 0:
        return jsonify(user_rs)
    if g.user['id'] != user_id:
        return jsonify(get_result_model(ResInfo.User_Func_Err))
    t = params.get('type', 1)
    is_safe = g.safe
    new_pwd = des_decrypt(new_pwd)
    rs = __build_user_data(t, params, user_id, new_pwd, is_safe)
    if rs['status'] != 0:
        return jsonify(rs)
    rs = usersvc.update_password(user_id, new_pwd, 1 if t == 0 else None, is_safe)
    if rs['status'] == 0:
        message = "update password userId:{0} username:{1} success".format(
            user_id, g.user['username'] if rs['data'] else '')
        logutil.add_opt_log(
            logutil.LogModule.User,
            logutil.LogFunction.User,
            logutil.LogOperation.Edit,
            message)
    return jsonify(rs)


def __build_user_data(t, params, user_id, new_pwd, is_safe):
    rs = get_result_model()
    old_pwd = params.get('oldPwd', None)
    if t == 1 and old_pwd is None:
        return get_result_model(ResInfo.Params_Error)
    if old_pwd:
        old_pwd = des_decrypt(old_pwd)
    if t == 1:
        user_rs = usersvc.get(user_id, True)
        if user_rs['status'] != 0:
            return get_result_model(ResInfo.User_Not_Exist)
        flag = password_verifier(user_rs['data']['password'], old_pwd, is_safe)
        if not flag:
            return get_result_model(ResInfo.User_Password_Error)
    if is_safe == SafeConfig.IsSafe:
        is_valid = usersvc.valid_user_password(new_pwd)
        if not is_valid:
            return get_result_model(ResInfo.User_Password_Invalid)
    return rs


@user_bp.route('/getOpenInfoList', methods=['GET', 'POST'])
@api_log
def find_open_info_list():
    """
    获取第三方项目信息列表
    :return: 结果对象
    """
    return jsonify(open_user_svc.get_open_info_list())


@user_bp.route('/getOpenInfoByName', methods=['GET', 'POST'])
@api_log
def find_open_info_by_name():
    """
    获取第三方项目信息列表
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    project_name = valid_params(params, 'projectName')
    return jsonify(open_user_svc.find_open_info_by_name(project_name))


@user_bp.route('/getSecretKey', methods=['GET', 'POST'])
@api_log
def find_open_info_secret_key():
    """
    随机生成秘钥
    :return: 结果对象
    """
    return jsonify(open_user_svc.find_open_info_secret_key())


@user_bp.route('/addOpenInfo', methods=['GET', 'POST'])
@api_log
def add_open_info():
    """
    添加第三方项目信息
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'projectName', 'secretKey')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))

    rs = open_user_svc.user_info_create(params)
    if rs['status'] == 0:
        o_id = rs['data']['id']
        rs = open_user_svc.get_user_info_by_id(o_id)
        if rs['status'] == 0:
            push_data_org(SyncDataName.Open_Info_Add, rs['data'])
    return jsonify(rs)


@user_bp.route('/editOpenInfo', methods=['GET', 'POST'])
@api_log
def edit_open_info():
    """
    修改第三方信息
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    o_id = params.get('id', 0)
    if o_id <= 0:
        return get_result_model(ResInfo.Params_Error)
    rs = open_user_svc.update_open_info(params)
    if rs['status'] == 0:
        ds = open_user_svc.get_user_info_by_id(o_id)
        if ds['status'] == 0:
            push_data_org(SyncDataName.Open_Info_Update, ds['data'])
    return jsonify(rs)


@user_bp.route('/deleteOpenInfo', methods=['GET', 'POST'])
@api_log
def delete_open_info():
    """
    删除第三方信息
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    o_id = params.get('id', 0)
    if o_id <= 0:
        return get_result_model(ResInfo.Params_Error)

    rs = open_user_svc.delete_open_info(o_id)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Open_Info_Delete, {'id': o_id})
    return jsonify(rs)


@user_bp.route('/forgetSetPwd', methods=['GET', 'POST'])
@api_log
def api_user_forget_pwd():
    """
    忘记密码通过授权码重置密码
    :return:
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))
    values = valid_params(params, 'idCode', 'password')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    id_code, pwd = values
    rs = usersvc.reset_admin_password(id_code, pwd)
    if rs['status'] == 0:
        user_id = 0
        user_rs = usersvc.get_admin_user()
        if user_rs['status'] == 0 and user_rs['data']:
            user_id = user_rs['data']['id']
            rs['data'] = {
                "userId": user_id,
                "password": pwd
            }
        message = "reset password userId:{0} username:{1} success".format(
            user_id, user_rs['data']['username'] if rs['data'] else '')
        logutil.add_opt_log(
            logutil.LogModule.User,
            logutil.LogFunction.User,
            logutil.LogOperation.Edit,
            message)
    return jsonify(rs)


@user_bp.route('/getVerifyCode', methods=['GET', 'POST'])
@api_log
def api_verify_code():
    """
    获取全部用户
    :return: 全部用户列表
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'username', 'verifyId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    username, verify_id = values
    if verify_id is None or username is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    rs = usersvc.get_verify_code(verify_id)
    if rs['status'] == 0:
        rs['data'] = {
            'img': rs['data']['img']
        }
    return jsonify(rs)


def valid_license_user(license_key, new_key):
    """
    验证授权用户
    :param license_key:
    :param new_key:
    :return:
    """
    is_safe = g.safe
    rs = usersvc.get_license_user()
    if rs['status'] != 0 or not rs['data']:
        return get_result_model(ResInfo.Device_Auth_User_Info_Err)
    user = rs['data']
    user_id = user['id']
    if is_safe == SafeConfig.IsNotSafe:
        if user['password'] and len(user['password'].strip()) > 0 and not user['password'].startswith('Nova_pass'):  # 设置过授权码
            if user['password'] != md5(license_key):  # 授权码不一致
                return get_result_model(ResInfo.Device_Auth_User_Info_Err)
            if new_key:
                rs = usersvc.update_password(user_id, new_key, None, is_safe)
        else:
            rs = usersvc.update_password(user_id, new_key, None, is_safe)
    if is_safe == SafeConfig.IsSafe:
        # 设置过授权码
        if not check_password_hash(user['password'], 'Nova_password.'):
            if not check_password_hash(user['password'], license_key): # 授权码不一致
                return get_result_model(ResInfo.Device_Auth_User_Info_Err)
            if new_key:
                rs = usersvc.update_password(user_id, new_key, None, is_safe)
        else:
            rs = usersvc.update_password(user_id, license_key, None, is_safe)
    return rs


def inited_license_user():
    is_safe = g.safe
    rs = usersvc.get_license_user()
    if rs['status'] == 0 and rs['data']:
        if is_safe == SafeConfig.IsSafe and check_password_hash(rs['data']['password'], 'Nova_password.'):
            rs['data'] = {"inited": 1}
        elif is_safe == SafeConfig.IsNotSafe and len(rs['data']['password'].strip()) > 0 and not rs['data']['password'].startswith('Nova_pass'):
            rs['data'] = {"inited": 1}
        else:
            rs['data'] = {"inited": 0}
    else:
        rs = get_result_model(ResInfo.Device_Auth_User_Info_Err)
    return rs


def get_user_data(user_rs, role_id):
    """
    获取当前登录用户的角色列表数据
    :return: 无分组信息
    """
    rs = usersvc.get_user_by_role_id(role_id)
    rs = rs['data'] if rs['status'] == 0 else []
    user_ids = [x['role']['id'] for x in rs]
    user_rs = user_rs if role_id in user_ids else []
    return user_rs

# --------------end 新增权限数据处理--- -----------
