'''
Created on 2019年6月6日
@author: GongQingBao
四代协议解析器：解析Socket-TCP Server返回的数据,从原始的通通讯数据中分析出所需要的业务数据.
'''

import json

from H_9U.util.log import logger
from H_9U.protocol.responseSatus import ResponseStatus


class ProtocolParser(object):

    def __init__(self):
        pass


    def parse(self, buff, data=None, transmit_p=None):
        """解析Server端/Tcp-Socket返回的数据
        :param buff:当前接收的字节数组
        :param data:上次解析出的数据(首次为None)
        :param transmit:透传数据(首次为None),格式定义为：{totalLength,}
        :return:元组(isGoon, revData, transmit, errCode)
            isGoon:是否继续接收[True-继续;False-终止]
            revData:解析出的数据
            errCode:错误码[0-正常、39-数据错误]
        """
        isGoon = False;
        revData = None;
        errCode = ResponseStatus.OK; # 正常-0

        if transmit_p != None:
            transmit = transmit_p;
        else:
            transmit = {"totalLength":None};

        if buff == None or len(buff) == 0:
            errCode = ResponseStatus.MIDDLEWARE_DATA_FALURE;
            logger.error("解析数据时-数据错误：为空或长度为0!");
            return isGoon, revData, transmit, errCode;

        # 获得数据包的总长度
        err,totalLength = self.getTotalLength(buff, transmit);

        if err == ResponseStatus.OK:
            transmit["totalLength"] = totalLength;
        else:
            errCode = err;
            return isGoon, revData, transmit, errCode;

        # 判断已接收数据是否为完整的一包数据
        if self.isCompleteData(buff, data, totalLength) == False: # 不是完整的一包数据:则进入数据再接收、拼接
            isGoon = True;
            revData = self.montageData(buff, data);
        else: # 是完整的一包数据:则进入数据解析
            isGoon = False;
            try:
                pkg = ProtocolV4Pkg(self.montageData(buff, data));
                errCode = pkg.Ack;
                revData = pkg.JsonArray;
            except Exception as e:
                print(e)
                errCode = ResponseStatus.MIDDLEWARE_DATA_FALURE
                revData = []

        return isGoon, revData, transmit, errCode;


    def getTotalLength(self, buff, transmit=None):
        """获得数据包总长度/总字节数.如果transmit["totalLength"]为空则从buff中获得长度,否则使用transmit["totalLength"]值.
        :param buff:当前接收的字节数组
        :param transmit:透传数据
        :return:元组(code,num)
            code:操作码[0-正常、39-数据错误]
            num:总字节数
        """
        if transmit != None and transmit["totalLength"] != None:
            return ResponseStatus.OK,transmit["totalLength"];
        else:
            if len(buff) >= 38:
                totalLength = buff[16] + buff[17]*256 + buff[18]*65536 + buff[19] * 16777216 + 20;
                return ResponseStatus.OK,totalLength;
            else:
                logger.error("解析数据(获得数据包总长度)时-数据错误：长度不足38!");
                return ResponseStatus.MIDDLEWARE_DATA_FALURE,0;


    def montageData(self, buff, data):
        """将当前数据与之前数据进行拼接.将buff拼接到data中,并将data返回.
        :param buff:当前接收的字节数组
        :param data:之前解析出的数据(首次为None)
        :return:业务数据
        """
        if data != None:
            data += buff;
            return data;
        else:
            return buff;


    def isCompleteData(self, buff, data, totalLength):
        """判断已接收数据是否为完整的一包数据
        :param buff:当前接收的字节数组
        :param data:上次解析出的数据(首次为None)
        :param totalLength:数据总长度
        :return:是否为完整一包数据[True-是、False-否]
        """
        if data == None:
            if len(buff) == totalLength:
                return True;
        else:
            if (len(buff)+len(data)) == totalLength:
                return True;

        return False;
    
    
    def isFirstData(self, buff, data=None):
        """判断是否为第一包数据
        :param buff:当前接收的字节数组
        :param data:上次解析出的数据(首次为None)
        :param totalLength:数据总长度
        :return:是否为第一包数据[True-是、False-否]
        """
        if data == None:
            return True;

        count = len(buff);
        
        if count >= 38:
            if buff[0] == 0xAA and buff[1] == 0x55 and buff[1] == 0x00:
                return True;

        return False;
    
    
    def packageSendData(self, fnCode, rwCode, data, *params):
        """将向TCP-Server请求/发送的数据进行打包
        :param fnCode:功能编码[Arg1(基础码、功能码),数字型]
        :param rwCode:操作类型{0:读,1:写}
        :param data:json数据[可为空/None]
        :param params:可变参数,如：deviceId、screenId、...
        :return:字节数组
        """
        
        # 打包数据成字节数组,且符合《视频协议》格式.
        # 协议分3部分组成：1.头/head;2.数据体/body;3.尾/tail/校验位.
        if(type(data) != list):
            data = [data]
            
        pkg = ProtocolV4Pkg()
        pkg.Arg1 = fnCode
        
        if(rwCode == 0):
            pkg.Code = 0
            pkg.ApolloCode = 1
        else:
            pkg.Code = 1
            pkg.ApolloCode = 2

        for i in range(0,len(params)):
            pkg.Arg2[i] = params[i]

        if(data != None):
            pkg.JsonArray = data
            
        #print("fnCode:%s, 参数个数%i, 16进制参数:%s, JSON数据:%s" % (fnName, fnCode, count, paramsHex, data));
        # 打包数据
        byteA = pkg.genBuff()
#         print("发送数据:%s" % byteA);
#         print([hex(x) for x in bytes(byteA)]);
#         print(str(binascii.b2a_hex(byteA))[2:-1]);
        
        return byteA;


class ProtocolV4Pkg:
    """四代协议对应的结构体
    """

    DEVICE_TYPE = 0x04
    PACKET_TYPE = 0x30

    def __init__(self,buff=None):
        self.Flag = 0xAA55
        self.Ack = 0
        self.Code = 0
        self.ApolloCode = 1
        self.Arg1 = 0x1000
        self.Arg2 = [0]*4
        self.JsonArray = []
        
        if buff != None and len(buff) > 0:
            self.Flag = buff[0]+buff[1]*256
            self.Ack = buff[2]
            dataLen = buff[16]+buff[17]*256 + buff[18]*65536 + buff[19] * 16777216 - 18
            jsonBytes = buff[36:dataLen+36]

            if(len(jsonBytes) > 0):
                self.JsonArray = json.loads(jsonBytes.decode())
                # try:
                #     self.JsonArray = json.loads(jsonBytes.decode())
                # except:
                #     json_str = jsonBytes.decode('unicode-escape')
                #     self.JsonArray = json.loads(json_str,  strict=False)


    def genBuff(self):
        """根据四代协议对数据进行打包.
        :return:返回字节数组
        """
        jsonBytesLen = 18
        jsonBytes = bytes(0)
        
        if len(self.JsonArray) > 0:
            jsonBytes = bytes(json.dumps(self.JsonArray),'utf-8')
            jsonBytesLen += len(jsonBytes)

            buff = [0]*36
            buff[0] = self.Flag % 256
            buff[1] = self.Flag // 256
            buff[6] = self.DEVICE_TYPE
            buff[10] = self.Code
            buff[11] = self.PACKET_TYPE
            buff[16] = jsonBytesLen % 256
            buff[17] = (jsonBytesLen & 0xffff) // 256
            buff[18] = (jsonBytesLen >> 16) % 256
            buff[19] = (jsonBytesLen >> 16) // 256
            buff[20] = self.ApolloCode
            buff[29] = self.Arg1 % 256
            buff[30] = self.Arg1 // 256
            buff[31] = self.Arg2[3]
            buff[32] = self.Arg2[2]
            buff[33] = self.Arg2[1]
            buff[34] = self.Arg2[0]
            bytes1 = bytes(buff)
            bytes2 = jsonBytes
            bytes3 = bytes(2)

            return bytes1+bytes2+bytes3


