﻿"""
Created on 2019年6月28日
@author: 刘益宗
用户模块数据库访问
返回值统一为 {data:JSON结构体, msg:"", status:0}
"""

from sqlite3 import IntegrityError

from H_9U.dao import _created_all_table
from H_9U.util.sqlitepool import pool, sqllock
from H_9U.models.result import get_result_model, ResInfo
from H_9U.util.log import logger
from H_9U.models.sysconst import UserType


class UserDao:

    def __init__(self, conn_pool):
        self.pool = conn_pool
        _created_all_table(conn_pool)


    def get_by_id(self, user_id):
        """
        根据用户Id获取用户
        :param user_id: 用户Id
        :return: 用户信息
        """
        rs = get_result_model()
        try:
            sqllock.acquire()
            conn = self.pool.get()
            with conn:
                cursor = conn.cursor()
                cursor.execute("select id, username, superadmin, status, tel, email, password from user where id=:id",
                               {"id": user_id})
                row = cursor.fetchone()
                if row:
                    rs['data'] = {
                        'id': row[0],
                        'username': row[1],
                        'superadmin': row[2],
                        'status': row[3],
                        'tel': str(row[4]),
                        'email': str(row[5]),
                        'password': row[6]
                    }
        except Exception as e:
            rs = get_result_model(ResInfo.DB_Error)
        finally:
            if sqllock.locked():
                sqllock.release()
        return rs

    def get_by_username(self, username, usertype=UserType.SysUser):
        """
        根据用户名访问用户
        :param username: 用户名
        :return: 用户数据
        """
        rs = get_result_model()
        try:
            sqllock.acquire()
            conn = self.pool.get()

            sql = "select id, username, password,superadmin, status from user where type=:usertype and username=:username {};".format(
                "COLLATE NOCASE" if usertype == UserType.SysUser else "")
            with conn:
                cursor = conn.cursor()
                cursor.execute(sql, {"usertype": usertype, "username": username})
                row = cursor.fetchone()
                if row:
                    rs['data'] = {
                        'id': row[0],
                        'username': row[1],
                        'password': row[2],
                        'superadmin': row[3],
                        'status': row[4]
                    }
        except Exception as e:
            rs = get_result_model(ResInfo.DB_Error)
        finally:
            if sqllock.locked():
                sqllock.release()

        return rs

    def new_get_by_username(self, username):
        """
        根据用户名访问用户
        :param username: 用户名
        :return: 用户数据
        """
        rs = get_result_model()
        try:
            sqllock.acquire()
            conn = self.pool.get()
            sql = "select id, username, password,tel,email,[type], status,superadmin from user where username=:username"
            with conn:
                cursor = conn.cursor()
                cursor.execute(sql, {"username": username})
                row = cursor.fetchone()
                if row:
                    rs['data'] = {
                        'id': row[0],
                        'username': row[1],
                        'password': row[2],
                        'tel': str(row[3]),
                        'email': str(row[4]),
                        'type': row[5],
                        'status': row[6],
                        'superadmin': row[7]
                    }
        except Exception as e:
            print(e)
            rs = get_result_model(ResInfo.DB_Error)
        finally:
            if sqllock.locked():
                sqllock.release()

        return rs

    def insert_user(self, username, password, tel, email, status, role_id):
        """
        新增用户
        :param username: 用户名
        :param password: 密码
        :param tel: 电话
        :param email: 邮箱
        :return: 标准对象
        """
        rs = get_result_model()
        try:
            sqllock.acquire()
            conn = self.pool.get()
            with conn:
                cursor = conn.cursor()
                cursor.execute(
                    "insert into user(username, password, tel, email, [type], status,superadmin) values(?,?,?,?,0,?,0)",
                    (username, password, tel, email, status))
                cursor.execute('select last_insert_rowid() newid;')
                rows = cursor.fetchone()
                user_id = rows[0]
                cursor.execute("insert into user_role(user_id, role_id) values(?,?)",(user_id, role_id))
                conn.commit()
                rs['data']['id'] = user_id
        except IntegrityError:
            rs = get_result_model(ResInfo.User_Name_Exist)
        except Exception as e:
            logger.exception(e)
            rs = get_result_model(ResInfo.User_Add_Faild)
        finally:
            if sqllock.locked():
                sqllock.release()
        return rs

    def insert(self, username, password, tel, email, status, funcs, usertype=UserType.SysUser):
        """
        新增用户
        :param username: 用户名
        :param password: 密码
        :param tel: 电话
        :param email: 邮箱
        :return: 标准对象
        """
        rs = get_result_model()
        try:
            sqllock.acquire()
            conn = self.pool.get()
            with conn:
                cursor = conn.cursor()
                cursor.execute(
                    "insert into user(usertype, username, password, tel, email, superadmin, status) values(?,?,?,?,?,0,?)",
                    (usertype, username, password, tel, email, status))
                cursor.execute('select last_insert_rowid() newid;')
                rows = cursor.fetchone()
                user_id = rows[0]
                if funcs:
                    sql = "insert into userFunc(userId, funcId) values"
                    for f in funcs:
                        sql += "({},{}),".format(user_id, f)
                    sql = sql[:-1]
                    cursor.execute(sql)
                conn.commit()
                rs['data']['id'] = user_id
        except IntegrityError:
            rs = get_result_model(ResInfo.User_Name_Exist)
        except Exception as e:
            logger.exception(e)
            rs = get_result_model(ResInfo.User_Add_Faild)
        finally:
            if sqllock.locked():
                sqllock.release()
        return rs

    def update(self, user_id, role_id, user_params):
        """
        更新用户状态
        :param userid: 用户Id
        :param active: 是否启动
        :return: 标准返回
        """
        try:
            conditions = []
            for key in user_params:
                conditions.append('{}=:{}'.format(key, key))
            sql = "update user set " + ', '.join(conditions) + " where id=:id"
            user_params['id'] = user_id
            sqllock.acquire()
            conn = self.pool.get()
            with conn:
                cursor = conn.cursor()
                cursor.execute(sql, user_params)
                cursor.execute("delete from user_role where user_id=:user_id", {"user_id": user_id})
                cursor.execute(
                    "insert into user_role(user_id, role_id) values(?,?)",
                    (user_id, role_id))
                conn.commit()
        except Exception as e:
            logger.exception(e)
        finally:
            if sqllock.locked():
                sqllock.release()
        return get_result_model()

    def get_all_user(self):
        """
        获取全部用户
        :return:
        """
        rs = get_result_model()
        try:
            sqllock.acquire()
            conn = self.pool.get()
            with conn:
                cursor = conn.cursor()
                cursor.execute("select id, username,tel, email, [type], status, superadmin from user")
                rows = cursor.fetchall()
                users = []
                if rows:
                    for row in rows:
                        users.append({
                            'id': row[0],
                            'username': row[1],
                            'tel': str(row[2]),
                            'email': str(row[3]),
                            'type': row[4],
                            'status': row[5],
                            'superadmin': row[6]
                        })
                rs['data'] = users
        except Exception as e:
            rs = get_result_model(ResInfo.DB_Error)
        finally:
            if sqllock.locked():
                sqllock.release()
        return rs

    def get_license_user(self):
        rs = get_result_model()
        try:
            sqllock.acquire()
            conn = self.pool.get()
            with conn:
                cursor = conn.cursor()
                cursor.execute("select id, username,password, tel, email, status from user where type=:usertype",
                               {"usertype": UserType.LicenseUser})
                row = cursor.fetchone()
                user = {}
                if row:
                    user = {
                        'id': row[0],
                        'username': row[1],
                        'password': row[2],
                        'tel': str(row[3]),
                        'email': str(row[4]),
                        'status': row[5]
                    }
                rs['data'] = user
        except Exception as e:
            rs = get_result_model(ResInfo.DB_Error)
        finally:
            if sqllock.locked():
                sqllock.release()
        return rs

    def delete(self, user_id):
        """
        删除用户
        :param user_id: 用户Id
        :return: 结果对象
        """
        rs = get_result_model()
        try:
            sqllock.acquire()
            conn = self.pool.get()
            with conn:
                cursor = conn.cursor()
                cursor.execute("delete from user where id=:id", {"id": user_id})
                cursor.execute("delete from user_role where user_id=:user_id", {"user_id": user_id})
                conn.commit()
        except Exception as e:
            logger.exception(e)
            rs = get_result_model(ResInfo.User_Delete_Error)
        finally:
            if sqllock.locked():
                sqllock.release()
        return rs

    def get_all_func(self):
        """
        获取全部用户
        :return:
        """
        rs = get_result_model()
        try:
            sqllock.acquire()
            conn = self.pool.get();
            with conn:
                cursor = conn.cursor()
                cursor.execute(
                    "select id, name,level, parentId, display,[order],status, endpoint, brief from function")
                rows = cursor.fetchall()
                functions = []
                if rows:
                    for row in rows:
                        functions.append({
                            'id': row[0],
                            'name': row[1],
                            'level': row[2],
                            'parentId': row[3],
                            'display': row[4],
                            'order': row[5],
                            'status': row[6],
                            'endpoint': row[7],
                            'brief': row[8]
                        })
                rs['data'] = functions
        except Exception as e:
            rs = get_result_model(ResInfo.DB_Error)
        finally:
            if sqllock.locked():
                sqllock.release()
        return rs

    def get_user_func(self, user_id):
        rs = get_result_model()
        try:
            sqllock.acquire()
            conn = self.pool.get()
            with conn:
                cursor = conn.cursor()
                cursor.execute(
                    "select id, userId,funcId from userFunc where userId=:userId", {"userId": user_id})
                rows = cursor.fetchall()
                functions = []
                if rows:
                    for row in rows:
                        functions.append({
                            'id': row[0],
                            'userId': row[1],
                            'funcId': row[2]
                        })
                rs['data'] = functions
        except Exception as e:
            rs = get_result_model(ResInfo.DB_Error)
        finally:
            if sqllock.locked():
                sqllock.release()
        return rs

    def update_password(self, userid, password, status):
        """
        更新用户状态
        :param userid: 用户Id
        :param password: 密码
        :return: 标准返回
        """
        try:
            sql = "update user set password=:password where id=:id"
            if status:
                sql = "update user set password=:password, status=" + str(status) +" where id=:id"
            sqllock.acquire()
            conn = self.pool.get()
            with conn:
                cursor = conn.cursor()
                cursor.execute(sql,
                    {"password": password, "id": userid})
                conn.commit()
        except Exception as e:
            logger.exception(e)
        finally:
            if sqllock.locked():
                sqllock.release()
        return get_result_model()

    def get_user_by_role_id(self, user_id):
        """
        根据角色获取用户列表
        :return:
        """
        rs = get_result_model()
        try:
            sqllock.acquire()
            conn = self.pool.get()
            with conn:
                cursor = conn.cursor()
                cursor.execute(
                    "select id, userId,funcId from userFunc where userId=:userId", {"userId": user_id})
                rows = cursor.fetchall()
                functions = []
                if rows:
                    for row in rows:
                        functions.append({
                            'id': row[0],
                            'userId': row[1],
                            'funcId': row[2]
                        })
                rs['data'] = functions
        except Exception as e:
            rs = get_result_model(ResInfo.DB_Error)
        finally:
            if sqllock.locked():
                sqllock.release()
        return rs

    def get_admin_user(self):
        """
        查询admin用户信息
        :return: 标准返回
        """
        rs = get_result_model()
        try:
            sqllock.acquire()
            conn = self.pool.get()
            with conn:
                cursor = conn.cursor()
                cursor.execute('select id, username,password, tel, email, status from user '
                               'where type = 0 and superadmin=1')
                row = cursor.fetchone()
                if row:
                    user = {
                        'id': row[0],
                        'username': row[1],
                        'password': row[2],
                        'tel': row[3],
                        'email': row[4],
                        'active': row[5]
                    }
                    rs['data'] = user
        except Exception as e:
            rs = get_result_model(ResInfo.DB_Error)
        finally:
            if sqllock.locked():
                sqllock.release()
        return rs

    def update_user_status(self, userid, status):
        rs = get_result_model()
        try:
            sqllock.acquire()
            conn = self.pool.get()
            with conn:
                cursor = conn.cursor()
                sql = "update user set status=:status where id=:id"
                cursor.execute(sql,  {"status": str(status), "id": userid})
                conn.commit()
        except Exception as e:
            logger.error(e)
            rs = get_result_model(ResInfo.DB_Error)
        finally:
            if sqllock.locked():
                sqllock.release()
        return rs


userdao = UserDao(pool)

