"""
Created on 2019年6月28日
@author: 刘益宗
输入接口.
对应3.5代接口协议中的 input
返回值统一为 {data:JSON结构体, msg:"", status:0}
"""
from H_9U.models.sysconst import ReadSign
from H_9U.protocol.fncode import FnCode
from H_9U.models.result import transfer_midware_resp
from H_9U import controller as ctrl
from H_9U.util.logwrapper import log_wrapper
from H_9U.util.formatdata import format_server_data, format_client_data


@log_wrapper
def input_list_read(device_id):
    """
    获取input接口的枚举
    :param device_id: 设备id
    :return: 参考协议文档：0x00 - key: inputList
    """
    rs = transfer_midware_resp(ctrl.read(FnCode.INPUT_LIST, device_id, ReadSign.Is_ALL))
    format_server_data(rs['data'])
    return rs


@log_wrapper
def input_full_detail_read(device_id, input_id):
    """
    读取输入详情信息
    :param device_id:
    :param input_id:
    :return:  参考协议文档：0x01 - key: inputFullDetail
    """
    rs = transfer_midware_resp(ctrl.read(FnCode.INPUT_FULL_DETAIL, device_id, input_id))
    # 格式化刷新率等数据
    format_server_data(rs['data'])
    return rs


@log_wrapper
def input_general_read(device_id, input_id):
    """
    读取input常规信息
    :param device_id: 设备号
    :param input_id: 输入源ID
    :return: 参考协议文档：0x02 - key: inputGeneral
    """
    rs = transfer_midware_resp(ctrl.read(FnCode.INPUT_GENERAL, device_id, input_id))
    return rs


@log_wrapper
def input_general_write(device_id, input_id, data):
    """
    写指定input的基本信息
    :param device_id: 设备id
    :param input_id: 输出id
    :param data:  参考协议文档：0x02 - key: inputGeneral
    :return:  参考协议文档：0x02 - key: inputGeneral
    """
    return transfer_midware_resp(ctrl.write(FnCode.INPUT_GENERAL, [data], device_id, input_id))


@log_wrapper
def input_timing_read(device_id, input_id):
    """
    读取指定输入的timing信息
    :param device_id: 设备Id
    :param input_id: 输入id
    :return: 参考协议文档：0x03 - key: inputTiming
    """
    return transfer_midware_resp(ctrl.read(FnCode.INPUT_TIMING, device_id, input_id))


@log_wrapper
def input_timing_write(device_id, input_id, data):
    """
    设置输入timing信息
    :param device_id: 设备ID
    :param input_id: 输入ID
    :param data: 参考协议文档：0x03 - key: inputTiming
    :return: 参考协议文档：0x03 - key: inputTiming
    """
    format_client_data(data)
    print("data==", data)
    return transfer_midware_resp(ctrl.write(FnCode.INPUT_TIMING, [data], device_id, input_id))


@log_wrapper
def input_edid_read(device_id, input_id):
    """
    读指定input的EDID
    :param device_id: 设备号
    :param input_id: 输入ID
    :return: 参考协议文档：0x04- key: inputEdid
    """
    return transfer_midware_resp(ctrl.read(FnCode.INPUT_EDID, device_id, input_id))


@log_wrapper
def input_edid_write(device_id, input_id, data):
    """
    写指定input的Edid
    :param device_id: 设备号
    :param input_id: 输入源ID
    :param data:  参考协议文档：0x04- key: inputEdid
    :return: 参考协议文档：0x04- key: inputEdid
    """
    return transfer_midware_resp(ctrl.write(FnCode.INPUT_EDID, data if isinstance(data, list) else [data], device_id, input_id))


@log_wrapper
def input_osd_read(device_id, input_id):
    """
    读指定input的OSD
    :param device_id: 设备号
    :param input_id: 输入ID
    :return: 参考协议文档：0x05 - key: inputOSD
    """
    return transfer_midware_resp(ctrl.read(FnCode.INPUT_OSD, device_id, input_id))


@log_wrapper
def input_osd_write(device_id, input_id, data):
    """
    写指定input的OSD
    :param device_id: 设备号
    :param input_id: 输入源ID
    :param data:  参考协议文档：0x05 - key: inputOSD
    :return: 参考协议文档：0x05 - key: inputOSD
    """
    return transfer_midware_resp(ctrl.write(FnCode.INPUT_OSD, [data], device_id, input_id))


@log_wrapper
def input_crop_list_read(device_id, input_id):
    """
    读取输入源截取列表
    :param device_id: 设备Id
    :param input_id: 输入Id
    :return: 参考协议文档：0x06 - key: inputCropList
    """
    return transfer_midware_resp(ctrl.read(FnCode.INPUT_CROP_LIST, device_id, input_id))


@log_wrapper
def input_crop_create_write(device_id, input_id, data):
    """
    创建截取
    :param device_id: 设备Id
    :param input_id: 输入Id
    :param data:  参考协议文档：0x07- key: inputCropCreate
    :return: 参考协议文档：0x07- key: inputCropCreate
    """
    return transfer_midware_resp(ctrl.write(FnCode.INPUT_CROP_CREATE, [data], device_id, input_id))


@log_wrapper
def input_crop_delete_write(device_id, input_id, crop_id, data):
    """
    删除截取
    :param device_id: 设备Id
    :param input_id: 输入Id
    :param crop_id: 截取id
    :param data:  0x08 - key: inputCropDelete
    :return: 0x08 - key: inputCropDelete
    """
    return transfer_midware_resp(ctrl.write(FnCode.INPUT_CROP_DELETE, [data], device_id, input_id, crop_id))


@log_wrapper
def input_crop_update_write(device_id, input_id, crop_id, data):
    """
    修改输入截取
    :param device_id: 设备id
    :param input_id:  输入Id
    :param crop_id:  截取Id
    :param data:  截取参数
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.INPUT_CROP_GENERAL, [data], device_id, input_id, crop_id))


@log_wrapper
def input_group_list_read(device_id):
    """
    读取输入分组列表
    :param device_id: 设备Id
    :return: 输入列表
    """
    return transfer_midware_resp(ctrl.read(FnCode.INPUT_GROUP_LIST_READ, device_id))


@log_wrapper
def input_group_create_write(device_id, data):
    """
    创建输入分组
    :param device_id: 设备Id
    :param data: 分组信息
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.INPUT_GROUP_CREATE, [data], device_id))


@log_wrapper
def input_group_list_update_write(device_id, data):
    """
    更新分组信息
    :param device_id: 设备Id
    :param data: 组内容
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.INPUT_GROUP_UPDATE, [data], device_id))


@log_wrapper
def input_group_delete_write(device_id, group_id, data):
    """
    删除输入分组信息
    :param device_id: 设备Id
    :param group_id: 组Id
    :param data: data
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.INPUT_GROUP_DELETE, [data], device_id, group_id))


@log_wrapper
def input_group_node_read(device_id, group_id):
    """
    读取组节点信息
    :param device_id: 设备Id
    :param group_id: 组Id
    :return: 组内容
    """
    return transfer_midware_resp(ctrl.read(FnCode.INPUT_GROUP_NODE_READ, device_id, group_id))


@log_wrapper
def input_group_node_update_write(device_id, group_id, data):
    """
    更新组节点信息
    :param device_id: 设备Id
    :param group_id: 组id
    :param data: 组内容
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.INPUT_GROUP_NODE_UPDATE, [data], device_id, group_id))


@log_wrapper
def input_group_node_add_write(device_id, group_id, input_id, data):
    """
    组内新增输入源
    :param device_id:设备Id
    :param group_id: 组Id
    :param input_id: 输入源Id
    :param data: 内容
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.INPUT_GROUP_NODE_ADD, [data], device_id, group_id, input_id))


@log_wrapper
def input_group_node_delete_write(device_id, group_id, input_id, data):
    """
    组内删除输入源
    :param device_id:设备Id
    :param group_id: 组Id
    :param input_id: 输入源Id
    :param data: 内容
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.INPUT_GROUP_NODE_DELETE, [data], device_id, group_id, input_id))


@log_wrapper
def input_regulate(device_id, input_id, data):
    """
    VGA 接口校验
    :param device_id: 设备Id
    :param input_id:  输入源Id
    :param data: 内容
    :return:  结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.INPUT_REGULATE, [data], device_id, input_id))


@log_wrapper
def reverse_control_info_write(device_id, input_id, data):
    """
    输入源反控设置
    :param device_id:  设备Id
    :param input_id:  输入源Id
    :param data:  数据
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.INPUT_REVERSE_CONTROL, [data], device_id, input_id))


@log_wrapper
def mosaic_montage_read(device_id, input_id):
    """
    读取光纤模板列表
    :param device_id:  设备Id
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.read(FnCode.INPUT_MOSAIC_MONTAGELIST, device_id, input_id))


@log_wrapper
def mosaic_montage_write(device_id, data):
    """
    设置光纤模板
    :param device_id:  设备Id
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.INPUT_MOSAIC_MONTAGELIST, [data], device_id))


def mosaic_rename(device_id, data):
    """
    光纤模板源重命名
    :param device_id:  设备Id
    return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.INPUT_MOSAIC_MONTAGELIST, [data], device_id))