"""
Created on 2019年6月28日
@author: 刘益宗
layer接口.
对应3.5代接口协议中的 layer
返回值统一为 {data:JSON结构体, msg:"", status:0}
"""
from H_9U.models.sysconst import ReadSign
from H_9U.protocol.fncode import FnCode
from H_9U.models.result import transfer_midware_resp
from H_9U import controller as ctrl
from H_9U.util.logwrapper import log_wrapper


@log_wrapper
def layer_list_read(device_id, screen_id):
    """
    图层列表
    :param device_id: 设备id
    :param screen_id: 屏幕id
    :return: 参考中间件接口文档layerList
    """
    return transfer_midware_resp(ctrl.read(FnCode.LAYER_LIST, device_id, screen_id, ReadSign.Is_Not_ALL))


@log_wrapper
def layer_full_detail_read(device_id, screen_id, layer_id):
    """
    图层详细信息
    :param device_id: 设备id
    :param screen_id:屏幕id
    :param layer_id: 图层id
    :return:参考中间件接口文档layerFullDetail
    """
    # 图层Id低字节
    layer_id_low = layer_id % 256
    # 图层Id高字节
    layer_id_high = layer_id // 256
    return transfer_midware_resp(
        ctrl.read(FnCode.LAYER_FULLDETAIL, device_id, screen_id, layer_id_low, layer_id_high))


@log_wrapper
def layer_create_write(device_id, screen_id, data):
    """
    创建图层
    :param device_id: 设备id
    :param screen_id: 屏幕id
    :param data: 图层信息json
    :return:参考中间件接口文档layerCreate
    """
    return transfer_midware_resp(
        ctrl.write(FnCode.LAYER_CREATE, data if isinstance(data, list) else [data], device_id, screen_id))


@log_wrapper
def layer_delete_write(device_id, screen_id, layer_id, data):
    """
    删除图层
    :param device_id: 设备id
    :param screen_id:屏幕id
    :param layer_id: 图层id
    :param data: json数据
    :return:参考中间件接口文档layerDelete
    """
    # 图层Id低字节
    layer_id_low = layer_id % 256
    # 图层Id高字节
    layer_id_high = layer_id // 256

    return transfer_midware_resp(
        ctrl.write(FnCode.LAYER_DELETE, [data], device_id, screen_id, layer_id_low, layer_id_high))


@log_wrapper
def layer_general_write(device_id, screen_id, layer_id, data):
    """
    设置指定图层的基础信息.
    :param device_id: 设备id
    :param screen_id:屏幕id
    :param layer_id: 图层id
    :param data: json data
    :return:参考中间件接口文档layerGeneral
    """
    # 图层Id低字节
    layer_id_low = layer_id % 256
    # 图层Id高字节
    layer_id_high = layer_id // 256

    return transfer_midware_resp(
        ctrl.write(FnCode.LAYER_GENERAL, [data], device_id, screen_id, layer_id_low, layer_id_high))


@log_wrapper
def layer_general_read(device_id, screen_id, layer_id):
    """
    读取指定图层的基础信息.
    :param device_id: 设备id
    :param screen_id:屏幕id
    :param layer_id: 图层id
    :return:参考中间件接口文档layerGeneral
    """
    # 图层Id低字节
    layer_id_low = layer_id % 256
    # 图层Id高字节
    layer_id_high = layer_id // 256
    return transfer_midware_resp(
        ctrl.write(FnCode.LAYER_GENERAL, device_id, screen_id, layer_id_low, layer_id_high))


@log_wrapper
def layer_layout_window_write(device_id, screen_id, layer_id, data):
    """
    设置指定图层的位置和大小信息
    :param device_id: 设备id
    :param screen_id:屏幕id
    :param layer_id: 图层id
    :param data: json data
    :return:参考中间件接口文档layerLayoutWindow
    """
    # 图层Id低字节
    layer_id_low = layer_id % 256
    # 图层Id高字节
    layer_id_high = layer_id // 256

    return transfer_midware_resp(
        ctrl.write(FnCode.LAYER_LAYOUTWINDOW, [data], device_id, screen_id, layer_id_low, layer_id_high))


@log_wrapper
def layer_layout_window_read(device_id, screen_id, layer_id):
    """
    读取指定图层的位置和大小信息
     :param device_id: 设备id
    :param screen_id:屏幕id
    :param layer_id: 图层id
    :return:参考中间件接口文档layerLayoutWindow
    """
    # 图层Id低字节
    layer_id_low = layer_id % 256
    # 图层Id高字节
    layer_id_high = layer_id // 256
    return transfer_midware_resp(
        ctrl.read(FnCode.LAYER_LAYOUTWINDOW, device_id, screen_id, layer_id_low, layer_id_high))


@log_wrapper
def layer_source_write(device_id, screen_id, layer_id, data):
    """
    设置指定layer的源信息.
    :param device_id: 设备id
    :param screen_id:屏幕id
    :param layer_id: 图层id
    :param data: json data
    :return:参考中间件接口文档
    """
    # 图层Id低字节
    layer_id_low = layer_id % 256
    # 图层Id高字节
    layer_id_high = layer_id // 256

    return transfer_midware_resp(
        ctrl.write(FnCode.LAYER_SOURCE, [data], device_id, screen_id, layer_id_low, layer_id_high))


@log_wrapper
def layer_source_read(device_id, screen_id, layer_id):
    """
    设置指定layer的源信息.
    :param device_id: 设备id
    :param screen_id:屏幕id
    :param layer_id: 图层id
    :return:参考中间件接口文档layerSource
    """
    # 图层Id低字节
    layer_id_low = layer_id % 256
    # 图层Id高字节
    layer_id_high = layer_id // 256
    return transfer_midware_resp(
        ctrl.read(FnCode.LAYER_SOURCE, device_id, screen_id, layer_id_low, layer_id_high))


@log_wrapper
def layer_z_order_read(device_id, screen_id):
    """
    设置指定 screen的所有Layer的Z序
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :return:参考中间件接口文档layersZOrder
    """
    return transfer_midware_resp(ctrl.read(FnCode.LAYER_Z_ORDER_LIST, device_id, screen_id))


@log_wrapper
def layer_full_detail_list(device_id, screen_id):
    """
    读取屏幕图层详细信息列表
    :param device_id:  设备id
    :param screen_id: 屏幕id
    :return:  参考中间件接口文档layerFullDetailList
    """
    return transfer_midware_resp(ctrl.read(FnCode.LAYER_FULL_DETAIL_LIST, device_id, screen_id))


@log_wrapper
def layer_z_order_write(device_id, screen_id, data):
    """
    设置图层z序
    :param device_id: 设备Id
    :param screen_id:  屏幕Id
    :param data: 参数json
    :return: z 序列列表
    """
    return transfer_midware_resp(ctrl.write(FnCode.LAYER_SET_Z_ORDER, [data], device_id, screen_id))


@log_wrapper
def layer_freeze(device_id, screen_id, layer_id, data):
    """
    删除图层
    :param device_id: 设备id
    :param screen_id:屏幕id
    :param layer_id: 图层id
    :param data: json数据
    :return:参考中间件接口文档layerFreeze
    """
    # 图层Id低字节
    layer_id_low = layer_id % 256
    # 图层Id高字节
    layer_id_high = layer_id // 256

    return transfer_midware_resp(
        ctrl.write(FnCode.LAYER_FREEZE, [data], device_id, screen_id, layer_id_low, layer_id_high))


@log_wrapper
def screen_layer_auto_play(device_id, screen_id, data):
    """
    删除图层
    :param device_id: 设备id
    :param screen_id:屏幕id
    :param data: json数据
    :return:参考中间件接口文档0419
    """
    return transfer_midware_resp(
        ctrl.write(FnCode.SCREEN_LAYER_AUTO_PLAY, [data], device_id, screen_id))

@log_wrapper
def layer_lock(device_id, screen_id, layer_id, data):
    return transfer_midware_resp(
        ctrl.write(FnCode.LAYER_LOCK, [data], device_id, screen_id, layer_id))


@log_wrapper
def write_audio_status(device_id, screen_id, layer_id, data):
    """
    设置指定Slayer的音频信息
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :param layer_id: layer_id
    :param data: data
    :return: 结果对象
    """
    # 图层Id低字节
    layer_id_low = layer_id % 256
    # 图层Id高字节
    layer_id_high = layer_id // 256
    return transfer_midware_resp(ctrl.write(FnCode.LAYER_AUDIO_STATUS, [data],
                                            device_id, screen_id, layer_id_low, layer_id_high))


@log_wrapper
def layer_lock_ratio(device_id, screen_id, layer_id, data):
    """
    设置指定layer的纵横比
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :param layer_id: 图层Id
    :param data: data
    :return: 结果对象
    """
    # 图层Id低字节
    layer_id_low = layer_id % 256
    # 图层Id高字节
    layer_id_high = layer_id // 256
    return transfer_midware_resp(ctrl.write(FnCode.LAYER_LAYOUTWINDOW, [data],  device_id, screen_id,
                                            layer_id_low, layer_id_high))