from H_9U.mao.impl import input, ipcmao
from H_9U.util.cache import cacher
from H_9U.models.cachekey import CacheKey, CacheDuration


class InputApi:
    """
    mao层接口api
    """

    def read_list(self, device_id):
        """
        读取输入列表并加入缓存
        :param device_id: 设备id
        :return: 输入列表
        """
        cacher.clear_all()
        key = CacheKey.input_list(device_id)
        return cacher.InputCache.try_get_value(key, CacheDuration.Default, input.input_list_read, device_id)

    def read_detail(self, device_id, input_id):
        """
        读取输入详情并加入缓存
        :param device_id:  设备id
        :param input_id: 输入id
        :return: 输入详情
        """
        key = CacheKey.input_detail(device_id, input_id)
        return cacher.InputCache.try_get_value(key, CacheDuration.Default, input.input_full_detail_read, device_id,
                                               input_id)

    def read_general(self, device_id, input_id):
        """
        读取基本信息
        :param device_id: 设备id
        :param input_id: 输入id
        :return: 基本信息
        """
        return input.input_general_read(device_id, input_id)

    def write_general(self, device_id, input_id, data):
        """
        设置输出基本信息
        清空全部缓存，因为输入源名称许多地方使用
        :param device_id: 设备id
        :param input_id: 输入id
        :param data: 基本信息
        :return: 结果对象
        """
        rs = input.input_general_write(device_id, input_id, data)
        if rs['status'] == 0:
            cacher.clear_all()
        return rs

    def read_timing(self, device_id, input_id):
        """
        读取timing
        :param device_id: 设备Id
        :param input_id:  输入id
        :return: timing信息
        """
        return input.input_timing_read(device_id, input_id)

    def write_timing(self, device_id, input_id, data):
        """
        设置timing
        清除输入详情缓存
        :param device_id: 设备id
        :param input_id: 输入id
        :param data: timing 信息
        :return: 结果对象
        """
        rs = input.input_timing_write(device_id, input_id, data)
        if rs['status'] == 0:
            cacher.InputCache.clear()
        return rs

    def read_edid(self, device_id, input_id):
        """
        读取edid
        :param device_id: 设备Id
        :param input_id: 输入id
        :return: 结果数据
        """
        return input.input_edid_read(device_id, input_id)

    def write_edid(self, device_id, input_id, lst_data):
        """
        设置edid
        移除输入详情缓存
        :param device_id: 设备id
        :param input_id: 输入id
        :param lst_data: edid list对象
        :return: 结果对象
        """
        rs = input.input_edid_write(device_id, input_id, lst_data)
        if rs['status'] == 0:
            cacher.InputCache.clear()
        return rs

    def read_osd(self, device_id, input_id):
        """
        读取osd
        :param device_id: 设备id
        :param input_id: 输入id
        :return: 结果对象
        """
        return input.input_osd_read(device_id, input_id)

    def write_osd(self, device_id, input_id, data):
        """
        设置osd
        清除输入详情缓存
        :param device_id: 设备id
        :param input_id: 输入id
        :param data: osd信息
        :return: 结果对象
        """
        rs = input.input_osd_write(device_id, input_id, data)
        if rs['status'] == 0:
            key = CacheKey.input_detail(device_id, input_id)
            list_key = CacheKey.input_list(device_id)
            cacher.InputCache.delete(key)
            cacher.InputCache.delete(list_key)
        return rs

    def read_crop_list(self, device_id, input_id):
        """
        读取输入crop信息并加入缓存
        :param device_id: 设备id
        :param input_id: 输入id
        :return: 结果数据
        """
        key = CacheKey.input_crop_list(device_id, input_id)
        return cacher.InputCache.try_get_value(key, CacheDuration.Default, input.input_crop_list_read, device_id,
                                               input_id)

    def write_crop_create(self, device_id, input_id, data):
        """
        创建输入截取
        清除输入截取列表缓存
        :param device_id: 设备id
        :param input_id: 输入id
        :param data: 截取信息
        :return: 结果对象
        """
        rs = input.input_crop_create_write(device_id, input_id, data)
        if rs['status'] == 0:
            key = CacheKey.input_crop_list(device_id, input_id)
            list_key = CacheKey.input_list(device_id)
            cacher.InputCache.delete(key)
            cacher.InputCache.delete(list_key)
        return rs

    def write_crop_delete(self, device_id, input_id, crop_id, data):
        """
        删除输入截取
        清除全部缓存，因为图层会变成无源图层
        :param device_id: 设备id
        :param input_id: 输入id
        :param crop_id: 截取id
        :param data:
        :return: 结果对象
        """
        rs = input.input_crop_delete_write(device_id, input_id, crop_id, data)
        if rs['status'] == 0:
            cacher.clear_all()
        return rs

    def write_crop_update(self, device_id, input_id, crop_id, data):
        """
        修改输入截取
        清除全部缓存，因为图层名称可能会变
        :param device_id: 设备id
        :param input_id: 输入id
        :param crop_id: 截取id
        :param data:
        :return: 结果对象
        """
        rs = input.input_crop_update_write(device_id, input_id, crop_id, data)
        if rs['status'] == 0:
            cacher.clear_all()
        return rs

    def read_group_list(self, device_id):
        """
        读取输入分组列表
        :param device_id: 设备Id
        :return: 结果对象
        """
        key = CacheKey.input_group_list(device_id)
        return cacher.InputCache.try_get_value(key, CacheDuration.Default, input.input_group_list_read, device_id)

    def write_group_create(self, device_id, data):
        key = CacheKey.input_group_list(device_id)
        rs = input.input_group_create_write(device_id, data)
        if rs['status'] == 0:
            cacher.InputCache.delete(key)
        return rs

    def write_group_update(self, device_id, data):
        """
        更新组信息
        :param device_id: 设备Id
        :return: 结果对象
        """
        key = CacheKey.input_group_list(device_id)
        rs = input.input_group_list_update_write(device_id, data)
        if rs['status'] == 0:
            cacher.InputCache.delete(key)
        return rs

    def write_group_delete(self, device_id, group_id, data):
        """
        更新组信息
        :param device_id: 设备Id
        :param group_id:  组Id
        :return: 结果对象
        """
        key = CacheKey.input_group_list(device_id)
        key2 = CacheKey.input_group_node_list(device_id, group_id)
        rs = input.input_group_delete_write(device_id, group_id, data)
        if rs['status'] == 0:
            cacher.InputCache.delete(key)
            cacher.InputCache.delete(key2)
        return rs

    def read_group_node_items(self, device_id, group_id):
        """
        读取组节点信息
        :param device_id: 设备Id
        :param group_id:  组Id
        :return:  结果对象
        """
        key = CacheKey.input_group_node_list(device_id, group_id)
        return cacher.InputCache.try_get_value(key, CacheDuration.Default, input.input_group_node_read, device_id, group_id)

    def write_group_node_items_update(self, device_id, group_id, data):
        """
        更改组节点信息
        :param device_id: 设备Id
        :param group_id:  组Id
        :param data:  组内容
        :return:  结果对象
        """
        key = CacheKey.input_group_node_list(device_id, group_id)
        rs = input.input_group_node_update_write(device_id, group_id, data)
        if rs['status'] == 0:
            cacher.InputCache.delete(key)
        return rs

    def write_group_node_item_add(self, device_id, group_id, input_id, data):
        """
        组内添加元素
        :param device_id: 设备Id
        :param group_id:  组Id
        :param input_id:  输入Id
        :param data:  输入源信息
        :return: 结果对象
        """
        key = CacheKey.input_group_node_list(device_id, group_id)
        rs = input.input_group_node_add_write(device_id, group_id, input_id, data)
        if rs['status'] == 0:
            cacher.InputCache.delete(key)
        return rs

    def write_group_node_item_delete(self, device_id, group_id, input_id, data):
        """
        组内删除元素
        :param device_id: 设备Id
        :param group_id:  组Id
        :param data:  输入源信息
        :return: 结果对象
        """
        key = CacheKey.input_group_node_list(device_id, group_id)
        rs = input.input_group_node_delete_write(device_id, group_id, input_id, data)
        if rs['status'] == 0:
            cacher.InputCache.delete(key)
        return rs

    def write_input_regulate(self, device_id, input_id, data):
        """
        VGA 校准
        :param device_id: 设备Id
        :param input_id: 输入Id
        :param data: 校准信息
        :return: 结果对象
        """
        return input.input_regulate(device_id, input_id, data)

    def reverse_control_info_write(self, device_id, input_id, data):
        """
        输入源反控设置
        :param device_id:  设备Id
        :param input_id:  输入源Id
        :param data:  数据
        :return: 结果对象
        """
        rs = input.reverse_control_info_write(device_id, input_id, data)
        if rs['status'] == 0:
            cacher.InputCache.clear()
            cacher.ScreenCache.clear()
            cacher.LayerCache.clear()
        return rs

    def mosaic_montage_read(self, device_id, input_id):
        """
        读取光纤模板列表
        :param device_id:  设备Id
        :return: 结果对象
        """

        return input.mosaic_montage_read(device_id, input_id)

    def mosaic_montage_write(self, device_id, data):
        """
        设置光纤模板信息
        :param device_id:  设备Id
        :return: 结果对象
        """
        return input.mosaic_montage_write(device_id, data)

    def mosaic_rename(self, device_id, data):
        """
        光纤模板源重命名
        :param device_id:  设备Id
        return: 结果对象
        """
        return input.mosaic_rename(device_id, data)


input_api = InputApi()
