import math
from copy import deepcopy

from H_9U.mao.impl import output
from H_9U.service.common import get_connect_capacity
from H_9U.util.cache import cacher
from H_9U.util.log import logger
from H_9U.models.cachekey import CacheKey, CacheDuration
from H_9U.models.result import get_result_model, ResInfo
from H_9U.models.sysconst import TwoInOneCardData_1040, TwoInOneCardData_1300, ModelId, FunctionType, InterfaceType, \
    Output4KCardData, FourFiberCardData, OutputCardMode, FourFiberCardBackupData, FourFiberCardCopyData, \
    ConnectCapacity, Output12GSDIConst

from H_9U.util.outputtiming import outputCustomResolution, outputResolution, outputFourFiberCustomResolution
from H_9U.mao.device_mao import two_in_one_card_slot, write_two_in_one_card_payload, read_two_in_one_card_payload, \
    device_api
from H_9U.mao.device_mao import output_dl_slot, output_4k_slot
from H_9U.models.sysconst import CardCategory


class OutputApi:

    def _read_4k_slots(self, device_id):
        return output_4k_slot(device_id)

    def read_4k_outputs(self, device_id):
        """
        返回4k output卡和对应的output
            {
                1:[1,2,3],
                2:[3,4,5]
            }
        :param device_id: 设备Id
        :return: 4k卡信息
        """
        rs = self.read_origin_list(device_id)
        slot_rs = self._read_4k_slots(device_id)
        if slot_rs['status'] != 0:
            rs = get_result_model(ResInfo.Middle_Data_Err)
            return rs
        slot_dict = {}
        # output存在，且存在4k
        if not rs['data']['outputs'] or not slot_rs['data']:
            rs['data'] = slot_dict
            return rs
        slots = slot_rs['data']
        for item in rs['data']['outputs']:
            # 获取输入详情
            if item['slotId'] in slots:
                if item['slotId'] in slot_dict:
                    slot_dict[item['slotId']].append(item['outputId'])
                else:
                    slot_dict[item['slotId']] = [item['outputId']]
        rs['data'] = slot_dict
        return rs

    def read_dl_outputs(self, device_id):
        """
        读取所有dl的output
        :param device_id:
        :return:
        """
        rs = output_dl_slot(device_id)
        if rs['status'] != 0:
            logger.error('读取DL output slot err code:' + str(rs['status']))
            return rs

        dis_list = []
        dl_list = []
        if rs['data']:
            o_rs = self.read_origin_list(device_id)
            if o_rs['status'] != 0:
                logger.error('读取中间件outputlist err code:' + str(o_rs['status']))
                return o_rs

            slots = rs['data']
            for item in o_rs['data']['outputs']:
                # 获取输入详情
                slot = [x for x in slots if x['slotId'] == item['slotId']]
                if slot:
                    slot = slot[0]
                    interface = \
                    [x for x in slot['interfaces'] if x['interfaceId'] == item['interfaceId']][0]
                    if interface['functionType'] == FunctionType.Disable:
                        dis_list.append(item['outputId'])
                    elif interface['functionType'] == FunctionType.DL:
                        dl_list.append(item['outputId'])
        rs['data'] = {
            'disList': dis_list,
            'dlList': dl_list
        }
        return rs

    def _read_tio_slots(self, device_id):
        """
        读取二合一卡的slotId列表
        :param device_id:  设备Id
        :return: [slotId1,slotId2, ...]
        """
        return two_in_one_card_slot(device_id)

    def _write_tio_payload(self, device_id, slot_id, payload):
        """
        设置二合一的带载信息
        :param device_id: 设备Id
        :param slot_id:  slotId
        :param payload: 带载信息
        :return: 结果对象
        """
        return write_two_in_one_card_payload(device_id, slot_id, payload)

    def _read_tio_payload(self, device_id, slot_id):
        """
        读取二合一卡的带载信息
        :param device_id: 设备Id
        :param slot_id: slotId
        :return: {width, height, refreshRate}
        """
        return read_two_in_one_card_payload(device_id, slot_id)

    def read_origin_list(self, device_id):
        """
        读取输出列表并加入缓存， 二合一卡显示四个接口
        :param device_id: 设备id
        :return: 输出列表
        """
        key = CacheKey.output_list(device_id)
        return cacher.OutputCache.try_get_value(key, CacheDuration.Default, output.output_list_read, device_id)

    def two_in_one_outputs(self, device_id):
        """
        返回设备二合一卡和对应的output
            {
                1:[1,2,3],
                2:[3,4,5]
            }
        :param device_id: 设备Id
        :return: 二合一卡信息
        """
        rs = self.read_origin_list(device_id)
        twoinone_rs = self._read_tio_slots(device_id)

        if twoinone_rs['status'] != 0:
            rs = get_result_model(ResInfo.Middle_Data_Err)
            return rs
        slot_dict = {}
        # output存在，且存在二合一卡
        if not rs['data']['outputs'] or not twoinone_rs['data']:
            rs['data'] = slot_dict
            return rs
        two_in_ones = twoinone_rs['data']
        for item in rs['data']['outputs']:
            if item ['slotId'] in two_in_ones:
                if item['slotId'] in slot_dict:
                    slot_dict[item['slotId']].append(item['outputId'])
                else:
                    slot_dict[item['slotId']] = [item['outputId']]
        rs['data'] = slot_dict
        return rs

    def read_origin_detail(self, device_id, output_id):
        """
        读取output的原始信息
        :param device_id:
        :param output_id:
        :return:
        """
        key = CacheKey.output_detail(device_id, output_id)
        return cacher.OutputCache.try_get_value(key, CacheDuration.Default, output.output_full_detail_read, device_id,
                                              output_id)

    def _samecard_tio_outputs(self, device_id, output_id):
        """
        相同二合一卡的所有outputIdList
        :param device_id: 设备Id
        :param output_id: outputId
        :return: output对应的二合一卡信息(slotId, [output1, output2, ...])，如果不是二合一卡返回None,
        """
        rs = get_result_model()
        rs['data'] = None
        tio_rs = self.two_in_one_outputs(device_id)
        if tio_rs['status'] == 0 and tio_rs['data']:
            tio_rs = self._get_twoinone_card_sorted_output(output_id, tio_rs['data'])
            if tio_rs:
                rs['data'] = tio_rs

        return rs

    def _samecard_k4_outputs(self, device_id, output_id):
        """
        相同4k的所有outputIdList
        :param device_id: 设备Id
        :param output_id: outputId
        :return: output对应的二合一卡信息(slotId, [output1, output2, ...])，如果不是4k返回None,
        """
        rs = get_result_model()
        rs['data'] = None
        k4_rs = self.read_4k_outputs(device_id)
        if k4_rs['status'] == 0 and k4_rs['data']:
            for key, value in k4_rs['data'].items():
                # 如果能找到interface的output，说明是二合一卡interface
                if output_id in value:
                    value.sort()
                    rs['data'] = (key, value)
        return rs

    def _get_twoinone_card_sorted_output(self, output_id, tio_outputs_dict):
        """
        判断output是否是二合一卡的output
        :param output_id: output_id
        :param tio_outputs_dict: 二合一卡 {slotId:[output1, output2, ...]}
        :return: (slotId, sorted(output_list))
        """
        for key, value in tio_outputs_dict.items():
            # 如果能找到interface的output，说明是二合一卡interface
            if output_id in value:
                value.sort()
                return key, value

    def read_origin_timing(self, device_id, output_id):
        """
        读取输出timing
        :param device_id: 设备id
        :param output_id: 输出id
        :return:
        """
        return output.output_timing_read(device_id, output_id)

    def read_list(self, device_id):
        """
        读取输出列表并加入缓存
        :param device_id: 设备id
        :return: 输出列表
        """
        rs = self.read_origin_list(device_id)
        if rs['status'] != 0:
            return rs
        # 二合一卡
        twoinone_rs = self.two_in_one_outputs(device_id)
        dl_rs = self.read_dl_outputs(device_id)
        k4_rs = self.read_4k_outputs(device_id)

        if dl_rs['status'] != 0:
            logger.error("outputlist DL output,errcode:" + str(dl_rs['status']))
            return get_result_model(ResInfo.Middle_Data_Err)
        if twoinone_rs['status'] != 0:
            logger.error("outputlist 读取二合一卡失败,errcode:" + str(twoinone_rs['status']))
            return get_result_model(ResInfo.Middle_Data_Err)
        if k4_rs['status'] != 0:
            logger.error("outputlist 读取4k output失败,errcode:" + str(k4_rs['status']))
            return get_result_model(ResInfo.Middle_Data_Err)

        twoinones = twoinone_rs['data']
        # output存在，且存在二合一卡
        if rs['data']['outputs'] and twoinones:
            for x in range(len(rs['data']['outputs'])-1, -1, -1):
                item = rs['data']['outputs'][x]
                for value in twoinones.values():
                    if item['outputId'] in value and item['outputId'] != min(value):
                        rs['data']['outputs'].pop(x)

        # dloutput卡
        if dl_rs['data'] and dl_rs['data']['disList']:
            for x in range(len(rs['data']['outputs']) - 1, -1, -1):
                item = rs['data']['outputs'][x]
                if item['outputId'] in dl_rs['data']['disList']:
                    rs['data']['outputs'].pop(x)

        # output存在，且存在4k卡
        if rs['data']['outputs'] and k4_rs['data']:
            for x in range(len(rs['data']['outputs']) - 1, -1, -1):
                item = rs['data']['outputs'][x]
                for value in k4_rs['data'].values():
                    if item['outputId'] in value and item['outputId'] != min(value):
                        rs['data']['outputs'].pop(x)
        return rs

    def read_detail(self, device_id, output_id):
        """
        读取输出详情并加入缓存
        :param device_id: 设备id
        :param output_id: 输出id
        :return: 输出详情
        """
        rs = self.read_origin_detail(device_id, output_id)
        if rs['status'] == 0 and rs['data']:
            rs['data']['resolution'] = {
                'width': rs['data']['timing']['horizontal']['addrTime'],
                'height': rs['data']['timing']['vertical']['addrTime'],
                'refresh': rs['data']['timing']['refreshRate'],
                'isInterlace': rs['data']['timing']['isInterlace']
            }
            # 二合一卡处理
            if rs['data']['modelId'] in ModelId.Two_In_One_Card_List:
                slot_id = rs['data']['slotId']
                slot_rs = self._read_tio_payload(device_id, slot_id)
                if slot_rs['status'] == 0:
                    rs['data']['resolution'] = {
                        'width': slot_rs['data']['width'],
                        'height': slot_rs['data']['height'],
                        'refresh': slot_rs['data']['refresh'],
                        'direction': slot_rs['data']['direction'],
                        'outputCount': slot_rs['data']['outputCount'],
                        'isInterlace': rs['data']['timing']['isInterlace']
                    }

            # dl output 处理
            if rs['data']['modelId'] in ModelId.DL_Output_Card_List:
                dl_rs = self.read_dl_outputs(device_id)
                if dl_rs['status'] == 0 and rs['data']['outputId'] in dl_rs['data']['dlList']:
                    if rs['data']['modelId'] in [ModelId.Output_4_HDMI_Card, ModelId.Pd_Output_4_HDMI_Card]:
                        rs['data']['interfaceType'] = InterfaceType.HDMI14
                    else:
                        rs['data']['interfaceType'] = InterfaceType.Dual_Link_DVI

                    rs['data']['resolution']['width'] *= 2
                    rs['data']['timing']['horizontal']['addrTime'] *= 2
                    rs['data']['timing']['horizontal']['frontPorch'] *= 2
                    rs['data']['timing']['horizontal']['sync'] *= 2
                    rs['data']['timing']['horizontal']['totalTime'] *= 2

            # 4K卡处理
            if rs['data']['modelId'] in [ModelId.Output_2_HDMI20_Card, ModelId.Pd_Output_2_HDMI20_Card]:
                slot_id = rs['data']['slotId']
                slot_rs = self._read_tio_payload(device_id, slot_id)
                output_count = 0
                if slot_rs['status'] == 0:
                    output_count = slot_rs['data']['outputCount']
                    rs['data']['resolution'] = {
                        'width': slot_rs['data']['width'],
                        'height': slot_rs['data']['height'],
                        'refresh': slot_rs['data']['refresh'],
                        'direction': slot_rs['data']['direction'],
                        'outputCount': slot_rs['data']['outputCount'],
                        'isInterlace': rs['data']['timing']['isInterlace']
                    }
                if output_count == 4:
                    rs['data']['timing']['horizontal']['addrTime'] *= 4
                    rs['data']['timing']['horizontal']['frontPorch'] *= 4
                    rs['data']['timing']['horizontal']['sync'] *= 4
                    rs['data']['timing']['horizontal']['totalTime'] *= 4

            if rs['data']['modelId'] == ModelId.Output_12_3G_SDI_Card:
                width =  rs['data']['resolution']['width']
                height = rs['data']['resolution']['height']
                refreshRate = rs['data']['resolution']['refresh']
                if width >= Output12GSDIConst.LimitWidth and \
                        height >= Output12GSDIConst.LimitHeight and \
                        refreshRate > Output12GSDIConst.LimitRefresh:
                    slot_id = rs['data']['slotId']
                    slot_rs = self._read_tio_payload(device_id, slot_id)
                    if slot_rs['status'] == 0:
                        width = slot_rs['data']['width']
                        height = slot_rs['data']['height']
                        refreshRate = slot_rs['data']['refresh']
                        rs['data']['resolution'] = {
                            'width': width,
                            'height': height,
                            'refresh': refreshRate,
                            'direction': slot_rs['data']['direction'],
                            'outputCount': slot_rs['data']['outputCount'],
                            'isInterlace': rs['data']['timing']['isInterlace']
                        }
                    rs['data']['timing']['horizontal']['addrTime'] *= 4
                    rs['data']['timing']['horizontal']['frontPorch'] *= 4
                    rs['data']['timing']['horizontal']['sync'] *= 4
                    rs['data']['timing']['horizontal']['totalTime'] *= 4
        return rs

    def _build_detail_data(self, device_id, output_item):
        """
        读取输出详情并加入缓存
        :param device_id: 设备id
        :param output_id: 输出id
        :return: 输出详情
        """
        output_item['connectCapacity'] = get_connect_capacity(output_item['interfaceType'])
        if output_item['modelId'] in ModelId.Two_In_One_Card_List:
            output_item['connectCapacity'] = ConnectCapacity.K4
        output_item['resolution'] = {
            'width': output_item['timing']['horizontal']['addrTime'],
            'height': output_item['timing']['vertical']['addrTime'],
            'refresh': output_item['timing']['refreshRate'],
            'isInterlace': output_item['timing']['isInterlace']
        }
        # 二合一卡处理
        if output_item['modelId'] in ModelId.Two_In_One_Card_List:
            slot_id = output_item['slotId']
            slot_rs = self._read_tio_payload(device_id, slot_id)
            if slot_rs['status'] == 0:
                output_item['resolution'] = {
                    'width': slot_rs['data']['width'],
                    'height': slot_rs['data']['height'],
                    'refresh': slot_rs['data']['refresh'],
                    'direction': slot_rs['data']['direction'],
                    'outputCount': slot_rs['data']['outputCount'],
                    'isInterlace': output_item['timing']['isInterlace']
                }

        # dl output 处理
        if output_item['modelId'] in ModelId.DL_Output_Card_List:
            dl_rs = self.read_dl_outputs(device_id)
            if dl_rs['status'] == 0 and output_item['outputId'] in dl_rs['data']['dlList']:
                if output_item['modelId'] in [ModelId.Output_4_HDMI_Card, ModelId.Pd_Output_4_HDMI_Card]:
                    output_item['interfaceType'] = InterfaceType.HDMI14
                elif output_item['modelId'] in [ModelId.Output_4_DVI_Card, ModelId.Pd_Output_4_DVI_Card]:
                    output_item['interfaceType'] = InterfaceType.Dual_Link_DVI
                elif output_item['modelId'] == ModelId.Output_4_HDBaseT_Card:
                    output_item['interfaceType'] = InterfaceType.DB9_MALE
                elif output_item['modelId'] in [ModelId.Output_4_RJ45_Card, ModelId.Pd_Output_4_RJ45_Card]:
                    output_item['interfaceType'] = InterfaceType.RJ45DL

                output_item['resolution']['width'] *= 2


                output_item['timing']['horizontal']['addrTime'] *= 2
                output_item['timing']['horizontal']['frontPorch'] *= 2
                output_item['timing']['horizontal']['sync'] *= 2
                output_item['timing']['horizontal']['totalTime'] *= 2

        # 4K卡处理
        if output_item['modelId'] in [ModelId.Output_2_HDMI20_Card, ModelId.Pd_Output_2_HDMI20_Card]:
            slot_id = output_item['slotId']
            slot_rs = self._read_tio_payload(device_id, slot_id)
            output_count = 4
            if slot_rs['status'] == 0:
                output_count = slot_rs['data']['outputCount']
                output_item['resolution'] = {
                    'width': slot_rs['data']['width'],
                    'height': slot_rs['data']['height'],
                    'refresh': slot_rs['data']['refresh'],
                    'direction': slot_rs['data']['direction'],
                    'outputCount': slot_rs['data']['outputCount'],
                    'isInterlace': output_item['timing']['isInterlace']
                }
            if output_count == 4:
                output_item['timing']['horizontal']['addrTime'] *= 4
                output_item['timing']['horizontal']['frontPorch'] *= 4
                output_item['timing']['horizontal']['sync'] *= 4
                output_item['timing']['horizontal']['totalTime'] *= 4

        if output_item['modelId'] == ModelId.Output_12_3G_SDI_Card:
            slot_id = output_item['slotId']
            slot_rs = self._read_tio_payload(device_id, slot_id)
            if slot_rs['status'] == 0:
                if slot_rs['data']['outputCount'] == 4:
                    width = slot_rs['data']['width']
                    height = slot_rs['data']['height']
                    refreshRate = slot_rs['data']['refresh']
                    output_item['resolution'] = {
                        'width': width,
                        'height': height,
                        'refresh': refreshRate,
                        'direction': slot_rs['data']['direction'],
                        'outputCount': slot_rs['data']['outputCount'],
                        'isInterlace': output_item['timing']['isInterlace']
                    }
                    output_item['timing']['horizontal']['addrTime'] *= 4
                    output_item['timing']['horizontal']['frontPorch'] *= 4
                    output_item['timing']['horizontal']['sync'] *= 4
                    output_item['timing']['horizontal']['totalTime'] *= 4
        return output_item

    def write_general(self, device_id, output_id, data):
        """
        设置输出基本信息
        清除输出列表缓存
        清除输出详情缓存
        :param device_id: 设备id
        :param output_id: 输出id
        :param data: 基本信息
        :return: 结果对象
        """
        rs = self.read_origin_detail(device_id, output_id)
        if rs['status'] == 0 and rs['data']:
            # 是否二合一卡
            if rs['data']['modelId'] in ModelId.Two_In_One_Card_List:
                rs = self._samecard_tio_outputs(device_id, output_id)
                if rs['status'] == 0 and rs['data']:
                    _, outputs = rs['data']
                    for o_id in outputs:
                        data['outputId'] = o_id
                        rs = output.output_general_write(device_id, o_id, data)
                        if rs['status'] == 0:
                            cacher.OutputCache.delete(CacheKey.output_detail(device_id, o_id))
            # 是否DL output卡
            elif rs['data']['modelId'] in ModelId.DL_Output_Card_List:
                dl_rs = self.read_dl_outputs(device_id)
                if dl_rs['status'] == 0:
                    rs = output.output_general_write(device_id, output_id, data)
                    if rs['status'] == 0:
                        cacher.OutputCache.delete(CacheKey.output_detail(device_id, output_id))
                    if output_id in dl_rs['data']['dlList']:
                        data['outputId'] = output_id-1
                        if 'name' in data:
                            data.pop('name')
                        if len(data) > 2:
                            rs = output.output_general_write(device_id, output_id-1, data)
                            if rs['status'] == 0:
                                cacher.OutputCache.delete(CacheKey.output_detail(device_id, output_id-1))
            # 是否4k卡
            elif rs['data']['modelId'] in [ModelId.Output_2_HDMI20_Card, ModelId.Pd_Output_2_HDMI20_Card]:
                k4_rs = self._samecard_k4_outputs(device_id, output_id)
                if k4_rs['status'] == 0 and k4_rs['data']:
                    _, outputs = k4_rs['data']
                    for o_id in outputs:
                        data['outputId'] = o_id
                        rs = output.output_general_write(device_id, o_id, data)
                        if rs['status'] == 0:
                            cacher.OutputCache.delete(CacheKey.output_detail(device_id, o_id))
            # 是否12G sdi 3840 2160 23
            elif rs['data']['modelId'] == ModelId.Output_12_3G_SDI_Card:
                slot_id = rs['data']['slotId']
                slot_rs = self._read_tio_payload(device_id, slot_id)
                if slot_rs['status'] == 0:
                    width = slot_rs['data']['width']
                    height = slot_rs['data']['height']
                    refreshRate = slot_rs['data']['refresh']
                    if width >= Output12GSDIConst.LimitWidth and \
                            height >= Output12GSDIConst.LimitHeight and \
                            refreshRate > Output12GSDIConst.LimitRefresh:
                        k4_rs = self._samecard_k4_outputs(device_id, output_id)
                        if k4_rs['status'] == 0 and k4_rs['data']:
                            _, outputs = k4_rs['data']
                            for o_id in outputs:
                                data['outputId'] = o_id
                                rs = output.output_general_write(device_id, o_id, data)
                                if rs['status'] == 0:
                                    cacher.OutputCache.delete(CacheKey.output_detail(device_id, o_id))
                    else:
                        rs = output.output_general_write(device_id, output_id, data)
                        if rs['status'] == 0:
                            cacher.OutputCache.delete(CacheKey.output_detail(device_id, output_id))
            # 普通卡
            else:
                rs = output.output_general_write(device_id, output_id, data)
                if rs['status'] == 0:
                    cacher.OutputCache.delete(CacheKey.output_detail(device_id, output_id))
            cacher.OutputCache.delete(CacheKey.output_list(device_id))

        return rs

    def write_test_pattern(self, device_id, output_id, data):
        """
        设置输出测试画面
        清除输出详情缓存
        :param device_id: 设备id
        :param output_id: 输出id
        :param data: 测试画面
        :return: 结果对象
        """
        rs = self.read_origin_detail(device_id, output_id)
        if rs['status'] == 0 and rs['data']:
            # 是否二合一卡
            if rs['data']['modelId'] in ModelId.Two_In_One_Card_List:
                rs = self._samecard_tio_outputs(device_id, output_id)
                if rs['status'] == 0 and rs['data']:
                    _, outputs = rs['data']
                    for o_id in outputs:
                        data['outputId'] = o_id
                        rs = output.output_test_pattern_write(device_id, o_id, data)
                        if rs['status'] == 0:
                            cacher.OutputCache.delete(CacheKey.output_detail(device_id, o_id))
            # 是否DL output卡
            elif rs['data']['modelId'] in ModelId.DL_Output_Card_List:
                dl_rs = self.read_dl_outputs(device_id)
                if dl_rs['status'] == 0:
                    rs = output.output_test_pattern_write(device_id, output_id, data)
                    if rs['status'] == 0:
                        cacher.OutputCache.delete(CacheKey.output_detail(device_id, output_id))
                    if output_id in dl_rs['data']['dlList']:
                        data['outputId'] = output_id - 1
                        rs = output.output_test_pattern_write(device_id, output_id - 1, data)
                        if rs['status'] == 0:
                            cacher.OutputCache.delete(CacheKey.output_detail(device_id, output_id - 1))

            # 是否4k卡
            elif rs['data']['modelId'] in [ModelId.Output_2_HDMI20_Card, ModelId.Pd_Output_2_HDMI20_Card]:
                k4_rs = self._samecard_k4_outputs(device_id, output_id)
                if k4_rs['status'] == 0 and k4_rs['data']:
                    _, outputs = k4_rs['data']
                    for o_id in outputs:
                        data['outputId'] = o_id
                        rs = output.output_test_pattern_write(device_id, o_id, data)
                        if rs['status'] == 0:
                            cacher.OutputCache.delete(CacheKey.output_detail(device_id, o_id))
            # 是否12G sdi 3840 2160 23
            elif rs['data']['modelId'] == ModelId.Output_12_3G_SDI_Card:
                slot_id = rs['data']['slotId']
                slot_rs = self._read_tio_payload(device_id, slot_id)
                if slot_rs['status'] == 0:
                    width = slot_rs['data']['width']
                    height = slot_rs['data']['height']
                    refreshRate = slot_rs['data']['refresh']
                    if width >= Output12GSDIConst.LimitWidth and \
                            height >= Output12GSDIConst.LimitHeight and \
                            refreshRate > Output12GSDIConst.LimitRefresh:
                        k4_rs = self._samecard_k4_outputs(device_id, output_id)
                        if k4_rs['status'] == 0 and k4_rs['data']:
                            _, outputs = k4_rs['data']
                            for o_id in outputs:
                                data['outputId'] = o_id
                                rs = output.output_test_pattern_write(device_id, o_id, data)
                                if rs['status'] == 0:
                                    cacher.OutputCache.delete(CacheKey.output_detail(device_id, o_id))
                    else:
                        rs = output.output_test_pattern_write(device_id, output_id, data)
                        if rs['status'] == 0:
                            cacher.OutputCache.delete(CacheKey.output_detail(device_id, output_id))
            # 普通卡
            else:
                rs = output.output_test_pattern_write(device_id, output_id, data)
                if rs['status'] == 0:
                    cacher.OutputCache.delete(CacheKey.output_detail(device_id, output_id))
            cacher.OutputCache.delete(CacheKey.output_list(device_id))
        return rs

    def write_timing(self, device_id, output_id, data):
        """
        设置输出timing
        清除全部缓存
        :param device_id: 设备id
        :param output_id: 输出id
        :param data: timing信息
        :return: 结果对象
        """
        rs = get_result_model()
        dl_rs = self.read_dl_outputs(device_id)
        if dl_rs['status'] == 0:
            if output_id in dl_rs['data']['dlList']:

                data['horizontal']['addrTime'] //= 2
                data['horizontal']['frontPorch'] //= 2
                data['horizontal']['sync'] //= 2
                data['horizontal']['totalTime'] //= 2

                # data['vertical']['addrTime'] //= 2
                # data['vertical']['frontPorch'] //= 2
                # data['vertical']['sync'] //= 2
                # data['vertical']['totalTime'] //= 2

                t = deepcopy(data)
                print("下发的数据为====", t)
                rs = output.output_timing_write(device_id, output_id, t)
                if rs['status'] == 0:
                    t = deepcopy(data)
                    t['outputId'] = output_id - 1
                    print("再次下发的数据为====", t)
                    rs = output.output_timing_write(device_id, output_id-1, t)
            else:
                rs = self.read_4k_outputs(device_id)
                if rs['status'] != 0:
                    return rs
                if rs['data']:
                    for slotId, output_id_info in rs['data'].items():
                        slot_rs = device_api.read_slot(device_id, slotId)
                        if slot_rs['status'] != 0:
                            return get_result_model(ResInfo.Device_Detail_Error)
                        model_id = slot_rs['data']['modelId']
                        if model_id == ModelId.Output_12_3G_SDI_Card:
                            for x in output_id_info:
                                if x == output_id:
                                    resolution = {
                                        'width': data['horizontal']['addrTime'],
                                        'height': data['vertical']['addrTime'],
                                        'refresh': data['refreshRate'],
                                        'outputCount': 1
                                    }
                                    data['resolution'] = resolution
                                    t = {
                                        'deviceId': device_id,
                                        'slotId': slotId,
                                        'resolution': {
                                            'width': data['horizontal']['addrTime'],
                                            'height':  data['vertical']['addrTime'],
                                            'refresh': data['refreshRate'],
                                            'direction':0,
                                            'outputCount': 1
                                        }
                                }
                                    rs = device_api.write_slot(device_id, slotId, t)
                                    if rs['status'] != 0:
                                        return get_result_model(ResInfo.Output_Timing_Err)
                rs = output.output_timing_write(device_id, output_id, data)
        cacher.clear_all()
        return rs


    def write_timing_original(self, device_id, output_id, data):
        rs = output.output_timing_write(device_id, output_id, data)
        if rs['status'] == 0:
            cacher.clear_all()
        return rs

    def write_image_quality(self, device_id, output_id, data):
        """
        设置画质信息
        清除输出详情缓存
        :param device_id: 设备id
        :param output_id: 输出id
        :param data: 画质
        :return: 结果对象
        """
        rs = self.read_origin_detail(device_id, output_id)
        if rs['status'] == 0 and rs['data']:
            # 是否二合一卡
            if rs['data']['modelId'] in ModelId.Two_In_One_Card_List:
                rs = self._samecard_tio_outputs(device_id, output_id)
                if rs['status'] == 0 and  rs['data']:
                    _, outputs = rs['data']
                    for o_id in outputs:
                        data['outputId'] = o_id
                        rs = output.output_image_quality_write(device_id, o_id, data)
                        if rs['status'] == 0:
                            cacher.OutputCache.delete(CacheKey.output_detail(device_id, o_id))
            # 是否DL output卡
            elif rs['data']['modelId'] in ModelId.DL_Output_Card_List:
                dl_rs = self.read_dl_outputs(device_id)
                if dl_rs['status'] == 0:
                    rs = output.output_image_quality_write(device_id, output_id, data)
                    if rs['status'] == 0:
                        cacher.OutputCache.delete(CacheKey.output_detail(device_id, output_id))
                    if output_id in dl_rs['data']['dlList']:
                        data['outputId'] = output_id - 1
                        rs = output.output_image_quality_write(device_id, output_id - 1, data)
                        if rs['status'] == 0:
                            cacher.OutputCache.delete(CacheKey.output_detail(device_id, output_id - 1))
            # 是否4k卡
            elif rs['data']['modelId'] in [ModelId.Output_2_HDMI20_Card, ModelId.Pd_Output_2_HDMI20_Card]:
                k4_rs = self._samecard_k4_outputs(device_id, output_id)
                if k4_rs['status'] == 0 and k4_rs['data']:
                    _, outputs = k4_rs['data']
                    for o_id in outputs:
                        data['outputId'] = o_id
                        rs = output.output_image_quality_write(device_id, o_id, data)
                        if rs['status'] == 0:
                            cacher.OutputCache.delete(CacheKey.output_detail(device_id, o_id))
            # 是否12G sdi 3840 2160 23
            elif rs['data']['modelId'] == ModelId.Output_12_3G_SDI_Card:
                slot_id = rs['data']['slotId']
                slot_rs = self._read_tio_payload(device_id, slot_id)
                if slot_rs['status'] == 0:
                    width = slot_rs['data']['width']
                    height = slot_rs['data']['height']
                    refreshRate = slot_rs['data']['refresh']
                    if width >= Output12GSDIConst.LimitWidth and \
                            height >= Output12GSDIConst.LimitHeight and \
                            refreshRate > Output12GSDIConst.LimitRefresh:
                        k4_rs = self._samecard_k4_outputs(device_id, output_id)
                        if k4_rs['status'] == 0 and k4_rs['data']:
                            _, outputs = k4_rs['data']
                            for o_id in outputs:
                                data['outputId'] = o_id
                                rs = output.output_image_quality_write(device_id, o_id, data)
                                if rs['status'] == 0:
                                    cacher.OutputCache.delete(CacheKey.output_detail(device_id, o_id))
                    else:
                        rs = output.output_image_quality_write(device_id, output_id, data)
                        if rs['status'] == 0:
                            cacher.OutputCache.delete(CacheKey.output_detail(device_id, output_id))
            # 普通卡
            else:
                rs = output.output_image_quality_write(device_id, output_id, data)
                if rs['status'] == 0:
                    cacher.OutputCache.delete(CacheKey.output_detail(device_id, output_id))
            cacher.OutputCache.delete(CacheKey.output_list(device_id))

        return rs


    def write_gamma(self, device_id, output_id, data):
        """
        设置gamma信息
        清除输出详情缓存
        :param device_id: 设备id
        :param output_id: 输出id
        :param data: gamma信息
        :return: 结果对象
        """
        rs = self.read_origin_detail(device_id, output_id)
        if rs['status'] == 0 and rs['data']:
            # 是否二合一卡
            if rs['data']['modelId'] in ModelId.Two_In_One_Card_List:
                rs = self._samecard_tio_outputs(device_id, output_id)
                if rs['status'] == 0 and rs['data']:
                    _, outputs = rs['data']
                    for o_id in outputs:
                        gamma = deepcopy(data)
                        gamma['outputId'] = o_id
                        rs = output.output_gamma_write(device_id, o_id, gamma)
                        if rs['status'] == 0:
                            cacher.OutputCache.delete(CacheKey.output_detail(device_id, o_id))
            # 是否DL output卡
            elif rs['data']['modelId'] in ModelId.DL_Output_Card_List:
                dl_rs = self.read_dl_outputs(device_id)
                if dl_rs['status'] == 0:
                    gamma = deepcopy(data)
                    rs = output.output_gamma_write(device_id, output_id, gamma)
                    if rs['status'] == 0:
                        cacher.OutputCache.delete(CacheKey.output_detail(device_id, output_id))
                    if output_id in dl_rs['data']['dlList']:
                        gamma = deepcopy(data)
                        gamma['outputId'] = output_id - 1
                        rs = output.output_gamma_write(device_id, output_id - 1, gamma)
                        if rs['status'] == 0:
                            cacher.OutputCache.delete(CacheKey.output_detail(device_id, output_id - 1))
            # 是否4k卡
            elif rs['data']['modelId'] in [ModelId.Output_2_HDMI20_Card, ModelId.Pd_Output_2_HDMI20_Card]:
                k4_rs = self._samecard_k4_outputs(device_id, output_id)
                if k4_rs['status'] == 0 and k4_rs['data']:
                    _, outputs = k4_rs['data']
                    for o_id in outputs:
                        gamma = deepcopy(data)
                        gamma['outputId'] = o_id
                        rs = output.output_gamma_write(device_id, o_id, gamma)
                        if rs['status'] == 0:
                            cacher.OutputCache.delete(CacheKey.output_detail(device_id, o_id))
            # 是否12G sdi 3840 2160 23
            elif rs['data']['modelId'] == ModelId.Output_12_3G_SDI_Card:
                slot_id = rs['data']['slotId']
                slot_rs = self._read_tio_payload(device_id, slot_id)
                if slot_rs['status'] == 0:
                    width = slot_rs['data']['width']
                    height = slot_rs['data']['height']
                    refreshRate = slot_rs['data']['refresh']
                    if width >= Output12GSDIConst.LimitWidth and \
                            height >= Output12GSDIConst.LimitHeight and \
                            refreshRate > Output12GSDIConst.LimitRefresh:
                        k4_rs = self._samecard_k4_outputs(device_id, output_id)
                        if k4_rs['status'] == 0 and k4_rs['data']:
                            _, outputs = k4_rs['data']
                            for o_id in outputs:
                                gamma = deepcopy(data)
                                gamma['outputId'] = o_id
                                rs = output.output_gamma_write(device_id, o_id, gamma)
                                if rs['status'] == 0:
                                    cacher.OutputCache.delete(CacheKey.output_detail(device_id, o_id))
                    else:
                        rs = output.output_gamma_write(device_id, output_id, data)
                        if rs['status'] == 0:
                            cacher.OutputCache.delete(CacheKey.output_detail(device_id, output_id))
            # 普通卡
            else:
                rs = output.output_gamma_write(device_id, output_id, data)
                if rs['status'] == 0:
                    cacher.OutputCache.delete(CacheKey.output_detail(device_id, output_id))
            cacher.OutputCache.delete(CacheKey.output_list(device_id))
        return rs

    def read_fixed_position(self, device_id, output_id):
        """
        读取输出定位
        :param device_id: 设备id
        :param output_id: 输出id
        :return: 结果对象
        """
        return output.output_fixed_position_read(device_id, output_id)

    def write_fixed_position(self, device_id, output_id, data):
        """
        设置输出定位
        :param device_id: 设备id
        :param output_id: 输出id
        :param data: 输出定位
        :return: 结果对象
        """
        return output.output_fixed_position_write(device_id, output_id, data)

    def write_output_timing_free_mapping(self, device_id, data):
        rs = output.output_timing_free_mapping(device_id, 0, data)
        if rs['status'] == 0:
            cacher.OutputCache.clear()
        return rs

    def read_edid(self, device_id, output_id):
        """
        读取edid
        :param device_id: 设备id
        :param output_id: 输出id
        :return: 结果对象
        """
        return output.output_edid_read(device_id, output_id)

    def write_payload(self, device_id, output_list, params):
        if output_list:
            try:
                rs = self.two_in_one_outputs(device_id)
                # 读取二合一卡信息失败，直接返回
                if rs['status'] != 0:
                    return rs
                # 设备没有二合一卡，直接返回
                if not rs['data']:
                    return get_result_model(ResInfo.Device_Slot_Not_Exist)
                tio_dict = rs['data']

                o_rs = self.read_detail(device_id, output_list[0])
                if o_rs['status'] != 0:
                    return get_result_model(ResInfo.Output_Timing_Err)
                model_id = o_rs['data']['modelId']
                card_mode = o_rs['data']['cardMode']
                if model_id == ModelId.Output_4_Fiber_Card:
                    card_type = card_mode
                    timing_rs = self._get_four_fiber_output_timing(card_type, params['width'], params['height'], params['refreshRate'])
                else:
                    if model_id in [ModelId.Output_Two_In_One_Card, ModelId.Pd_Output_Two_In_One_Card]:
                        card_type = OutputCardMode.OutputTwoInOneCard
                    elif model_id in [ModelId.Output_Two_In_One_Card_20, ModelId.Pd_Output_Two_In_One_Card_20]:
                        card_type = OutputCardMode.OutputTwoInOneCard_20
                    else:
                        return get_result_model(ResInfo.Middle_Data_Err)
                    timing_rs = self._get_tio_output_timing(card_type, params['width'], params['height'],
                                                            params['refreshRate'])
                # 计算timing失败
                if timing_rs['status'] != 0:
                    return timing_rs

                timing = timing_rs['data']
                direction = timing_rs['data'].pop('direction')
                params['direction'] = direction
                logger.info("二合一卡设置带载output：%s， timing: %s" % (output_list, timing))
                # 循环output列表
                for output_id in output_list:
                    tio_data = self._get_twoinone_card_sorted_output(output_id, tio_dict)
                    if tio_data is None:  # 不是二合一卡，返回失败
                        return get_result_model(ResInfo.Output_Timing_Err)
                    else:
                        slot_id, outputs = tio_data
                        rs = self._write_single_tio_payload(device_id, slot_id, outputs, timing, params)
                        if rs['status'] != 0:  # 设置单个二合一卡失败，返回失败
                            return get_result_model(ResInfo.Output_Timing_Err)
            except Exception as e:
                logger.error("设置二合一卡带载错误")
                logger.exception(e)
                return get_result_model(ResInfo.Output_Timing_Err)
            finally:
                cacher.clear_all()
        return get_result_model()

    def _write_single_tio_payload(self, device_id, slot_id, outputs, timing, resolution):
        """
        设置单个二合一卡的带载信息
        :param device_id:
        :param slot_id:
        :param outputs:
        :param timing:
        :return:
        """
        timing['deviceId'] = device_id
        partial = False
        for o_id in outputs:
            timing['outputId'] = o_id
            rs = self.write_timing(device_id, o_id, deepcopy(timing))
            if rs['status'] != 0:
                partial = True

        # 修改slot中的宽高
        rs = self._write_tio_payload(device_id, slot_id, resolution)
        if rs['status'] == 0 and partial:
            rs = get_result_model(ResInfo.Output_Write_Timing_List_Error)
        return rs

    def _get_tio_output_timing(self, card_type, full_width, full_height, output_refresh):
        """
        1.是否超出最大带载 数据限制：
            是否超出最大带载？1041W*60 / 1312W*60
            否：是否满足宽高限制 宽（最大/最小）：2560*4/800 高：8192/600
                否：不能带载
                是：是否满足时钟频率限制：Htotal*Vtotal*refresh <165M
                    否：不能带载
                    是：正常逻辑

        2. 设置二合一卡的负载信息
        屏幕是否大于1920*1080*60 ？
            是：
                水平是否大于1920？
                    是：水平四分割 (水平建屏）(优先CVT规则，满足不了的情况下，采用减少水平逆程的规则）
                    否：屏幕是否大于1920*1080*60*3 ？
                        是：  使用实际水平大小向上对2取整且垂直建屏并做截取，单接口可支持垂直8192(按照CVT规则计算，一个接口不满足，采用两个接口建屏，依次类推，在4个接口垂直建屏的情况下，如果CVT规则满足不了，尝试采用减少水平逆程规则）（垂直建屏）
                        否：  水平四分割 (水平建屏）(优先CVT规则，满足不了的情况下，采用减少水平逆程的规则）
            否：
                水平是否大于1920 ？
                    是：使用多个水平2560并做截取（水平建屏）
                    否：htotal*vtotal*refresh <165M ?
                        是 ：    使用实际大小向上对2取整建屏
                        否  ： （垂直建屏）
                            使用实际水平大小向上对2取整且垂直建屏并做截取，单接口可支持垂直8192(按照CVT规则计算，一个接口不满足，采用两个接口建屏，依次类推）


        :param full_width: 二合一卡总体宽度
        :param full_height: 二合一卡总体高度
        :param output_refresh: 二合一卡刷新率
        :return: 结果对象
        """
        if card_type == OutputCardMode.OutputTwoInOneCard_20:
            two_in_one_card_data = TwoInOneCardData_1300
        elif card_type == OutputCardMode.OutputTwoInOneCard:
            two_in_one_card_data = TwoInOneCardData_1040
        else:
            return get_result_model(ResInfo.Middle_Data_Err)

        standard_timing = None
        direction = 0
        full_area = output_refresh * full_width * full_height
        timing_height = full_height
        # 判断是否在带载范围内
        slot_max_area = two_in_one_card_data.SlotMaxArea
        if full_area > slot_max_area \
                or full_width > two_in_one_card_data.SlotMaxWidth \
                or full_height > two_in_one_card_data.SlotMaxHeigth\
                or full_width < two_in_one_card_data.SlotMinWidth\
                or full_height < two_in_one_card_data.SlotMinHeight:
            return get_result_model(ResInfo.Output_Max_Payload_Exceed)

        # 大于1920*1080*60， 一个放不下
        if full_area > two_in_one_card_data.SingleOutputDefaultResolution:

            #  # 水平大于1920 或 总体小于1920*1080*60*3
            if full_width > two_in_one_card_data.SingleOutputStandardWidth\
                    or full_area <= two_in_one_card_data.SingleOutputDefaultResolution * 3:
                # 水平四分割 (水平建屏）(优先CVT规则，满足不了的情况下，采用减少水平逆程的规则）
                new_width = math.ceil(math.ceil(full_width / 4) / 4) * 4
                # 修改分辨率值
                timing_width = new_width
            # 否
            else:
                direction = 1
                timing_width = math.ceil(full_width / 2) * 2
                # 单接口可支持垂直8192（按照cvt规则计算，一个接口不满足采用2-4个接口，cvt满足不了尝试减少水平逆程的规则）
                for x in range(1, 5):
                    timing_height = math.ceil(full_height / x)
                    standard_timing = outputResolution(timing_width, timing_height, output_refresh, two_in_one_card_data.TimingMaxBandWidth_1)
                    if standard_timing:
                        break
                # cvt算法不能满足，则使用自定义算法
                if not standard_timing:
                    timing_height = math.ceil(full_height / 4)
                    standard_timing = outputCustomResolution(timing_width, timing_height, output_refresh, two_in_one_card_data.TimingMaxBandWidth_1)
                    if not standard_timing:  # 不能带载，直接返回
                        return get_result_model(ResInfo.Output_Max_Payload_Exceed)

        # 小于1920*1080*60，一个放得下
        # 宽度大于单接口最大宽度2560，使用多个水平2560并做截取（水平建屏）
        elif full_width > two_in_one_card_data.SingleOutputMaxWidth:
            timing_width = two_in_one_card_data.SingleOutputMaxWidth
            timing_height = full_height
        # 一个放的下，宽度小于2560
        else:
            timing_width = math.ceil(full_width / 2) * 2
            timing_height = full_height
            standard_timing = outputResolution(timing_width, timing_height, output_refresh, two_in_one_card_data.TimingMaxBandWidth_2)

            # htotal*vtotal*refresh >165M
            # 计算timing，如果timing为空
            if not standard_timing:  # 垂直建屏
                direction = 1
                for x in range(2, 5):
                    timing_height = math.ceil(full_height / x)
                    standard_timing = outputResolution(timing_width, timing_height, output_refresh,
                                                       two_in_one_card_data.TimingMaxBandWidth_2)
                    if standard_timing:
                        break
                if not standard_timing:  # 不能带载，直接返回
                    return get_result_model(ResInfo.Output_Max_Payload_Exceed)

        # 新的算法计算Timing
        if not standard_timing:
            standard_timing = outputResolution(timing_width, timing_height, output_refresh, two_in_one_card_data.TimingMaxBandWidth_1)
        if not standard_timing:
            standard_timing = outputCustomResolution(timing_width, timing_height, output_refresh, two_in_one_card_data.TimingMaxBandWidth_1)
        if standard_timing:
            rs = get_result_model()
            standard_timing['direction'] = direction
            rs['data'] = standard_timing
            return rs

        return get_result_model(ResInfo.Output_Max_Payload_Exceed)

    def _get_four_fiber_output_timing(self, card_type, full_width, full_height, output_refresh):
        """
        1.是否超出最大带载 数据限制：
            是否超出最大带载？2081W*60
            否：是否满足宽高限制 宽（最大/最小）：4096*4/800 高：16384/600
                否：不能带载
                是：是否满足时钟频率限制：Htotal*Vtotal*refresh <330M(8bit模式）
                    否：不能带载
                    是：正常逻辑

        2. 设置建屏信息
        屏幕是否大于1920*1080*60 ？
            是：
                水平是否大于2560？
                    是：水平四分割 (水平建屏）(优先CVT规则，满足不了的情况下，采用减少水平逆程的规则）
                    否：使用实际水平大小向上对2取整且垂直建屏并做截取，
                        单接口可支持垂直8192(按照CVT规则计算，垂直4分割，在4个接口垂直建屏的情况下，如果CVT规则满足不了，尝试采用减少水平逆程规则）（垂直建屏）

            否：
                水平是否大于2560 ？
                    是：使用多个水平2560并做截取（水平建屏）
                    否：htotal*vtotal*refresh <330M ?
                        是 ：  使用实际大小向上对2取整建屏
                        否  ： （垂直建屏）
                            使用实际水平大小向上对2取整且垂直建屏并做截取，单接口可支持垂直8192(按照CVT规则计算）
        :param full_width: 二合一卡总体宽度
        :param full_height: 二合一卡总体高度
        :param output_refresh: 二合一卡刷新率
        :return: 结果对象
        """
        if card_type == OutputCardMode.IndependentMode:
            two_in_one_card_data = FourFiberCardData
        elif card_type == OutputCardMode.BackupMode:
            two_in_one_card_data = FourFiberCardBackupData
        elif card_type == OutputCardMode.CopyMode:
            two_in_one_card_data = FourFiberCardCopyData
        else:
            return get_result_model(ResInfo.Middle_Data_Err)

        standard_timing = None
        direction = 0
        full_area = output_refresh * full_width * full_height
        timing_height = full_height
        # 判断是否在带载范围内
        slot_max_area = two_in_one_card_data.SlotMaxArea
        if full_area > slot_max_area \
                or full_width > two_in_one_card_data.SlotMaxWidth \
                or full_height > two_in_one_card_data.SlotMaxHeigth \
                or full_width < two_in_one_card_data.SlotMinWidth \
                or full_height < two_in_one_card_data.SlotMinHeight:
            return get_result_model(ResInfo.Output_Max_Payload_Exceed)

        # 大于1920*1080*60， 一个放不下
        if full_area > two_in_one_card_data.SingleOutputDefaultResolution:
            if full_width > two_in_one_card_data.SingleOutputStandardWidth:
                # 水平四分割 (水平建屏）(优先CVT规则，满足不了的情况下，采用减少水平逆程的规则）
                new_width = math.ceil(math.ceil(full_width / 4) / 4) * 4
                # 修改分辨率值
                timing_width = new_width
            # 否
            else:
                direction = 1
                timing_width = math.ceil(full_width / 2) * 2
                # 单接口可支持垂直8192（按照cvt规则计算，一个接口不满足采用2-4个接口，cvt满足不了尝试减少水平逆程的规则）
                timing_height = math.ceil(full_height / 4)
                standard_timing = outputResolution(timing_width, timing_height, output_refresh,
                                                   two_in_one_card_data.TimingMaxBandWidth_1)
                # cvt算法不能满足，则使用自定义算法
                if not standard_timing:
                    timing_height = math.ceil(full_height / 4)
                    standard_timing = outputFourFiberCustomResolution(timing_width, timing_height, output_refresh,
                                                             two_in_one_card_data.TimingMaxBandWidth_1)

                    if not standard_timing:  # 不能带载，直接返回
                        return get_result_model(ResInfo.Output_Max_Payload_Exceed)

        # 小于1920*1080*60，一个放得下
        # 宽度大于单接口最大宽度2560，使用多个水平2560并做截取（水平建屏）
        elif full_width > two_in_one_card_data.SingleOutputMaxWidth:
            timing_width = two_in_one_card_data.SingleOutputMaxWidth
            timing_height = full_height
        # 一个放的下，宽度小于2560
        else:
            timing_width = math.ceil(full_width / 2) * 2
            timing_height = full_height
            standard_timing = outputResolution(timing_width, timing_height, output_refresh,
                                               two_in_one_card_data.TimingMaxBandWidth_2)

            # htotal*vtotal*refresh >165M
            # 计算timing，如果timing为空
            if not standard_timing:  # 垂直建屏
                direction = 1
                timing_height = math.ceil(full_height / 4)
                standard_timing = outputResolution(timing_width, timing_height, output_refresh,
                                                       two_in_one_card_data.TimingMaxBandWidth_2)

                if not standard_timing:  # 不能带载，直接返回
                    return get_result_model(ResInfo.Output_Max_Payload_Exceed)

        # 新的算法计算Timing
        if not standard_timing:
            standard_timing = outputResolution(timing_width, timing_height, output_refresh,
                                               two_in_one_card_data.TimingMaxBandWidth_1)
        if not standard_timing:
            standard_timing = outputFourFiberCustomResolution(timing_width, timing_height, output_refresh,
                                                     two_in_one_card_data.TimingMaxBandWidth_1)

        if standard_timing:
            rs = get_result_model()
            standard_timing['direction'] = direction
            rs['data'] = standard_timing
            return rs
        return get_result_model(ResInfo.Output_Max_Payload_Exceed)

    def _get_4k_output_timing(self, width, height, refresh, timing):
        if timing:
            standard_timing = timing
        else:
            standard_timing = outputResolution(width, height, refresh, Output4KCardData.TimingMaxBandWidth)

        rs = get_result_model()
        if not standard_timing:
            rs['status'] = ResInfo.Output_Max_Payload_Exceed
        elif (float(standard_timing['horizontal']["totalTime"])
            * float(standard_timing['vertical']["totalTime"])
            * float(standard_timing['refreshRate']) > Output4KCardData.TimingMaxBandWidth):
                rs['status'] = ResInfo.Output_Max_Payload_Exceed
        else:
            rs['data'] = standard_timing
        return rs

    def output_write_4k_timing(self, device_id, output_list, params):
        if output_list:
            try:
                rs = self.read_4k_outputs(device_id)
                # 读取卡信息失败，直接返回
                if rs['status'] != 0:
                    return rs
                # 设备没有4k卡，直接返回
                if not rs['data']:
                    return get_result_model(ResInfo.Device_Slot_Not_Exist)
                k4_dict = rs['data']

                resolution = {
                    'width': params['width'],
                    'height': params['height'],
                    'refreshRate': params['refreshRate'],
                    'outputCount': 1
                }
                timing_rs = self._get_4k_output_timing(params['width'], params['height'],
                                                    params['refreshRate'], params.get('timing', None))
                # 计算timing失败
                if timing_rs['status'] != 0:
                    return timing_rs

                timing = timing_rs['data']
                if (float(timing['horizontal']["totalTime"])
                        * float(timing['vertical']["totalTime"])
                        * float(timing['refreshRate']) > Output4KCardData.SingleOutputTimingMaxBandWidth):
                    resolution['outputCount'] = 4
                    timing['horizontal']['addrTime'] = math.ceil(timing['horizontal']['addrTime'] / 16) * 4
                    timing['horizontal']["frontPorch"] = math.ceil(timing['horizontal']["frontPorch"] / 4)
                    timing['horizontal']["totalTime"] = math.ceil(timing['horizontal']["totalTime"] / 4)
                    timing['horizontal']["sync"] = math.ceil(timing['horizontal']["sync"] / 4)

                logger.info("4k卡设置带载output：%s， timing: %s" % (output_list, timing))
                # 循环output列表
                for output_id in output_list:
                    tio_data = self._get_twoinone_card_sorted_output(output_id, k4_dict)
                    if tio_data is None:  # 不是二合一卡，返回失败
                        return get_result_model(ResInfo.Output_Timing_Err)
                    else:
                        slot_id, outputs = tio_data
                        timing['deviceId'] = device_id
                        if 'isInterlace' in params:
                            timing['isInterlace'] = params['isInterlace']
                        rs = self._write_tio_payload(device_id, slot_id, resolution)
                        if rs['status'] != 0:
                            return get_result_model(ResInfo.Output_Timing_Err)
                        for out in outputs:
                            timing['outputId'] = out
                            print("output_id==",out)
                            print("timing==", deepcopy(timing))
                            rs = self.write_timing_original(device_id, out, deepcopy(timing))
                            if rs['status'] != 0:  # 设置单个二合一卡失败，返回失败
                                return get_result_model(ResInfo.Output_Timing_Err)

                        if rs['status'] != 0:
                            return get_result_model(ResInfo.Output_Timing_Err)
            except Exception as e:
                logger.error("设置4k卡带载错误")
                logger.exception(e)
                return get_result_model(ResInfo.Output_Timing_Err)
            finally:
                cacher.clear_all()
        return get_result_model()

    def output_write_sdl_timing(self, device_id, output_list, params):
        if output_list:
            try:
                rs = self.read_4k_outputs(device_id)
                # 读取卡信息失败，直接返回
                if rs['status'] != 0:
                    return rs
                # 设备没有4k卡，直接返回
                if not rs['data']:
                    return get_result_model(ResInfo.Device_Slot_Not_Exist)
                k4_dict = rs['data']

                resolution = {
                    'width': params['width'],
                    'height': params['height'],
                    'refreshRate': params['refreshRate'],
                    'outputCount': 1
                }
                timing_rs = self._get_4k_output_timing(params['width'], params['height'],
                                                    params['refreshRate'], params.get('timing', None))
                # 计算timing失败
                if timing_rs['status'] != 0:
                    return timing_rs

                timing = timing_rs['data']
                if (float(timing['horizontal']["totalTime"])
                        * float(timing['vertical']["totalTime"])
                        * float(timing['refreshRate']) > Output4KCardData.SingleOutputTimingMaxBandWidth):
                    resolution['outputCount'] = 4
                    timing['horizontal']['addrTime'] = math.ceil(timing['horizontal']['addrTime'] / 16) * 4
                    timing['horizontal']["frontPorch"] = math.ceil(timing['horizontal']["frontPorch"] / 4)
                    timing['horizontal']["totalTime"] = math.ceil(timing['horizontal']["totalTime"] / 4)
                    timing['horizontal']["sync"] = math.ceil(timing['horizontal']["sync"] / 4)

                logger.info("4k卡设置带载output：%s， timing: %s" % (output_list, timing))
                # 循环output列表
                for output_id in output_list:
                    tio_data = self._get_twoinone_card_sorted_output(output_id, k4_dict)
                    if tio_data is None:  # 不是二合一卡，返回失败
                        return get_result_model(ResInfo.Output_Timing_Err)
                    else:
                        slot_id, outputs = tio_data
                        timing['deviceId'] = device_id
                        rs = self._write_tio_payload(device_id, slot_id, resolution)
                        if rs['status'] != 0:
                            return get_result_model(ResInfo.Output_Timing_Err)
                        for out in outputs:
                            timing['outputId'] = out
                            rs = self.write_timing_original(device_id, out, deepcopy(timing))
                            if rs['status'] != 0:  # 设置单个二合一卡失败，返回失败
                                return get_result_model(ResInfo.Output_Timing_Err)

                        if rs['status'] != 0:
                            return get_result_model(ResInfo.Output_Timing_Err)
            except Exception as e:
                logger.error("设置4k卡带载错误")
                logger.exception(e)
                return get_result_model(ResInfo.Output_Timing_Err)
            finally:
                cacher.clear_all()
        return get_result_model()

output_api = OutputApi()


def two_in_one_card_outputs(device_id):
    """
    二合一卡和对应的output列表
    :param device_id:
    :return:
    """
    return output_api.two_in_one_outputs(device_id)


def read_output_origin_resolution(device_id, output_id):
    """
    读取output刷新率
    :param device_id:
    :param output_id:
    :return:
    """
    rs = output_api.read_origin_detail(device_id, output_id)
    if rs['status'] == 0:
        rs['data'] = {
            'refreshRate': rs['data']['timing']['refreshRate'],
            'width': rs['data']['timing']['horizontal']['addrTime'],
            'height': rs['data']['timing']['vertical']['addrTime']
        }
    return rs


def read_dl_outputs(device_id):
    return output_api.read_dl_outputs(device_id)


def read_4k_outputs(device_id):
    return output_api.read_4k_outputs(device_id)






