"""
Created on 2019年6月28日
@author: 刘益宗
input接口.
返回值统一为 {data:JSON结构体, msg:"", status:0}
"""
import time
import os
from copy import deepcopy

from H_9U import BASE_DIR
from H_9U.util.log import logger
from H_9U.models.result import get_result_model, ResInfo
from H_9U.mao.input_mao import input_api
from H_9U.service import common
from H_9U.models.sysconst import UploadFileType, InputData
from H_9U.service.common import upload_file
from H_9U.service.middlewarefilepath import input_osd_manager


class InputSvc:

    def read_general(self, device_id, input_id):
        """
        读取基本信息
        :param device_id: 设备Id
        :param input_id:  输入Id
        :return: 结果对象
        """
        rs = input_api.read_general(device_id, input_id)
        if rs["status"] != 0:
            logger.error("inputGeneral_read-失败:%i" % rs["status"])
        return rs

    def write_general(self, device_id, input_id, params):
        """
        设置基本信息
        :param params: JSON对象参数
        :param device_id: 设备Id
        :param input_id:  输入Id
        :return: 结果对象
        """
        rs = input_api.write_general(device_id, input_id, params)

        if rs["status"] != 0:
            logger.error("设置General-失败! CODE:%i, params:%s" % (rs["status"], params))
        return rs

    def read_timing(self, device_id, input_id):
        """
        读取timing,从输入详情中获取
        :param device_id: 设备Id
        :param input_id:  输出Id
        :return: 结果对象
        """
        rs = self.read_detail(device_id, input_id)
        if rs["status"] == 0 and rs['data']:
            rs['data'] = rs['data']['timing']
            rs['data']['deviceId'] = device_id
            rs['data']['inputId'] = input_id
        else:
            logger.error("读取timing-失败:%i" % rs["status"])

        return rs

    def write_timing(self, device_id, input_list, params):
        """
        批量设置timimg
        :param params: json 参数
        :return: 结果对象
        """
        data = []
        try:
            # 循环设置
            partial = False
            for input_id in input_list:
                timing = deepcopy(params)
                timing['inputId'] = input_id
                rs = input_api.write_timing(device_id, input_id, timing)
                success = 1
                if rs["status"] != 0:
                    logger.debug("设置Timing-失败! CODE:%i, params:%s" % (rs["status"], input_id))
                    success = 0
                    # 是否部分成功
                    partial = True
                data.append({
                    'inputId': input_id,
                    'success': success
                })

            if partial:  # 部分成功
                rs = get_result_model(ResInfo.Input_Write_Timing_List_Partial_Success)
                rs['data'] = data
            else:  # 全部成功
                rs = get_result_model()
                rs['data'] = data
        except Exception as e:
            logger.error(e)
            rs = get_result_model(ResInfo.Input_Write_Timing_List_Error)
        return rs

    def read_crop_list(self, device_id, input_id):
        """
        读取crop列表
        :param device_id: 设备id
        :param input_id: 输出id
        :return: 结果对象
        """
        rs = input_api.read_crop_list(device_id, input_id)

        if rs["status"] != 0:
            logger.error("读取截取-失败:%i, inputId:%s" % (rs["status"], input_id))
        return rs

    def read_crop_detail(self, device_id, input_id, crop_id):
        """
        读取crop详情
        :param device_id: 设备id
        :param input_id: 输出id
         :param crop_id: crop_id
        :return: 结果对象
        """
        crop_rs = self.read_crop_list(device_id, input_id)
        if crop_rs['status'] == 0 and crop_rs['data']:
            crops = crop_rs['data']['crops']
            if crops:
                crop_rs['data'] = [x for x in crops if x['cropId'] == crop_id][0]
        return crop_rs

    def crop_create(self, params):
        """
        创建结果参数
        :param params: json参数
        :return: 结果对象
        """
        device_id = params.get('deviceId')
        input_id = params.get('inputId')

        rs = input_api.write_crop_create(device_id, input_id, params)

        if rs["status"] == 0:
            logger.debug("创建CROP-成功! CODE:%i, params:%s" % (rs["status"], params))
        else:
            logger.error("创建CROP-失败! CODE:%i, params:%s" % (rs["status"], params))
        return rs

    def crop_update(self, device_id, input_id, crop_id, params):
        """
        修改输入截取
        :param params: 参考接口文档updateCrop
        :param device_id: 设备Id
        :param input_id: input id
        :param crop_id: 截取 id
        :return: 结果对象
        """
        rs = input_api.write_crop_update(device_id, input_id, crop_id, params)

        if rs["status"] == 0:
            logger.debug("修改CROP-成功! CODE:%i, params:%s" % (rs["status"], params))
        else:
            logger.error("修改CROP-失败! CODE:%i, params:%s" % (rs["status"], params))
        return rs

    def crop_delete(self, params):
        """
        删除crop
        :param params: json参数
        :return: 结果对象
        """
        rs = get_result_model()
        device_id = params.get('deviceId')
        input_id = params.get('inputId')
        crop_id_list = params.get('cropId')
        for crop_id in crop_id_list:
            param_info = {
                "inputId": input_id,
                "cropId": crop_id,
                "deviceId": device_id
            }
            rs = input_api.write_crop_delete(device_id, input_id, crop_id, param_info)
            if rs["status"] == 0:
                logger.debug("删除CROP-成功! CODE:%i, params:%s" % (rs["status"], param_info))
            else:
                logger.error("删除CROP-失败! CODE:%i, params:%s" % (rs["status"], param_info))
        return rs

    def read_osd(self, device_id, input_id):
        """
        读取osd信息
        :param device_id: 设备id
        :param input_id: 输入id
        :return: osd信息
        """
        rs = input_api.read_osd(device_id, input_id)

        if rs["status"] == 0:
            # 处理图片路径和url
            if rs['data']['image']['file']['fileName']:
                file_path = file_url = ''
                if rs['data']['image']['file']['fileName'].startswith('./data'):
                    file_path, file_url = input_osd_manager.get_osd_path_info(rs['data']['image']['file']['fileName'])

                elif rs['data']['image']['file']['fileName'].startswith(BASE_DIR):
                    file_path = rs['data']['image']['file']['fileName']
                    file_url = rs['data']['image']['file']['fileName'].replace(os.path.join(BASE_DIR, 'static/'), '')

                rs['data']['image']['file']['fileName'] = os.path.split(file_path)[1] if file_path else ''
                rs['data']['image']['file']['fileUrl'] = file_url
        else:
            logger.error("读取osd-失败:%i" % rs["status"])
        return rs

    def write_osd(self, device_id, input_id, params, has_file=False):
        """
        设置osd信息
        :param device_id: 设备id
        :param input_id: 输入id
        :param params: 参数信息
        :param has_file: 是否包含图片文件
        :return: 结果对象
        """
        osd_type = params['type']
        if osd_type == 1:  # 图片osd
            if has_file:  # 有图片
                # 有图片则上传图片
                up_rs = upload_file(UploadFileType.OSD)
                if not up_rs["bln"]:
                    return get_result_model(ResInfo.Upload_File_Fail)
                path_nm = up_rs["pathNm"]
                params['image']['file']['fileName'] = path_nm
                params['image']['file']['hashSum'] = 0
            else:
                # 没有图片文件则删除file节点
                if 'image' in params and 'file' in params['image']:
                    params['image'].pop('file')
        rs = input_api.write_osd(device_id, input_id, params)
        if rs["status"] != 0:
            logger.error("设置osd-失败:%i" % rs["status"])
        return rs

    def read_detail(self, device_id, input_id):
        """
        读取详细信息
        :param device_id: 设备号
        :param input_id: 输入号
        :return: 见文档readDetail
        """
        rs = input_api.read_detail(device_id, input_id)
        if rs['status'] == 0:
            # 补充截取信息
            crops = []
            if rs['data']['online'] == 1:
                crop_rs = self.read_crop_list(device_id, input_id)
                if crop_rs['status'] == 0 and crop_rs['data']:
                    crops = crop_rs['data']['crops']
            rs['data']['crops'] = crops

        return rs

    def read_detail_by_input_id(self, input_id):
        """
        读取详细信息
        :param device_id: 设备号
        :param input_id: 输入号
        :return: 见文档readDetail
        """
        device_id = 0
        return  input_api.read_detail(device_id, input_id)

    def read_list(self, device_id):
        """
        读取简化字段的输入列表
        :param device_id: 设备Id
        :return: input 基本信息
        """

        rs = input_api.read_list(device_id)
        return rs

    def input_group_list(self, device_id):
        """
        读取组列表
        :param devce_id: 设备Id
        :return: 组内容
        """
        rs = input_api.read_group_list(device_id)
        if rs['status'] == 0 and rs['data']:
            for group in rs['data']['inputGroupList']:
                detail_rs = input_api.read_group_node_items(device_id, group['inputGroupId'])
                inputs = []
                if detail_rs['status'] == 0 and detail_rs['data']:
                    inputs = detail_rs['data']['inputs']
                group['inputs'] = inputs

        return rs

    def input_group_detail(self, device_id, group_id):
        return input_api.read_group_node_items(device_id, group_id)

    def input_group_create(self, device_id, data):
        """
        创建输入分组
        :param device_id: 设备Id
        :param data: 组内容
        :return: 结果对象
        """
        rs = input_api.read_group_list(device_id)
        if rs['status'] == 0 and rs['data']:
            if len(rs['data']['inputGroupList']) > InputData.Group_Max_Count:
                return get_result_model(ResInfo.Input_Group_Max_Count_Err)
        return input_api.write_group_create(device_id, data)

    def input_group_delete(self, device_id, group_id, data):
        """
        删除输入分组
        :param device_id: 设备Id
        :param group_id: 组Id
        :param data: 分组数据
        :return: 结果对象
        """
        return input_api.write_group_delete(device_id, group_id, data)

    def input_group_list_update(self, device_id, data):
        """
        输入分组节点枚举
        :param device_id: 设备Id
        :param data: 组列表信息
        :return: 结果对象，见文档
        """
        return input_api.write_group_update(device_id, data)

    def input_group_node_update(self, device_id, group_id, data):
        """
        输入源分组更输入源全部数据
        :param device_id: 设备id
        :param group_id: 组Id
        :param data: 输入源列表
        :return: 结果对象
        """
        return input_api.write_group_node_items_update(device_id, group_id, data)

    def input_group_node_add(self, device_id, group_id, input_id, data):
        """
        输入源组添加元素
        :param device_id: 设备Id
        :param group_id:  组Id
        :param input_id:  输入源Id
        :param data: 内容
        :return: 结果对象
        """
        return input_api.write_group_node_item_add(device_id, group_id, input_id, data)

    def input_group_node_delete(self, device_id, group_id, input_id, data):
        """
        group node 删除 item
        :param device_id:  设备Id
        :param group_id:  组Id
        :param input_id:  输入源Id
        :param data:  数据
        :return: 结果对象
        """
        return input_api.write_group_node_item_delete(device_id, group_id, input_id, data)

    def input_edid_read(self, device_id, input_id):
        """
        读取edid数据
        :param device_id: 设备Id
        :param input_id:  输入Id
        :return:  timing数据简化版
        """
        rs = self.read_timing(device_id, input_id)
        if rs['status'] == 0:
            edid = {
                "inputId": input_id,
                "deviceId": device_id,
                "width": rs['data']['horizontal']['addrTime'],
                "height": rs['data']['vertical']['addrTime'],
                "refreshRate": rs['data']['refreshRate']
            }
            rs['data'] = edid
        return rs

    def input_edid_download(self, device_id, input_id):
        """
        读取edid
        :param device_id: 设备Id
        :param input_id: inputId
        :return: 结果数据
        """
        return input_api.read_edid(device_id, input_id)

    def input_get_connect_capacity(self, device_id, input_id):
        """
        读取接口容量信息
        :param device_id: 设备Id
        :param input_id:  接口Id
        :return: connectCapacity信息
        """
        rs = self.read_detail(device_id, input_id)
        if rs['status'] == 0 and rs['data']:
            rs['data'] = {'connectCapacity': rs['data']['general']['connectCapacity']}
        return rs

    def input_edid_import(self, device_id, input_ids):
        """
        批量导入edid
        :param device_id: 设备Id
        :param input_ids: 输入Id列表
        :return: 结果对象
        """
        up_rs = common.upload_file(UploadFileType.Input_EDID, extract=False)

        # 文件上传成功后还要调用中间件进行设备升级
        if up_rs["bln"] is not True:
            logger.error("导入edid-文件上传-失败!")
            return get_result_model(ResInfo.Upload_File_Fail)
        input_id = 255 if len(input_ids) > 1 else int(input_ids[0])
        path_nm = up_rs["pathNm"]
        lst_data = []
        for x in input_ids:
            lst_data.append({
                "inputId": int(x),
                "deviceId": device_id,
                "file": {
                    "fileName": path_nm,
                    "fileData": '',
                    "hashSum": 0
                }
            })
        rs = input_api.write_edid(device_id,input_id, lst_data)
        if rs['status'] == 0:
            for x in input_ids:
                data = {
                    'inputId': int(x),
                    'deviceId': device_id,
                    'templateOperationType': 1
                }
                input_api.write_timing(device_id, int(x), data)
        return rs

    def input_edid_export(self, device_id, input_id):
        """
        输入edid导出
        :param device_id: 设备id
        :param input_id: 输入id
        :return: edid文件
        """
        return input_api.read_edid(device_id, input_id)

    def input_regulate(self, device_id, input_id, data):
        """
        VGA 自动校准
        :param device_id: 设备Id
        :param input_id: 输入Id
        :param data: 校准信息
        :return: 结果对象
        """
        return input_api.write_input_regulate(device_id, input_id, data)

    def reverse_control_info_write(self, device_id, input_id, data):
        """
        输入源反控设置
        :param device_id:  设备Id
        :param input_id:  输入源Id
        :param data:  数据
        :return: 结果对象
        """
        return input_api.reverse_control_info_write(device_id, input_id, data)

    def mosaic_montage_read(self, device_id, input_id):
        """
        读取光纤模板列表
        :param device_id:  设备Id
        :return: 结果对象
        """
        return input_api.mosaic_montage_read(device_id, input_id)

    def mosaic_montage_write(self, device_id, data):
        """
        设置光纤模板信息
        :param device_id:  设备Id
        :return: 结果对象
        """
        return input_api.mosaic_montage_write(device_id, data)

    def mosaic_rename(self, device_id, data):
        """
        光纤模板源重命名
        :param device_id:  设备Id
        return: 结果对象
        """
        return input_api.mosaic_rename(device_id, data)


input_svc = InputSvc()
