"""
Created on 2021年11月01日
@author: 程声清
ipc服务类
"""
from H_9U.mao.ipc_mao import ipc_api
from H_9U.middle import open_ipc_middle, open_input_middle, open_device_middle
from H_9U.models.cachekey import CacheKey, CacheDuration
from H_9U.models.sysconst import CardCategory, IPCType, IPCSlotId
import time
from H_9U.models.sysconst import IPCDefaultValue
from H_9U.models.result import get_result_model, ResInfo
from copy import deepcopy

from H_9U.util.assembly_data import cutting_ip_data, cutting_data
from H_9U.util.cache import cacher
from H_9U.util.common import check_ip


def ipc_read_source_list(device_id, data):
    """
    读取信号源列表，data为分页数据
    :param device_id: 设备Id
    :param data: 分页参数
    :return 结果数据
    """
    return open_ipc_middle.ipc_read_source_list_middle(device_id, data)


def ipc_source_list_by_slot_id(device_id, slot_id):
    """
    通过soltId查询源
    :param device_id: 设备Id
    :param slot_id: slot_id
    :return 结果数据
    """
    rs = get_result_model()
    index = 1
    size = 200
    ipc_list = []
    while index > 0:
        data = {
            "seqPageIndex": index - 1,
            "seqPageSize": size,
            "deviceId": device_id,
            "slotId": slot_id
        }
        rs = open_ipc_middle.ipc_read_source_list_middle(device_id, data)
        if rs['status'] != 0:
            break
        if rs['data']['sourceList']:
            ipc_list.extend(rs['data']['sourceList'])
        if rs['data']['nums'] > index + size:
            index += size
        else:
            index = 0
    rs['data']['sourceList'] = ipc_list
    return rs


def ipc_read_camera_list(device_id, slot_id):
    """
    ipc摄像头列表
    :param device_id: 设备id
    :param slot_id: slot id
    :return: 摄像头列表
    """
    return open_ipc_middle.ipc_read_camera_list_middle(device_id, slot_id)


def ipc_channel_detail_list(device_id, source_id):
    """
      读取信号源通道列表
      :param device_id: 设备Id
      :param source_id: 源Id
      :return 结果数据
      """
    key = CacheKey.ipc_source_channel_detail_list(device_id, source_id)
    rs = cacher.IPCCache.get(key, None)
    if not rs:
        rs = open_ipc_middle.ipc_read_ipc_channel_list_middle(device_id, source_id)
        if rs['status'] == 0 and rs['data']:
            channel_list = []
            for channel in rs['data']['channelList']:
                channel_id = channel['channelId']
                if channel_id != 65535:
                    channel_rs = open_ipc_middle.ipc_read_channel_detail_middle(device_id, source_id, channel_id)
                    if channel_rs['status'] == 0 and channel_rs['data']:
                        channel_list.append(channel_rs['data'])
            rs['data']['channelList'] = channel_list
            cacher.IPCCache.set(key, rs, CacheDuration.Default)
    return rs


def ipc_channel_list_data(device_id, source_id, slot_id, montage_id):
    """
    获取信号源通道筛选后的数据
    :param device_id: device_id
    :param source_id: source_id
    :param slot_id: slot_id
    :param montage_id: montage_id
    :return 结果数据
    """
    ms = ipc_montage_list(device_id)
    des = get_result_model()
    if ms['status'] == 0:
        stream_ids = _get_stream_id(ms, slot_id, montage_id)
        rs = ipc_channel_detail_list(device_id, source_id)
        if ms['status'] == 0:
            des = _get_channel_data(rs, stream_ids)
            if des['status'] == 0:
                channel_list = des['data']['channelList']
                for index_c in range(len(channel_list) - 1, -1, -1):
                    channel = channel_list[index_c]
                    stream_list = channel['streamList']
                    if len(stream_list) == 0:
                        channel_list.remove(channel_list[index_c])
    return des


def ipc_read_channel_detail(device_id, source_id, channel_id):
    """
    读取ipc通道详情
    :param device_id: 设备id
    :param source_id: source id
    :param channel_id: channel_id
    :return: ipc列表
    """
    return open_ipc_middle.ipc_read_channel_detail_middle(device_id, source_id, channel_id)


def ipc_source_create(device_id, data):
    """
    创建ipc信号源
    :param device_id: 设备id
    :param slot_id: slot_id
    :param data: ipc信号源信息
    :return: 信号源id
    """
    rs = _assembly_data(data)
    if rs['status'] != 0:
        return rs
    channel_list = rs['data']['channelList']
    end_ips = rs['data']['endIp']
    ip_list = rs['data']['ipList']
    source_ip = rs['data']['sourceIp']
    source_name = rs['data']['sourceName']
    count_number = 1
    ips = cutting_ip_data(source_ip, end_ips, ip_list)
    datas = []
    if len(ips) == 0:
        return get_result_model(ResInfo.Params_Error)
    elif len(ips) == 1:
        if source_name == "":
            data['sourceName'] = ips[0]
        data['ip'] = ips[0]
        for channel in channel_list:
            stream_list = channel['streamList']
            for stream in stream_list:
                source_type = stream['protocol']['type']
                # onvif 自动扫描时 需要我们补IP, 所以此处需要特殊处理
                if source_type == 2:
                    stream['protocol']['onvif']['onvifIp'] = ips[0]
        datas.append(data)
    else:
        for ip in ips:
            channel_list_copy = deepcopy(channel_list)
            if source_name == "":
                data['sourceName'] = ip + "_" + str(count_number)
            else:
                data['sourceName'] = source_name + "_" + str(count_number)
            for channel in channel_list_copy:
                channel['createTime'] = str(time.time())
                stream_list = channel['streamList']
                for stream in stream_list:
                    source_type = stream['protocol']['type']
                    # onvif
                    if source_type == 2:
                        stream['protocol']['onvif']['onvifIp'] = ip
                    # rtsp manufacturer
                    elif source_type == 1:
                        if stream['protocol']['rtsp']['rtspPort'] == "" \
                                and stream['protocol']['rtsp']['manufacturer'] == 255:
                            stream['protocol']['rtsp']['rtspUrl'] = "rtsp://" + ip + "/"
                        else:
                            url_data = stream['protocol']['rtsp']['rtspUrl'].split('//')[1].split(':')[1]
                            # rtsp://192.168.10:554/cam/realmonitor?channel=1&subtype=0
                            stream['protocol']['rtsp']['rtspUrl'] = "rtsp://" + ip + ":" + str(url_data)
                        stream['protocol']['rtsp']['rtspIp'] = ip
                    # gb28181
                    else:
                        stream['protocol']['gb28181']['platformIp'] = ip
            data['channelList'] = channel_list_copy
            data['ip'] = ip
            datas.append(deepcopy(data))
            count_number += 1
    if data['ipcType'] == IPCType.NVR:
        rs = _nvr_batch_create(device_id, datas)
    else:
        rs = _ipc_batch_create(device_id, datas)
    return rs


def _nvr_batch_create(device_id, datas):
    """
	NVR模式添加源
	:param device_id: 设备id
	:param datas: 数据
	:return: 结果对象
	"""
    rs = get_result_model()
    for data in datas:
        index = IPCDefaultValue.Batch_Create_Default_Count
        # 复制一份channelList 数据
        channels = deepcopy(data['channelList'])
        # 取通道数据
        data['channelList'] = channels[:index]
        # 创建数据
        rs = ipc_api.write_ipc_source_create(device_id, IPCSlotId.Invalid, data)
        if rs['status'] == 0 and rs['data']:
            # 拿到创建之后返回的sourceId
            data['sourceId'] = rs['data']['sourceId']
            while index < len(channels):
                data['channelList'] = channels[index:index + IPCDefaultValue.Batch_Create_Default_Count]
                append_rs = ipc_api.write_ipc_source_append(device_id, data)
                if append_rs['status'] != 0:
                    return get_result_model(
                        ResInfo.Input_IPC_Slot_Source_Append_Err)
                index += IPCDefaultValue.Batch_Create_Default_Count
    return rs


def _ipc_batch_create(device_id, datas):
    """
	IPC模式添加源
	:param device_id: 设备id
	:param datas: 数据
	:return: 结果对象
	"""
    index = 0
    rs = get_result_model()
    while index < len(datas):
        # 批量创建源
        data = datas[index:index + IPCDefaultValue.Batch_Create_Default_Count]
        rs = ipc_api.write_ipc_source_create(device_id, IPCSlotId.Invalid, data)
        if rs['status'] != 0:
            return rs
        index += IPCDefaultValue.Batch_Create_Default_Count
    return rs


def ipc_source_delete(device_id, data):
    """
    删除输入信号源
    :param device_id: 设备id
    :param data: data
    :return: 结果对象
    """
    return open_ipc_middle.ipc_source_delete_middle(device_id, data)


def ipc_source_rename(device_id, source_id, name):
    """
    修改ipc general 信息
    :param device_id: slot id
    :param source_id: source
    :param name: name
    :return: 结果对象
     """

    data = {'deviceId': device_id, 'sourceId': source_id, 'sourceName': name}
    return open_ipc_middle.ipc_source_update_middle(device_id, source_id, data)


def ipc_channel_edit(device_id, params):
    """
    修改通道数据
    :param device_id: 设备Id
    :param params: params
    :return 结果数据
    """
    return open_ipc_middle.ipc_write_channel_edit_middle(device_id, params)


def ipc_channel_add(device_id, slot_id, source_id, params):
    """
    add通道数据
    :param device_id: 设备Id
    :param slot_id: slot_id
    :param source_id: source_id
    :param params: params
    :return 结果数据
    """
    return _ipc_channel_append(device_id, slot_id, source_id, params)


def ipc_channel_delete(device_id, params):
    """
    删除通道数据
    :param device_id: 设备Id
    :param params: params
    :return 结果数据
    """
    return open_ipc_middle.ipc_channel_delete_middle(device_id, params)


def ipc_channel_detail(device_id, source_id, channel_id):
    """
    读取信号源通道详情
    :param device_id: 设备Id
    :param source_id: 源Id
    :param channel_id: 通道Id
    :return 结果数据
    """
    return open_ipc_middle.read_ipc_channel_detail_middle(device_id, source_id, channel_id)


def ipc_slot_count_list(device_id):
    """
    ipc slot 数量列表
    :param device_id: device_id
    :return: 结果对象
    """
    slot_list = _get_ipc_card_list(device_id)
    rs = get_result_model()
    rs['data'] = {
        'deviceId': device_id,
        'ipcSlotList': []
    }
    if slot_list:
        for slot_id in slot_list:
            mode_rs = open_device_middle.ipc_read_mode_middle(device_id, slot_id)
            ipc_rs = ipc_slot_source_count(device_id, slot_id)
            if ipc_rs['status'] == 0 and mode_rs['status'] == 0:
                rs['data']['ipcSlotList'].append({
                    'slotId': slot_id,
                    'sourceCount': ipc_rs['data']['sourceCount'],
                    'videoMode': mode_rs['data']['videoMode']})
    return rs


def _get_ipc_card_list(device_id):
    """
    ipc 卡 数量列表
    :param device_id: device_id
    :return: 结果对象
    """
    rs = _ipc_card_list(device_id)
    ipc_list = []
    if rs['status'] == 0:
        ipc_list = rs['data']
    return ipc_list


def _ipc_card_list(device_id):
    """
    ipc 卡 数量列表
    :param device_id: device_id
    :return: 结果对象
    """
    rs = open_device_middle.device_detail_read_middle(device_id)
    if rs['status'] == 0:
        if rs['data']:
            ipc_list = [x['slotId'] for x in rs['data']['slotList'] if x['cardCategory'] == CardCategory.IPCCard]
            rs['data'] = ipc_list
        else:
            rs['data'] = []

    return rs


def ipc_source_slot_count(device_id):
    """
    ipc slot 数量
    :param device_id: slot id
    :return: 结果对象
    """
    rs = open_input_middle.ipc_card_list_middle(device_id)
    ipc_list = []
    if rs['status'] == 0:
        ipc_list = rs['data']
    rs = get_result_model()
    rs['data'] = {
        'deviceId': device_id,
        'ipcSlotList': []
    }
    if ipc_list:
        for slot_id in ipc_list:
            ipc_rs = open_ipc_middle.ipc_slot_source_count_middle(device_id, slot_id)
            if ipc_rs['status'] == 0:
                rs['data']['ipcSlotList'].append({'slotId': slot_id, 'sourceCount': ipc_rs['data']['count']})
    return rs


def ipc_slot_source_count(device_id, slot_id):
    """
    ipc源数量
    :param device_id: 设备id
    :param slot_id: slot_id
    :return: 结果对象
    """
    source_count_rs = open_ipc_middle.ipc_read_slot_source_count_middle(device_id, slot_id)
    if source_count_rs['status'] == 0:
        source_channels_count = 0
        source_list = source_count_rs['data']['sourceList']
        for source in source_list:
            source_channels_count += source['channelCount']
        # 剩余的通道数量
        source_count_rs['data']['slotMostChannelCount'] = IPCDefaultValue.Slot_Max_Channel_Count - source_channels_count
    return source_count_rs


def ipc_search_source_list(device_id, source_id, name):
    """
    ipc 源搜索结果列表
    :param device_id: 设备Id
    :param source_id: source_id
    :param name: name
    :return:
    """
    data = {'deviceId': device_id, 'name': name}
    rs = open_ipc_middle.ipc_search_source_list_middle(device_id, data)
    repeat = 0
    if rs['status'] == 0:
        for x in rs['data']['sourceList']:
            if x['sourceName'] == name and x['sourceId'] != source_id:
                repeat = 1
                break
    else:
        repeat = 1

    rs = get_result_model()
    rs['data'] = {
        'deviceId': device_id,
        'sourceId': source_id,
        'name': name,
        'repeat': repeat
    }
    return rs


def ipc_group_list(device_id):
    """
    ipc分组列表
    :param device_id: 设备Id
    :return: 结果数据
    """
    return open_ipc_middle.ipc_group_list_middle(device_id)


def ipc_group_node_list_without_page(device_id, group_id):
    """
    获取ipc分组全部节点数据，不分页
    :param device_id: 设备Id
    :param group_id: group_id
    :return: 结果数据
    """
    return open_ipc_middle.ipc_group_node_list_without_page_middle(device_id, group_id)


def ipc_group_create(device_id, params):
    """
    获取ipc分组创建
    :param device_id: 设备Id
    :param params: params
    :return: 结果数据
    """
    if len(params["sourceList"]) > IPCDefaultValue.Group_Max_Source_Count:
        return get_result_model(ResInfo.Input_IPC_Slot_Source_Max_Count_Err)
    rs = open_ipc_middle.ipc_group_list_middle(device_id)
    if rs['status'] == 0 and rs['data']:
        if len(rs['data']['groupList']) >= IPCDefaultValue.Group_Max_Count:
            return get_result_model(ResInfo.Input_Group_Max_Count_Err)
    return open_ipc_middle.ipc_group_create_middle(device_id, params)


def ipc_group_delete(device_id, params):
    """
    获取ipc分组删除
    :param device_id: 设备Id
    :param params: params
    :return: 结果数据
    """
    return open_ipc_middle.ipc_group_delete_middle(device_id, params)


def ipc_group_update(device_id, group_id, params):
    """
    更新ipc分组
    :param device_id: 设备Id
    :param group_id: group_id
    :param params: params
    :return: 结果数据
    """
    rs = open_ipc_middle.ipc_group_clear_middle(device_id, group_id)
    if rs['status'] == 0:
        rs = open_ipc_middle.ipc_group_update_middle(device_id, params)
    return rs


def ipc_group_node_delete(device_id, params):
    """
    ipc分组删除节点
    :param device_id: 设备Id
    :param group_id: group_id
    :param params: params
    :return: 结果数据
    """
    return open_ipc_middle.ipc_group_node_delete_middle(device_id, params)


def ipc_montage_list(device_id):
    """
    ipc 拼接源列表
    :param device_id: device_id
    :return:
    """
    card_list = []
    rs = ipc_slot_count_list(device_id)
    if rs['status'] == 0:
        card_list = [x for x in rs['data']
                     ['ipcSlotList'] if x['videoMode'] == 0]
    montage_list = []
    for x in card_list:
        rs = open_ipc_middle.ipc_montage_list_middle(device_id, x['slotId'])
        if rs['status'] == 0 and rs['data']:
            montage_list.append(rs['data'])
    rs = get_result_model()
    rs['data'] = {"deviceId": device_id, "slotList": montage_list}
    return rs


def ipc_montage_list_get(device_id, slot_id):
    """
    根据slot_id 获取 ipc 拼接源列表
    :param device_id: device_id
    :param slot_id: slot_id
    :return:
    """
    return open_ipc_middle.ipc_montage_list_middle(device_id, slot_id)


def ipc_montage_template_create(device_id, params):
    """
    ipc 拼接源创建模板
    :param device_id: device_id
    :param params: params
    :return:
    """
    params['createTime'] = str(time.time())
    return open_ipc_middle.ipc_montage_template_create_middle(device_id, params)


def ipc_montage_template_update(device_id, params):
    """
    ipc 拼接源模板修改
    :param device_id: device_id
    :param params: params
    :return:
    """
    rs = open_ipc_middle.ipc_montage_list_middle(device_id, params.get('slot_id'))
    if rs['status'] == 0 and rs['data'] and rs['data']['montageList']:
        montages = [x for x in rs['data']['montageList'] if x['montageId'] == params.get('montage_id')]
        if not montages:
            return get_result_model(ResInfo.Middle_Data_Err)
        templates = [x for x in montages[0]['templateList'] if x['templateId'] == params.get('template_id')]
        if not templates:
            return get_result_model(ResInfo.Middle_Data_Err)
        params['createTime'] = templates[0]['createTime'] if 'createTime' in templates[0] else str(time.time())
    else:
        return get_result_model(ResInfo.Middle_Data_Err)
    return open_ipc_middle.ipc_montage_template_update_middle(device_id, params)


def ipc_montage_template_delete(device_id, params):
    """
    ipc 拼接源模板删除
    :param device_id: device_id
    :param params: params
    :return:
    """
    return open_ipc_middle.ipc_montage_template_delete_middle(device_id, params)


def ipc_montage_template_apply(device_id, params):
    """
    ipc 拼接源应用模板
    :param device_id: device_id
    :param params: params
    :return:
    """
    return open_ipc_middle.ipc_montage_template_apply_middle(device_id, params)


def ipc_montage_rename(device_id, params):
    """
    ipc 拼接源模板重命名
    :param device_id: device_id
    :param params: params
    :return:
    """
    return open_ipc_middle.ipc_montage_rename_middle(device_id, params)


def _assembly_data(data):
    """
    获取参数，
    :param data: 数据
    :return: 结果对象
    """
    rs = get_result_model()
    source_name = data.get('sourceName')
    ip_list = data.get('ipList')
    data['createTime'] = str(time.time())
    channel_list = data.get('channelList')
    end_ips = data.get('endIp')
    source_ip = data.get('ip')
    if source_ip != "":
        ip = check_ip(source_ip)
        if not ip:
            return get_result_model(
                ResInfo.Input_IPC_Slot_Source_Add_ip_Err)
    if end_ips != "":
        end_ip_result = check_ip(end_ips)
        if not end_ip_result:
            return get_result_model(
                ResInfo.Input_IPC_Slot_Source_Add_ip_Err)
    if len(ip_list) > 0:
        for ip_info in ip_list:
            ip_result = check_ip(ip_info)
            if not ip_result:
                return get_result_model(
                    ResInfo.Input_IPC_Slot_Source_Add_ip_Err)
    rs['data']['channelList'] = channel_list
    rs['data']['endIp'] = end_ips
    rs['data']['ipList'] = ip_list
    rs['data']['sourceIp'] = source_ip
    rs['data']['sourceName'] = source_name
    return rs


def _ipc_source_create_info(device_id, slot_id, data):
    """
    单卡1024个channel，单个源512个channel, 每次最多创建50个，剩余部分追加
    创建单个ipc源
    :param device_id: 设备id
    :param slot_id: slotId
    :param data: 数据
    :return: 结果对象
    """
    handle_rs = _ipc_source_append_handle(device_id, slot_id, None)
    if handle_rs['status'] != 0:
        return handle_rs
    index = handle_rs['data']['index']
    # 复制一份channelList 数据
    channels = deepcopy(data['channelList'])
    # 保证最多取50个通道数据
    data['channelList'] = channels[:index]
    # 创建 50个数据
    rs = open_ipc_middle.write_ipc_source_create_middle(device_id, slot_id, data)
    if rs['status'] == 0 and rs['data']:
        if len(channels) > index:
            source_id = rs['data']['sourceId']
            # 拿到创建之后返回的sourceId
            data['sourceId'] = source_id
            data['channelList'] = channels[index:]
            append_rs = _ipc_channel_append(device_id, slot_id, None, data)
            if append_rs['status'] != 0:
                return get_result_model(
                    ResInfo.Input_IPC_Slot_Source_Append_Err)
    return rs


def _ipc_channel_append(device_id, slot_id, source_id, data):
    """
    单卡1024个channel 每次最多创建50个，剩余部分追加
    ipc源追加通道
    :param device_id: 设备id
    :param slot_id: slotId
    :param data: 数据
    :return: 结果对象
    """
    if source_id is None:
        source_id = data['sourceId']
    channels = deepcopy(data['channelList'])
    index = 0
    while index < len(channels):
        handle_rs = _ipc_source_append_handle(
            device_id, slot_id, source_id)
        if handle_rs['status'] != 0:
            return handle_rs
        count = handle_rs['data']['index']
        data['channelList'] = channels[index:index + count]
        rs = open_ipc_middle.ipc_source_append_middle(device_id, data)
        if rs['status'] == 0:
            index += count
        else:
            return rs
    return get_result_model()


def _ipc_source_append_handle(device_id, slot_id, source_id):
    """
    ipc源追加数据处理
    :param device_id: 设备id
    :param slot_id: slot_id
    :param source_id: source_id
    :return: 结果对象
    """
    rs = _ipc_slot_channel_count(device_id, slot_id, source_id)
    if rs['status'] != 0:
        return rs
    # slotSourceCount  该卡已经存在的源数量
    # slotChannelCount 还能创建的通道数量
    # 该卡只支持 512个源  最多支持1024个通道
    if rs['data']['slotSourceCount'] >= IPCDefaultValue.Slot_Max_Source_Count \
            or rs['data']['slotChannelCount'] >= IPCDefaultValue.Slot_Max_Channel_Count \
            or rs['data']['source_exist_channelCount'] >= IPCDefaultValue.Source_Max_Channel_Count:
        return get_result_model(
            ResInfo.Input_IPC_Slot_Source_Max_Count_Err)
    # 该源下还能创建的通道数量
    source_channel_count = IPCDefaultValue.Source_Max_Channel_Count - rs['data']['source_exist_channelCount']

    # 剩余能创建的通道个数
    channel_count = IPCDefaultValue.Slot_Max_Channel_Count - rs['data']['slotChannelCount']

    # 该源下能创建的与该卡能创建的取最小值
    number = min(source_channel_count, channel_count)
    # 从 50 与 能创建的通道数的 最小值
    index = min(
        number,
        IPCDefaultValue.Source_Channel_Create_Default_Count)
    rs['data']['index'] = index
    return rs


def _ipc_slot_channel_count(device_id, slot_id, source_id):
    """
    ipc通道数据处理
    :param device_id: 设备id
    :param slot_id: slot_id
    :param source_id: source_id
    :return: 结果对象
    """
    source_count_rs = open_ipc_middle.ipc_read_slot_source_count_middle(device_id, slot_id)
    data = {}
    if source_count_rs['status'] == 0:
        source_channels_count = 0
        source_list = source_count_rs['data']['sourceList']
        for source in source_list:
            source_channels_count += source['channelCount']
        # 剩余的通道数量
        source_count_rs['data']['slotMostChannelCount'] = IPCDefaultValue.Slot_Max_Channel_Count - source_channels_count
        exist_number = 0
        exist_source_channels_count = 0
        # 卡下已经存在的源的数量 （单卡 512个源  单源1024个通道）
        exist_source_count = source_count_rs['data']['sourceCount']
        source_list = source_count_rs['data']['sourceList']
        for source in source_list:
            exist_number += source['channelCount']
            if source['sourceId'] == source_id:
                exist_source_channels_count += source['channelCount']
        data = {
            'slotSourceCount': exist_source_count,  # 卡下已经存在的源数量
            'slotChannelCount': exist_number,  # 卡下所有已经存在的通道的数量
            'slotMostChannelCount': min(IPCDefaultValue.Slot_Max_Channel_Count - exist_number,
                                        IPCDefaultValue.Source_Max_Channel_Count - exist_source_channels_count),
            'source_exist_channelCount': exist_source_channels_count  # 指定源下存在的通道数量
        }
    rs = get_result_model()
    rs['data'] = data
    return rs


def _get_stream_id(ms, slot_id, montage_id):
    """
    组装流id数据
    :param ms: ms数据
    :param slot_id: slot_id
    :param montage_id: montage_id
    :return: 结果对象
    """
    stream_ids = []
    ms1 = deepcopy(ms)
    if ms1['status'] == 0 and ms1['data']['slotList']:
        for slotInfos in ms1['data']['slotList']:
            s_id = slotInfos['slotId']
            if s_id == slot_id:
                for slotInfo in slotInfos['montageList']:
                    m_id = slotInfo['montageId']
                    template_list = slotInfo['templateList']
                    for montage in template_list:
                        if montage['streamList']:
                            for stream in montage['streamList']:
                                if montage_id != m_id:
                                    stream_ids.append(stream['streamId'])
    stream_ids = set(stream_ids)
    return stream_ids


def _get_channel_data(rs, stream_ids):
    """
    组装流id数据
    :param rs: ms数据
    :param stream_ids: stream_ids
    :return: 结果对象
    """
    des = deepcopy(rs)
    channel_list = des['data']['channelList']
    if channel_list:
        for index_c in range(len(channel_list) - 1, -1, -1):
            channel = channel_list[index_c]
            stream_list = channel['streamList']
            if stream_list:
                for index in range(len(stream_list) - 1, -1, -1):
                    stream = stream_list[index]
                    stream_id = stream['streamId']
                    if stream_id in stream_ids:
                        stream_list.remove(stream_list[index])
    return des
