"""
Created on 2019年6月28日
@author: 刘益宗
screen 服务
"""

import time

from H_9U.models.result import get_result_model, ResInfo
from H_9U.util.log import logger
from H_9U.models.sysconst import ModelId, outputModeType
from H_9U.models.sysconst import UploadFileType
from H_9U.mao.screen_mao import screen_api
from H_9U.service.common import upload_file
from H_9U.service.middlewarefilepath import screen_osd_manager


class ScreenSvc:

    def read_list(self, device_id):
        """
        读取列表
        :param device_id: 设备id
        :return: 统一结果
        """
        return screen_api.read_list(device_id)

    def read_detail(self, device_id, screen_id):
        """
        读取屏幕详细信息
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :return: json对象数据
        """
        st = time.time()  # 返回当前时间的时间戳（1970纪元后经过的浮点秒数）
        rs = screen_api.read_detail(device_id, screen_id)

        if rs["status"] != 0 or not rs['data']:
            logger.error("screenFullDetail查询失败! screenId:%i, status:%i" % (screen_id, rs["status"]))
            et = time.time()  # 返回当前时间的时间戳（1970纪元后经过的浮点秒数）
            logger.debug("读取屏幕详情耗时(秒)：%s" % str(et - st))
            return rs
        for output in rs['data']['outputMode']['screenInterfaces']:
            if 'dotpitch' not in output:
                output['dotpitch'] = 2
        if 'Osd' not in rs['data']:
            rs['data']['Osd'] = {'enable': 0}
        if 'OsdImage' not in rs['data']:
            rs['data']['OsdImage'] = {'enable': 0}
        else:
            file_path, file_url = screen_osd_manager.get_osd_path_info(rs['data']['OsdImage']['image']['file']['fileName'])
            rs['data']['OsdImage']['image']['file']['fileName'] = file_url

        # 大屏亮度，默认值-1不设置
        rs['data']['screenBrightness'] = -1
        # 二合一卡接口则读取屏幕亮度值
        if len([x for x in rs['data']['outputMode']['screenInterfaces']
                if x['modelId'] in ModelId.Two_In_One_Card_List]) > 0:
            rs['data']['screenBrightness'] = 0
            # 亮度数据
            brightness_rs = screen_api.read_screen_brightness(device_id, screen_id)
            if brightness_rs['status'] == 0:
                rs['data']['screenBrightness'] = brightness_rs['data']['brightness']
        et = time.time()  # 返回当前时间的时间戳（1970纪元后经过的浮点秒数）
        logger.debug("读取屏幕详情耗时(秒)：%s" % str(et - st))
        return rs

    def create(self, device_id, params):
        """
        创建单个Screen. 参数不含screenId
        :param device_id: 设备Id
        :param params: 创建屏幕参数
        :return: 屏幕Id
        """
        rs = screen_api.write_create(device_id, params)
        if rs['status'] != 0:
            logger.error("创建屏幕-失败:%i" % rs["status"])
        return rs

    def delete(self, device_id, screen_id):
        """
        删除屏幕
        :param device_id: 设备Id
        :param screen_id:  屏幕Id
        :return: 结果数据
        """
        data = {'deviceId': device_id, 'screenId': screen_id}
        rs = screen_api.write_delete(device_id, screen_id, data)
        if rs["status"] == 0:
            logger.debug("删除screenDelete_write-成功,deviceId:%i, screenId:%i" % (device_id, screen_id))
        else:
            logger.error(
                "删除screenDelete_write-失败,deviceId:%i, screenId:%i, status:%i" % (device_id, screen_id, rs["status"]))
        return rs

    def general_write(self, device_id, screen_id, params):
        """
        设置屏幕基本信息
        :param device_id: 设备Id
        :param screen_id:  屏幕Id
        :param params:  json数据
        :return:  结果对象
        """
        rs = screen_api.write_general(device_id, screen_id, params)
        return rs

    def ftb(self, device_id, screen_id, params):
        """
        屏幕设置ftb(ftb与freeze互斥)
        :param device_id:  设备Id
        :param screen_id:  屏幕Id
        :param params: ftb数据
        :return: 结果对象
        """
        type_info = params.get('type')
        # format_client_data(params)
        freeze_rs = True
        if type_info == 0:  # 黑屏关闭freeze
            rs = screen_api.write_freeze(device_id, screen_id, {
                    "deviceId": device_id,
                    "screenId": screen_id,
                    "enable": 0
                }
            )
            if rs['status'] != 0:
                freeze_rs = False
        if freeze_rs:
            rs = screen_api.write_ftb(device_id, screen_id, params)
        return rs

    def delete_batch(self, params):
        """
        批量删除屏幕
        :param params: 删除参数。
        model删除模式0-不指定模式，1-指定模式
        备注：
        model=1时screenIdList参数不可为空。
        model=0时screenIdList参数可为空，即：清除该备上所有屏幕。
        :return: 结果列表
        """
        device_id = params.get('deviceId')
        model = params.get('model')
        del_rs = []

        # 按照传入Id删除
        if model == 1:
            id_list = params.get('screenIdList')
            if not id_list:
                return get_result_model(ResInfo.Params_Error)

            for s_id in id_list:
                d_rs = self.delete(device_id, s_id)
                del_rs.append({
                    'deviceId': device_id,
                    'screenId': s_id,
                    'status': d_rs['status']
                })

        # 全部删除
        elif model == 0:
            rs = self.read_list(device_id)
            if rs['status'] != 0:
                return rs
            if rs['data'] and rs['data']['screens']:
                for screen in rs['data']['screens']:
                    d_rs = self.delete(device_id, screen['screenId'])
                    del_rs.append({
                        'deviceId': device_id,
                        'screenId': screen['screenId'],
                        'status': d_rs['status']
                    })
        rs = get_result_model()
        rs['data'] = del_rs
        return rs

    def write_freeze(self, device_id, screen_id, params):
        """
        屏幕设置freeze，(ftb与freeze互斥)
        :param params: 屏幕冻结数据
        :param device_id:  设备Id
        :param screen_id:  屏幕Id
        :return: 结果对象
        """
        enable = params.get('enable')
        ftb_rs = True
        if enable == 1:  # 启用freeze，关闭ftb
            rs = screen_api.write_ftb(device_id, screen_id, {
                    "deviceId": device_id,
                    "screenId": screen_id,
                    "type": 1,
                    "time": 15
                }
            )
            if rs['status'] != 0:  # ftb成功，freeze
                ftb_rs = False
        if ftb_rs:
            rs = screen_api.write_freeze(device_id, screen_id, params)
        return rs

    def screen_bkg_write(self, device_id, screen_id, params):
        """
        屏幕写bkg
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :param params: 参数
        :return: 结果数据
        """
        return screen_api.write_bkg(device_id, screen_id, params)

    def write_output_mode(self, device_id, screen_id, params):
        """
        设置屏幕布局
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :param params: 布局数据，参考前端接口文档
        :return: 结果数据
        """
        return screen_api.write_output_mode(device_id, screen_id, params)

    def get_screen_output_list(self, device_id, screen_id):
        """
        获取屏幕下所有输出口列表
        :param device_id:  设备Id
        :param screen_id:  屏幕Id
        :return:  输出列表
        """
        rs = self.read_detail(device_id, screen_id)
        if rs['status'] == 0 and rs['data']:
            rs['data'] = rs['data']['outputMode']['screenInterfaces']
        return rs

    def freeze_screen(self, device_id):
        """
        获取屏幕冻结信息
        :param device_id: 设备Id
        :return:  屏幕Id
        """
        pass

    def osd_write(self, device_id, screen_id, params, has_file):
        """
        屏幕写osd
        :param params: osd参数，参考前端接口文档writeOSD
        :return: 结果对象
        """
        # 0：文字；1：图片
        type_info = params['type']
        if type_info == 1:
            if 'words' in params:
                params.pop('words')
            if has_file:  # 有图片
                # 有图片则上传图片
                up_rs = upload_file(UploadFileType.OSD)
                if not up_rs["bln"]:
                    return get_result_model(ResInfo.Upload_File_Fail)
                path_nm = up_rs["pathNm"]
                params['image']['file']['fileName'] = path_nm
                params['image']['file']['hashSum'] = 0
            else:
                # 没有图片文件则删除file节点
                if 'image' in params and 'file' in params['image']:
                    params['image'].pop('file')
        if type_info == 0 and 'image' in params:
            params.pop('image')
        return screen_api.write_osd(device_id, screen_id, params)

    def screen_gamma_write(self, device_id, screen_id, data):
        """
        设置屏幕gamma
        :param device_id: 设备Id
        :param screen_id:  屏幕Id
        :param data:  gamma参数
        :return:  结果对象
        """
        # format_client_data(data)
        return screen_api.write_gamma(device_id, screen_id, data)

    def screen_image_quality_write(self, device_id, screen_id, data):
        """
        设置屏幕画质
        :param device_id: 设备Id
        :param screen_id:  屏幕Id
        :param data:  画质参数
        :return:  结果对象
        """
        # format_client_data(data)
        return screen_api.write_image_quality(device_id, screen_id, data)

    def freezed_output(self, device_id):
        """
        冻结、ftb的屏幕output
        :param device_id: 设备id
        :return:
        """
        return screen_api.freezed_output(device_id)

    def write_brightness(self, device_id, screen_id, data):
        """
        设置屏幕亮度，只有二合一卡可设置
        :param device_id: 设备Id
        :param screen_id: 屏幕id
        :param data: 亮度
        :return: 结果对象
        """
        rs = self.read_detail(device_id, screen_id)
        if rs['status'] != 0:
            return get_result_model(ResInfo.Screen_Detail_Error)

        if len([x for x in rs['data']['outputMode']['screenInterfaces']
                if x['modelId'] in ModelId.Two_In_One_Card_List]) <= 0:
            return get_result_model(ResInfo.Screen_Setting_Not_Allowed)

        return screen_api.write_screen_brightness(device_id, screen_id, data)

    def save_brightness(self, device_id, screen_id, data):
        """
        固化屏幕亮度，只有二合一卡可设置
        :param device_id: 设备Id
        :param screen_id: 屏幕id
        :param data: 亮度
        :return: 结果对象
        """
        rs = self.read_detail(device_id, screen_id)
        if rs['status'] != 0:
            return get_result_model(ResInfo.Screen_Detail_Error)

        if len([x for x in rs['data']['outputMode']['screenInterfaces']
                if x['modelId'] in ModelId.Two_In_One_Card_List]) <= 0:
            return get_result_model(ResInfo.Screen_Setting_Not_Allowed)

        return screen_api.save_screen_brightness(device_id, screen_id, data)

    def write_3d(self, device_id, screen_id, data):
        """
        设置屏幕3D
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :param data: 3D数据
        :return: 结果对象
        """
        return screen_api.write_screen_3d(device_id, screen_id, data)

    def write_genlock_phase_offset(self, device_id, screen_id, data):
        """
        设置屏幕3D
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :param data: 相位偏移数据
        :return: 结果对象
        """
        return screen_api.write_genlock_phase_offset(device_id, screen_id, data)

    def write_hdr(self, device_id, screen_id, data):
        """
        设置屏幕hdr
        :param device_id: 设备Id
        :param screen_id:  屏幕id
        :param data:  hdr参数
        :return:  结果对象
        """
        return screen_api.write_screen_hdr(device_id, screen_id, data)

    def read_hdr(self, device_id, screen_id):
        """
        屏幕读取hdr
        :param device_id: 设备Id
        :param screen_id:  屏幕id
        :return: hdr数据
        """
        return screen_api.read_screen_hdr(device_id, screen_id)

    def screen_clear_layers(self, device_id, screen_id):
        """
        清空屏幕图层
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :return: 结果对象
        """
        return screen_api.write_screen_clear_layers(device_id, screen_id, {'deviceId': device_id, 'screenId': screen_id})

    def screen_notime_play(self, device_id, screen_id, data):
        """
        屏幕非实时上屏take
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :param data: 图层数据
        :return: 结果对象
        """
        return screen_api.write_notime_play(device_id, screen_id, data)

    def nontime_detail_read(self, device_id, screen_id):
        """
        读取非实时上屏全部数据
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :return: 结果对象
        """
        return screen_api.read_nontime_detail(device_id, screen_id)

    def read_template_list(self, device_id):
        """
        读取屏幕布局模板列表数据
        :param device_id: 设备Id
        :return: 结果对象
        """
        rs = get_result_model()
        screen_rs = screen_api.read_template_list(device_id)
        standard_template_list = []
        customer_template_list = []
        if rs['status'] == 0 and screen_rs['data']:
            for cell in screen_rs['data']['cellList']:
                if cell['type'] == outputModeType.Standard_Type:
                    standard_template_list.append(cell)
                if cell['type'] == outputModeType.Customer_Type:
                    customer_template_list.append(cell)
        rs['data']['customerTemplateList'] = customer_template_list
        rs['data']['standardTemplateList'] = standard_template_list
        return rs

    def template_add(self, device_id, data):
        """
        添加屏幕布局模板列表数据
        :param device_id: 设备Id
        :param data: data
        :return: 结果对象
        """
        create_time = time.strftime('%Y-%m-%d %H:%M:%S', time.localtime())
        data['createTime'] = create_time
        data['type'] = outputModeType.Customer_Type
        return screen_api.template_add(device_id, data)

    def template_delete(self, device_id, template_id):
        """
        删除屏幕布局模板列表数据
        :param device_id: 设备Id
        :param template_id: template_id
        :return: 结果对象
        """
        data = {'deviceId': device_id, 'id': template_id}
        return screen_api.template_delete(device_id, data)

    def read_bright_adj_list(self, device_id, screen_id):
        """
        读取屏幕亮度调节列表
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :return: 结果对象
        """
        return screen_api.read_bright_adj_list(device_id, screen_id)

    def write_bright_adj_list(self, device_id, screen_id, data):
        """
        设置屏幕亮度调节列表
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :return: 结果对象
        """
        return screen_api.write_bright_adj_list(device_id, screen_id, data)

    def write_audio_attribute(self, device_id, screen_id, data):
        """
        设置屏幕音频属性
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :return: 结果对象
        """
        return screen_api.write_audio_attribute(device_id, screen_id, data)

    def volume_write(self, device_id, screen_id, data):
        """
        设置屏幕音频属性
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :return: 结果对象
        """
        return screen_api.volume_write(device_id, screen_id, data)

    def screen_bkg_read(self, device_id, screen_id):
        """
        屏幕写bkg
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :param params: 参数
        :return: 结果数据
        """
        rs = get_result_model()
        screen_detail_rs = screen_api.read_detail(device_id, screen_id)
        if screen_detail_rs['status'] != 0:
            return screen_detail_rs
        rs['data'] = screen_detail_rs['data']['Bkg']
        return rs

    def osd_read(self, device_id, screen_id):
        """
        屏幕读osd
        :param device_id: 设备Id
        :param screen_id:  屏幕Id
        :return: 结果对象
        """
        return screen_api.read_osd(device_id, screen_id)


screen_svc = ScreenSvc()
