'''
Created on 2018年5月7日
@author: Administrator
Sqlite数据库连接池-工具类
'''

from queue import Queue
from threading import Lock


class PoolException(Exception):
    pass


class Pool(object):
    '''''一个数据库连接池'''

    def __init__(self, maxActive=50, maxWait=None, init_size=0, db_type="SQLite3", **config):
        self.__freeConns = Queue(maxActive)
        self.maxWait = maxWait
        self.db_type = db_type
        self.config = config

        if init_size > maxActive:
            init_size = maxActive

        for i in range(init_size):
            self.free(self._create_conn())

    def __del__(self):
        self.release()

    '''''释放资源，关闭池中的所有连接'''

    def release(self):
        while self.__freeConns and not self.__freeConns.empty():
            con = self.get()
            con.release()
        self.__freeConns = None

    '''''创建连接 '''

    def _create_conn(self):
        if self.db_type in dbcs:
            return dbcs[self.db_type](**self.config);

    '''''获取一个连接 
        @param timeout:超时时间 
    '''

    def get(self, timeout=None):

        if timeout is None:
            timeout = self.maxWait

        conn = None

        if self.__freeConns.empty():  # 如果容器是空的，直接创建一个连接
            conn = self._create_conn()
        else:
            conn = self.__freeConns.get(timeout=timeout)
        conn.pool = self

        return conn

    '''''将一个连接放回池中 
        @param conn: 连接对象 
    '''

    def free(self, conn):
        conn.pool = None
        if (self.__freeConns.full()):  # 如果当前连接池已满，直接关闭连接
            conn.release()
            return
        self.__freeConns.put_nowait(conn)


from abc import ABCMeta, abstractmethod


class PoolingConnection(object, metaclass=ABCMeta):
    def __init__(self, **config):
        self.conn = None
        self.config = config
        self.pool = None

    def __del__(self):
        self.release()

    def __enter__(self):
        pass

    def __exit__(self, exc_type, exc_value, traceback):
        self.close()

    def release(self):
        if (self.conn is not None):
            self.conn.close()
            self.conn = None
        self.pool = None

    def close(self):
        if self.pool is None:
            raise PoolException("连接已关闭")
        self.pool.free(self)

    def __getattr__(self, val):
        if self.conn is None and self.pool is not None:
            self.conn = self._create_conn(**self.config)
        if self.conn is None:
            raise PoolException("无法创建数据库连接 或连接已关闭")
        return getattr(self.conn, val)

    @abstractmethod
    def _create_conn(self, **config):
        pass


class SQLit3PoolConnection(PoolingConnection):
    def _create_conn(self, **config):
        import sqlite3
        return sqlite3.connect(**config, check_same_thread=False)


dbcs = {"SQLite3": SQLit3PoolConnection}

# pool = Pool(init_size=5, database=r'H_9U/conf/h9u.db')
pool = Pool(init_size=5, database=r'../data/h9u.db')
# pool = Pool(init_size=5, database=r'../h9u.db')
sqllock = Lock()


def test():
    conn = pool.get()
    # cursor = conn.cursor() conn竟然也可以直接操作库表
    with conn:
        row = conn.execute("SELECT * FROM sysUser")
        print(row)

    #     conn2 = pool.get()
    #     with conn2:
    #         conn2.execute("SELECT * FROM USER")
    #
    # conn3 = pool.get()
    # with conn3:
    #     for a in conn3.execute("SELECT * FROM USER"):
    #         print(a)
    #     print('conn3 after')
    #     print('conn3 after')


if __name__ == "__main__":
    test()

