"""
Created on 2019年6月28日
@author: 刘益宗
bkg接口.
对应前端接口bkg
返回值统一为 {data:JSON结构体, msg:"", status:0}
"""

from flask.globals import request
from flask.json import jsonify
from flask import Blueprint

from H_9U.service.bkg import bkg_svc
from H_9U.util.common import valid_json, valid_params
from H_9U.models.result import get_result_model, ResInfo
from H_9U.api.websender import push_data, push_data_org
from H_9U.models.syncdataname import SyncDataName
from H_9U.util.logwrapper import api_log


bkg_bp = Blueprint('bkg', __name__)


@bkg_bp.route('/readAllList', methods=['GET', 'POST'])
@api_log
def read_all_list():
    """
    读取全部bkg信息
    :return: 见前端bkg文档
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    rs = bkg_read_list(device_id)
    return jsonify(rs)


def bkg_read_list(device_id):
    """
    读取全部bkg详细信息
    :param device_id: 设备Id
    :return: bkg列表
    """
    return bkg_svc.read_all_list(device_id)


@bkg_bp.route('/readDetail', methods=['GET', 'POST'])
def read_detail():
    """
    读取全部bkg信息
    :return:见前端bkg文档
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'bkgId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, bkg_id = values

    return jsonify(bkg_svc.bkg_read_detail(device_id, bkg_id))


@bkg_bp.route('/create', methods=['GET', 'POST'])
def bkg_create():
    """
    创建BKG
    :return: bkg Id数据
    """
    params = request.form.get('json')
    params = valid_json(params)
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'name')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, name = values

    rs = bkg_svc.bkg_create(device_id, name)
    if rs['status'] == 0:
        bkg_id = rs['data']['id']
        detail_rs = bkg_svc.bkg_read_detail(device_id, bkg_id)
        push_data_org(SyncDataName.Bkg_Read_Detail, detail_rs['data'])
    return jsonify(rs)


@bkg_bp.route('/delete', methods=['GET', 'POST'])
@api_log
def bkg_delete():
    """
    删除bkg
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'bkgId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, bkg_id = values

    from H_9U.api.screen import screen_used_bkg
    rs = screen_used_bkg(device_id)
    if rs['status'] == 0:
        if bkg_id in rs['data']:
            rs = get_result_model(ResInfo.BKG_Used_In_Screen)
        else:
            rs = bkg_svc.bkg_delete(device_id, bkg_id)
    else:
        rs = get_result_model(ResInfo.BKG_Used_In_Screen)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Bkg_Delete, params)
    return jsonify(rs)


@bkg_bp.route('/deleteBatch', methods=['GET', 'POST'])
@api_log
def bkg_delete_batch():
    """
    bkg批量删除
    :return: 见前端文档bkg
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'model')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, model = values

    bkg_ids = params.get('bkgIds')
    # 验证models
    if model not in [0, 1]:
        return get_result_model(ResInfo.Params_Error)
    # model=1时bkgIds参数不可为空
    if model == 1 and not bkg_ids:
        return get_result_model(ResInfo.Params_Error)

    from H_9U.api.screen import screen_used_bkg
    rs = screen_used_bkg(device_id)
    if rs['status'] != 0:
        return rs
    bkg_used = rs['data']
    rs = bkg_svc.bkg_delete_batch(device_id, model, bkg_ids, bkg_used)
    if rs['status'] == 0:
        push_data(bkg_read_list, SyncDataName.Bkg_Read_All_List, params.get('deviceId'))
    return jsonify(rs)


@bkg_bp.route('/rename', methods=['GET', 'POST'])
@api_log
def bkg_rename():
    """
    bkg重命名
    :return:  无
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'bkgId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, bkg_id = values

    rs = bkg_svc.bkg_general_write(device_id, bkg_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Bkg_Rename, params)
    return jsonify(rs)


def bkg_image_url(device_id, bkg_id):
    """
    bkg url数据
    :param device_id: 设备Id
    :param bkg_id: bkgId
    :return: {imageUrl:}
    """
    rs = bkg_svc.bkg_read_detail(device_id, bkg_id)
    if rs['status'] == 0 and rs['data']:
        rs['data'] = {
            'imageUrl': rs['data']['data']['image']['imgUrl']
        }
    return rs
