'''
Created on 2019年8月18日
@author: GongQingBao
选择通讯器.
'''

import random
import threading
import time
import traceback

from H_9U.protocol.fncode import FnCode
from H_9U.util.log import logger


class SelectCommunicator(object):


    def __init__(self, creator, createNum=5, maxCommunicatorNum=50):
        """构造函数
        :param creator:通讯器创建器
        :param createNum:每次创建通讯器的数量
        :param maxCommunicatorNum:通讯器数量上限
        """
        self.creator = creator
        self.createNum = createNum
        self.maxCommunicatorNum = maxCommunicatorNum

        self.optType_read = 0  # 操作类型-读
        self.optType_write = 1 # 操作类型-写
        self.timeoutCache = {}    # 缓存：记录功能码对应的超时时间

        self.threadLock = threading.Lock() # 线程同步锁


    def select(self, sessionId, fnCode, communicatorList, optType):
        """根据sessionID选择匹配的'通讯器',并设置通讯timeout.
        :param sessionId：会话ID
        :param fnCode:功能码
        :param communicatorList:所有'通讯器'实例集合
        :param optType:操作类型[0-读、1-写]
        :return:一个Communicator实例[可能为None]
        """
        self.threadLock.acquire() # 同步
        communicator = None
        
        try:
            if sessionId == None:
                logger.error("选择 通讯器-失败:sessionId不可为空!采用默认值.")
                sessionId = "Auto_" + str(random.uniform(10,20)).replace('.','')
    
            if fnCode == None:
                logger.error("选择 通讯器-失败:fnCode不可为空!")
                self.threadLock.release() # 释放锁
                return communicator
            else:
                if fnCode == 0x0109: # 如果是'强制保存'功能时打印当前socket链接数量
                    logger.warning("当前socket链接数量:%i" % len(communicatorList))
    
            if communicatorList == None or len(communicatorList) == 0:
                logger.error("选择 通讯器-失败:communicatorList不可为空!")
                self.threadLock.release() # 释放锁
                return communicator
    
            # 根据'操作类型'使用不同的方式选择通讯器
            if optType == self.optType_read:
                communicator = self.select_read(sessionId, communicatorList)
            elif optType == self.optType_write:
                communicator = self.select_write(sessionId, communicatorList)
            else:
                logger.error("选择 通讯器-失败:optType不在定义范围之内!")
    
            # 设置通讯器的超时时间
            if communicator != None:
                self.setTimeout(communicator, fnCode, optType)

        except Exception:
            msg = traceback.format_exc()
            logger.error("根据sessionID选择匹配的'通讯器',并设置通讯timeout:出现异常!excetion:%s" % msg)

        self.threadLock.release() # 释放锁
        return communicator


    def select_read(self, sessionId, communicatorList):
        """根据sessionID选择匹配的'通讯器'[读操作]
        :param sessionId：会话ID
        :param communicatorList:所有'通讯器'实例集合
        :return:一个Communicator实例
        """
        # 选择一个空闲且处于'活动中'的通讯器
        for temp in communicatorList:
            logger.debug("select_read. 空闲的通讯器(1)! num:%i, busying:%s" % (len(communicatorList), temp.busying))
            if not temp.busying and temp.pipe.linked:
                self.flushStatus(temp, sessionId) # 刷新通讯器状态
                logger.debug("select_read. 空闲的通讯器(2)! num:%i, busying:%s" % (len(communicatorList), temp.busying))
                return temp

        # 判断通讯器数量是否达到上限
        isFull = True if len(communicatorList) == self.maxCommunicatorNum else False
        communicator = None

        # 若未达到数量上限则创建新的通讯器
        if not isFull:
            logger.debug("select_read. 无空闲、创建新的通讯器! beforeNum:%i" % (len(communicatorList)))
            communicator = self.createCommunicator(communicatorList) # 创建新的通讯器
            logger.debug("select_read. 无空闲、创建新的通讯器! afterNum:%i" % (len(communicatorList)))

        # 如果新建失败,则使用集合中的首个'活动中'的通讯器
        if communicator == None or (communicator != None and not communicator.pipe.linked):
            communicator = self.getFirstActiveCommunicator(communicatorList)
            logger.debug("select_read. 新建失败,则使用集合中的最后一个'活动中'的通讯器!")

        self.flushStatus(communicator, sessionId) # 刷新通讯器状态

        return communicator


    def select_write(self, sessionId, communicatorList):
        """根据sessionID选择匹配的'通讯器'[写操作]
        :param sessionId：会话ID
        :param communicatorList:所有'通讯器'实例集合
        :return:一个Communicator实例
        """
        # 从通讯器集合中找到与指定sessionId相同的通讯器,并返回.
        for temp in communicatorList:
            if temp.lastSessionId == sessionId and temp.lastOptType == self.optType_write:
                self.flushStatus(temp, sessionId, self.optType_write) # 刷新通讯器状态
                logger.debug("select_write. sessionId相同的通讯器,并返回! num:%i" % (len(communicatorList)))
                return temp

        # 从通讯器集合中找到未被使用其'活动中'的通讯器,并返回.
        for temp in communicatorList:
            if temp.lastSessionId == None and temp.pipe.linked:
                self.flushStatus(temp, sessionId, self.optType_write) # 刷新通讯器状态
                logger.debug("select_write. 未被使用的通讯器,并返回! num:%i" % (len(communicatorList)))
                return temp

        # 从通讯器集合中找到空闲且'活动中'的通讯器,并返回.
        for temp in communicatorList:
            if not temp.busying and temp.pipe.linked:
                self.flushStatus(temp, sessionId, self.optType_write) # 刷新通讯器状态
                logger.debug("select_write. 空闲的通讯器,并返回! num:%i" % (len(communicatorList)))
                return temp

        # 判断通讯器数量是否达到上限
        isFull = True if len(communicatorList) == self.maxCommunicatorNum else False

        # 若未达到数量上限则创建新的通讯器
        if not isFull:
            logger.debug("select_write. 未达到数量上限,新建通讯器! beforeNum:%i" % (len(communicatorList)))
            temp = self.createCommunicator(communicatorList) # 创建新的通讯器
            self.flushStatus(temp, sessionId, self.optType_write) # 刷新通讯器状态
            logger.debug("select_write. 未达到数量上限,新建通讯器! afterNum:%i" % (len(communicatorList)))

            return temp
        else: # 数量达到上限：从通讯器集合中找'逾期时间'最长的通讯器,并返回.
            communicator = self.findDueTimeMaxLong(communicatorList)

            if communicator != None:
                self.flushStatus(communicator, sessionId, self.optType_write) # 刷新通讯器状态
                logger.debug("select_write. 数量达到上限：从通讯器集合中找'逾期时间'最长的通讯器,并返回! num:%i" % (len(communicatorList)))
            else:
                logger.error("选择 通讯器-失败(写):正常情况下communicator不应该为空!")

            return communicator


    def getFirstActiveCommunicator(self, communicatorList, model=True):
        """返回首个'活动中'的通讯器,若没有'活动中'则返回首个通讯器.
        :param communicatorList:所有'通讯器'实例集合
        :param model:没有'活动中'时的处理模式[True-返回首个、False-返回None]
        :return:Communicator实例[可能为None]
        """
        for temp in communicatorList:
            if temp.pipe.linked:
                return temp
        
        if model:
            return communicatorList[0] # 若没有'活动中'则返回首个通讯器
        else:
            return None


    def createCommunicator(self, communicatorList):
        """创建新的通讯器,并加入到communicatorList中,并返回新建的首个'活动中'的通讯器.
        :param communicatorList:所有'通讯器'实例集合
        :return:Communicator实例
        """
        logger.warning("创建socket链接-前-数量:%i" % len(communicatorList))

        communicators = self.creator.create(self.createNum, True) # 创建新的通讯器
        communicatorList.extend(communicators) # 将新建的通讯器加入到集合中

        logger.warning("创建socket链接-后-数量:%i" % len(communicatorList))

        for temp in communicators:
            if temp.pipe.linked:
                return temp

        logger.warning("新创建的%i个socket链接-中没有链接成功的!" % self.createNum)

        return communicators[0] # 如果没有'活动中'的则返回首个


    def findDueTimeMaxLong(self, communicatorList):
        """从通讯器集合中找'逾期时间'最长且'活动中'的通讯器.
        :param communicatorList:所有'通讯器'实例集合
        :return:communicator实例[有可能为空]
        """
        sysTime = time.monotonic() # 系统时间.返回当前时间的时间戳（1970纪元后经过的浮点秒数）
        dueTime = 0 # 逾期时间
        communicator = None
        communicatorActive = None

        for temp in communicatorList:
            if temp != None:
                if temp.lastAccesTime != None:
                    dueTimeTemp = abs(sysTime - temp.lastAccesTime)
                else:
                    dueTimeTemp = sysTime
    
                if dueTimeTemp > dueTime:
                    dueTime = dueTimeTemp
                    communicator = temp
    
                    if communicator.pipe.linked:
                        communicatorActive = communicator
            else:
                logger.error("不应成出现None对象-程序逻辑错误!")

        # 优先使用'逾期时间'最长且'活动中'的,若没有获得的则使用'逾期时间'最长的.
        if communicatorActive != None:
            return communicatorActive
        else:
            if communicator == None:
                logger.error("communicator不应为None-程序逻辑错误!")
            return communicator


    def flushStatus(self, communicator, sessionId, optType=0):
        """刷新通讯器状态.
        :param communicator:被更新状态的通讯器
        :param sessionId:会话ID
        :param optType:操作类型[0-读|1-写]
        """
        if communicator != None:
            sysTime = time.monotonic() # 系统时间.返回当前时间的时间戳（1970纪元后经过的浮点秒数）
            
            # 更新：lastAccesTime、lastSessionId属性值.
            communicator.lastSessionId = sessionId
            communicator.lastAccesTime = sysTime
            communicator.lastOptType = optType
            communicator.busying = True


    def setTimeout(self, communicator, fnCode, optType):
        """设置通讯器的超时时间.
        :param communicator:被设置超时时间的通讯器
        :param fnCode:功能代码
        :param optType:操作类型[0-读|1-写]
        """
        key = str(fnCode) + "_" + str(optType) # 在缓存中的key值

        # 优先从缓存中获取
        maxUsedTime = self.timeoutCache.get(key, None)
        
        if maxUsedTime == None:
            default_maxUsedTime = 5 # 默认5秒
            find_maxUsedTime = None
    
            # 获得指定的fnCode规定执行的'最大耗时时间'
            for temp in FnCode.FnCode_Group:
                fnCodeList = temp.get("fnCodeList")
    
                if fnCodeList != None:
                    if fnCode in fnCodeList:
                        find_maxUsedTime = temp.get("maxUsedTime")
                        break
                else: # fnCodeList==None表示默认值
                    default_maxUsedTime = temp.get("maxUsedTime")
    
            # 如果未找到合适的'最大耗时时间',则使用默认的
            if find_maxUsedTime == None:
                maxUsedTime = default_maxUsedTime
            else:
                maxUsedTime = find_maxUsedTime
    
            self.timeoutCache[key] = maxUsedTime

        communicator.setTimeout(maxUsedTime)


