"""
Created on 2021年9月08日
@author: 程声清
layer api
参考前端接口文档layer部分
返回值统一为 {body:加密串, msg:"", sign:"", status:0}
"""
from flask import Blueprint, g
from flask.json import jsonify
from H_9U.api.input import input_get_connect_capacity
from H_9U.api.layer import __build_ipc_data, _screen_nontime, screen_layer_detail_list
from H_9U.api.websender import push_data_org, push_data
from H_9U.models.result import get_result_model, ResInfo
from H_9U.models.sysconst import SourceType, LayerType, ConnectCapacity
from H_9U.open.service import open_layer_svc, open_screen_svc
from H_9U.util.common import get_params
from H_9U.util.logwrapper import open_log
from H_9U.models.syncdataname import SyncDataName

open_layer_bp = Blueprint('open_layer', __name__)


@open_layer_bp.route('/screenLayerTake', methods=['GET', 'POST'])
@open_log
def screen_layer_take():
    """
    屏幕上屏参考接口文档screenLayerTake
    :return: 结果对象
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    rs = open_layer_svc.layer_screen_no_time_play(
        device_id, screen_id, params)
    # 推送
    if rs['status'] == 0:
        push_data_org(SyncDataName.Layer_Screen_Take, params)
    return jsonify((rs))


@open_layer_bp.route('/create', methods=['GET', 'POST'])
@open_log
def layer_create():
    """
    添加Layer至screen
    :return: 结果对象
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    if not params['general']:
        return jsonify(get_result_model(ResInfo.Params_Error))

    input_id = params['source']['inputId']

    # 图层类型
    params['general']['type'] = LayerType.Default

    if params['source']['sourceType'] == SourceType.InputSource:
        # 输入接口容量：
        rs = input_get_connect_capacity(device_id, input_id)
        # 获取不到连接容量，直接返回错误
        if rs['status'] != 0:
            return jsonify(get_result_model(ResInfo.Middle_Data_Err))
        # 兼容以前的数据
    elif params['source']['sourceType'] == SourceType.IPC:  # 255 表示使用IPC源
         params['source']['sourceType'] = 3

    params['general']['sizeType'] = ConnectCapacity.SL  # conn
    rs = open_layer_svc.layer_create(device_id, screen_id, params)
    if rs['status'] == 0:
        layer_id = rs['data']['id']
        params['id'] = layer_id
        push_data_org(SyncDataName.Layer_Create, params)
    return jsonify((rs))


@open_layer_bp.route('/delete', methods=['GET', 'POST'])
@open_log
def layer_delete():
    """
     删除Layer
    :return: 结果对象
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId', 'layerId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, layer_id = values
    rs = open_layer_svc.layer_delete(device_id, screen_id, layer_id)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Layer_Delete, {
            "deviceId": device_id,
            "screenId": screen_id,
            "layerId": layer_id
        })
    return jsonify((rs))


@open_layer_bp.route('/detailList', methods=['GET', 'POST'])
@open_log
def layer_read_list():
    """
    获取图层详细信息列表
    :return: 结果对象
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    rs = open_screen_svc.screen_layer_detail_list(device_id, screen_id)
    return jsonify((rs))


@open_layer_bp.route('/readDetail', methods=['GET', 'POST'])
@open_log
def layer_detail():
    """
     读取详细信息
     :return: 参考前端接口文档readDetail
     """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId', 'layerId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, layer_id = values
    rs = open_layer_svc.layer_read_detail(device_id, screen_id, layer_id)
    return jsonify((rs))


@open_layer_bp.route('/writeGeneral', methods=['GET', 'POST'])
@open_log
def layer_write_general():
    """
    设置Layer基础信息
    :return: 参考文档writeGeneral
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId', 'layerId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, layer_id = values
    rs = open_layer_svc.layer_write_general(
        device_id, screen_id, layer_id, params)
    if rs['status'] == 0:
        nontime = _screen_nontime(device_id, screen_id)
        if nontime == 0:
            push_data(layer_read_detail, SyncDataName.Layer_Write_General, device_id, screen_id, layer_id)
        elif nontime == 1:
            from H_9U.api.screen import screen_non_time_detail
            rs = screen_non_time_detail(device_id, screen_id)
            if rs['status'] == 0:
                layers = [x for x in rs['data']['layers'] if x['layerId'] == layer_id]
                if layers:
                    layer = layers[0]
                    layer['screenId'] = screen_id
                    layer['deviceId'] = device_id
                    push_data_org(SyncDataName.Layer_Write_General, layer)
    return jsonify((rs))

def layer_read_detail(device_id, screen_id, layer_id):
    """
    读取屏幕的图层数据
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :param layer_id: 图层Id
    :return: 参考readDetail文档    """
    rs = screen_layer_detail_list(device_id, screen_id)
    if rs['status'] == 0 and rs['data']:
        detail = [x for x in rs['data'] if x['layerId'] == layer_id]
        if detail:
            rs['data'] = detail[0]
        else:
            rs['data'] = {}
    return rs


@open_layer_bp.route('/readZIndex', methods=['GET', 'POST'])
@open_log
def layer_read_z_index():
    """
    读取屏幕图层Z index
    :return:  参考文档 readZIndex
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    rs = open_layer_svc.layer_read_z_index(device_id, screen_id)
    return jsonify((rs))


@open_layer_bp.route('/writeWindow', methods=['GET', 'POST'])
@open_log
def layer_write_layout_window():
    """
   设置指定图层的位置和大小信息
    :return:  参考文档 writeWindow
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId', 'layerId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, layer_id = values
    rs = open_layer_svc.layer_write_layout_window(
        device_id, screen_id, layer_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Layer_Write_Window, params)
    return jsonify((rs))


@open_layer_bp.route('/clear', methods=['GET', 'POST'])
@open_log
def layer_clear():
    """
    屏幕图层清除
    :return:  参考文档 clear
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    rs = open_layer_svc.layer_clear(device_id, screen_id)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Layer_Delete_All, {'deviceId': device_id, 'screenId': screen_id, 'layers': []})
    return jsonify((rs))


@open_layer_bp.route('/writeZIndex', methods=['GET', 'POST'])
@open_log
def layer_write_z_index():
    """
    读取屏幕图层Z index
    :return:  参考文档 readZIndex
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    rs = open_layer_svc.layer_write_z_index(device_id, screen_id, params)
    if rs['status'] == 0:
        nontime = _screen_nontime(device_id, screen_id)
        z_rs = None
        if nontime == 0:
            z_rs = open_layer_svc.layer_read_z_index(device_id, screen_id)
        elif nontime == 1:
            from H_9U.api.screen import screen_non_time_detail
            rs = screen_non_time_detail(device_id, screen_id)
            if rs['status'] == 0:
                layers = [x for x in rs['data']['layers']]
                data = {"screenId": screen_id, "deviceId": device_id, "layersZOrder": []}
                for layer in layers:
                    data["layersZOrder"].append({'layerId': layer['layerId'], 'zOrder': layer['general']['zorder']})
                z_rs = get_result_model()
                z_rs['data'] = data
        if z_rs and z_rs['status'] == 0:
            rs = z_rs
            # 推送
            push_data_org(SyncDataName.Layer_Write_Z_Index, rs['data'])
    return jsonify((rs))


@open_layer_bp.route('/writeLock', methods=['GET', 'POST'])
@open_log
def layer_write_z_lock():
    """
    锁定开启、关闭
    :return:  参考文档 writeLock
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId', 'layerId', 'lock')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, layer_id, is_lock = values
    nontime = _screen_nontime(device_id, screen_id)
    if nontime is not None:
        rs = open_layer_svc.layer_write_z_lock(
            device_id, screen_id, layer_id, is_lock)
    else:
        rs = get_result_model(ResInfo.Middle_Data_Err)
    if nontime == 0:
        push_data(layer_read_detail, SyncDataName.Layer_Write_Lock, device_id, screen_id, layer_id)
    elif nontime == 1:
        from H_9U.api.screen import screen_non_time_detail
        rs = screen_non_time_detail(device_id, screen_id)
        if rs['status'] == 0:
            layers = [x for x in rs['data']['layers'] if x['layerId'] == layer_id]
            if layers:
                layer = layers[0]
                layer['screenId'] = screen_id
                layer['deviceId'] = device_id
                push_data_org(SyncDataName.Layer_Write_Lock, layer)
    return jsonify((rs))


@open_layer_bp.route('/writeSource', methods=['GET', 'POST'])
@open_log
def layer_write_source():
    """
    更改图层源信息
    :return:  参考文档 writeSource
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId', 'layerId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, layer_id = values
    rs = open_layer_svc.layer_write_source(
        device_id, screen_id, layer_id, params)
    if rs['status'] == 0:
        nontime = _screen_nontime(device_id, screen_id)
        if nontime == 0:
            push_data(layer_read_detail, SyncDataName.Layer_Write_Source, device_id, screen_id, layer_id)
        elif nontime == 1:
            from H_9U.api.screen import screen_non_time_detail
            rs = screen_non_time_detail(device_id, screen_id)
            if rs['status'] == 0:
                layers = [x for x in rs['data']['layers'] if x['layerId'] == layer_id]
                if layers:
                    layer = layers[0]
                    layer['screenId'] = screen_id
                    layer['deviceId'] = device_id
                    push_data_org(SyncDataName.Layer_Write_Source, layer)
        if rs['status'] == 0:
            nontime = _screen_nontime(device_id, screen_id)
            if nontime == 0:
                push_data(layer_read_detail, SyncDataName.Layer_Write_Source, device_id, screen_id, layer_id)
            elif nontime == 1:
                from H_9U.api.screen import screen_non_time_detail
                rs = screen_non_time_detail(device_id, screen_id)
                if rs['status'] == 0:
                    layers = [x for x in rs['data']['layers'] if x['layerId'] == layer_id]
                    if layers:
                        layer = layers[0]
                        layer['screenId'] = screen_id
                        layer['deviceId'] = device_id
                        push_data_org(SyncDataName.Layer_Write_Source, layer)
    return jsonify((rs))


@open_layer_bp.route('/freeze', methods=['GET', 'POST'])
@open_log
def layer_freeze():
    """
    冻结图层
    :return:  参考文档 freeze
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId', 'layerId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, layer_id = values
    rs = open_layer_svc.layer_freeze(
        device_id, screen_id, layer_id, params)
    if rs['status'] == 0:
        nontime = _screen_nontime(device_id, screen_id)
        if nontime == 0:
            push_data(layer_read_detail, SyncDataName.Layer_Freeze, device_id, screen_id, layer_id)
        elif nontime == 1:
            from H_9U.api.screen import screen_non_time_detail
            rs = screen_non_time_detail(device_id, screen_id)
            if rs['status'] == 0:
                layers = [x for x in rs['data']['layers'] if x['layerId'] == layer_id]
                if layers:
                    layer = layers[0]
                    layer['screenId'] = screen_id
                    layer['deviceId'] = device_id
                    push_data_org(SyncDataName.Layer_Freeze, layer)
    return jsonify((rs))


@open_layer_bp.route('/screenLayerLayout', methods=['GET', 'POST'])
@open_log
def screen_layer_layout():
    """
    图层模板开窗
    :return:  参考文档 screenLayerLayout
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    rs = open_layer_svc.screen_layer_layout(device_id, screen_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Layer_Screen_Layout, params)
    return jsonify((rs))



@open_layer_bp.route('/writeStreamRule', methods=['GET', 'POST'])
@open_log
def write_stream_rule():
    """
     设置ipc源图层使用主辅码流
    :return:
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId', 'layerId', 'channelId', 'streamIndex')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = params.get('deviceId')
    rs = open_layer_svc.write_stream_rule(device_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Layer_Screen_Layout, params)
    return jsonify((rs))