"""
Created on 2021年9月14日
@author: 程声清
input服务类
"""
from copy import deepcopy
from H_9U.models.result import get_result_model, ResInfo
from H_9U.open.middle import open_device_middle, open_input_middle
from H_9U.models.sysconst import CardType, UploadFileType, InputData, Max_Band_Capacity_Map, SourceType
from H_9U.service import common
from H_9U.util.log import logger
from H_9U.util.outputtiming import outputResolution


def input_read_list(device_id):
    """
    输入列表
    :param device_id
    :return: 无分组信息
    """
    # 读取全部输入
    input_list_rs = open_input_middle.input_read_list_middle(device_id)
    rs = get_result_model()
    if input_list_rs["status"] == 0:
        input_list = input_list_rs["data"]['inputs']
        inputs = [item for item in input_list if item['online'] != 0]
        rs['data']['inputs'] = inputs
    return rs


def input_detail(device_id, input_id):
    """
    读取input详细信息
    :param device_id:  设备Id
    :param input_id:  输入Id
    :return:  参考接口readDetail
    """
    rs = get_result_model()
    input_device_detail = open_input_middle.read_input_detail_middle(
        device_id, input_id)
    if input_device_detail['status'] == 0:
        # 补充截取信息
        crops = []
        if input_device_detail['data']['online'] == 1:
            input_crops = open_input_middle.read_crop_lists_middle(
                device_id, input_id)
            if input_crops['status'] == 0 and input_crops['data']:
                crops = input_crops['data']['crops']
        rs['data']['crops'] = crops
        rs['data']['iSignal'] = 1 if input_device_detail['data']['resolution']['refresh'] > 0 else 0
        rs['data']['general'] = input_device_detail['data']['general']
        rs['data']['resolution'] = input_device_detail['data']['resolution']
        rs['data']['inputId'] = input_device_detail['data']['inputId']
        rs['data']['deviceId'] = device_id
        rs['data']['slotId'] = input_device_detail['data']['slotId']
        rs['data']['interfaceId'] = input_device_detail['data']['interfaceId']
        rs['data']['interfaceType'] = input_device_detail['data']['interfaceType']
    else:
        rs = get_result_model(ResInfo.Middle_Data_Err)
    return rs


def input_rename(device_id, input_id, data):
    """
    设置基本信息
    :param device_id: 设备Id
    :param input_id:  输入Id
    :param data: 数据
    :return: 结果对象
    """
    rs = open_input_middle.input_rename_middle(device_id, input_id, data)
    if rs["status"] != 0:
        logger.error("设置重命名-失败! CODE:%i, params:%s" % (rs["status"], data))
    return rs


def input_crop_create(device_id, input_id, data):
    """
    创建结果参数
    :param device_id: json参数
    :param input_id: json参数
    :param data: json参数
    :return: 结果对象
    """
    if 'sourceType' not in data:
        data['sourceType'] = SourceType.InputSource
    if 'sourceSize' not in data:
        data['sourceSize'] = {
            "width": data['width'],
            "height": data['height'],
            "x": 0,
            "y":0
        }
    rs = open_input_middle.crop_create_middle(device_id, input_id, data)
    crop_rs = get_result_model()
    crop_rs['msg'] = rs['msg']
    crop_rs['status'] = rs['status']
    if rs['status'] == 0:
        crop_rs['data']['cropId'] = rs['data']['id']
    return crop_rs


def input_crop_delete(device_id, input_id, crop_id, params):
    """
    删除crop
    :param device_id: 设备id
    :param input_id: 输入id
    :param crop_id: 截取Id
    :param params: json参数
    :return: 结果对象
    """
    return open_input_middle.input_crop_delete_middle(
        device_id, input_id, crop_id, params)


def input_crop_update(device_id, input_id, crop_id, params):
    """
    修改输入截取
    :param params: 参考接口文档updateCrop
    :param device_id: 设备Id
    :param input_id: input id
    :param crop_id: 截取 id
    :return: 结果对象
    """
    if 'sourceType' not in params:
        params['sourceType'] = SourceType.InputSource
    if 'sourceSize' not in params:
        params['sourceSize'] = {
            "width": params['width'],
            "height": params['height'],
            "x": 0,
            "y": 0
        }
    return open_input_middle.input_crop_update_middle(
        device_id, input_id, crop_id, params)


def input_ed_id_import(device_id, input_id, file_data, file_name):
    """
    批量导入ed_id
    :param device_id: 设备Id
    :param input_id: 输入Id列表
    :param file_data: 文件数据
    :param file_name: 文件名称
    :return: 结果对象
    """
    # st = time.time()  # 返回当前时间的时间戳（1970纪元后经过的浮点秒数）
    up_rs = common.open_upload_file(
        UploadFileType.Input_EDID,
        file_data,
        file_name,
        extract=False)
    # 文件上传成功后还要调用中间件进行设备升级
    if up_rs["bln"] is not True:
        logger.error("导入ed_id-文件上传-失败!")
        return get_result_model(ResInfo.Upload_File_Fail)
    data_list = []
    path_nm = up_rs["pathNm"]
    data_list.append({
        "inputId": int(input_id),
        "deviceId": device_id,
        "file": {
            "fileName": path_nm,
            "fileData": '',
            "hashSum": 0
        }
    })
    rs = open_input_middle.input_write_ed_id_middle(
        device_id, input_id, data_list)
    if rs['status'] == 0:
        data = {
            'inputId': int(input_id),
            'deviceId': device_id,
            'templateOperationType': 1
        }
        open_input_middle.input_write_timing_middle(
            device_id, int(input_id), data)

    return rs


def input_write_timing(device_id, input_id, params):
    """
    批量设置
    :param device_id: 设备Id
    :param input_id: input_id
    :param params: json 参数
    :return: 结果对象
    """
    data = []
    try:
        # 处理刷新率
        timing = deepcopy(params)
        timing['inputId'] = input_id
        rs = open_input_middle.input_write_timing_middle(
            device_id, input_id, timing)
        if rs["status"] != 0:
            logger.debug(
                "设置Timing-失败! CODE:%i, params:%s" %
                (rs["status"], input_id))
            success = 0  # 是否部分成功
        else:
            success = 1
        data.append({
            'inputId': input_id,
            'success': success
        })
        rs['data'] = data
    except Exception as e:
        logger.error(e)
        rs = get_result_model(ResInfo.Input_Write_Timing_Error)
    return rs


def input_group_create(device_id, params):
    """
    创建输入分组
    :param device_id: 设备Id
    :param params: 组内容
    :return: 结果对象
    """
    rs = open_input_middle.read_group_list_middle(device_id)
    if rs['status'] == 0 and rs['data']:
        if len(rs['data']['inputGroupList']) > InputData.Group_Max_Count:
            return get_result_model(ResInfo.Input_Group_Max_Count_Err)
    group_rs = open_input_middle.write_group_create_middle(device_id, params)
    return group_rs


def input_group_order(device_id, data):
    """
    输入分组节点枚举
    :param device_id: 设备Id
    :param data: 组列表信息
    :return: 结果对象，见文档
    """
    return open_input_middle.input_group_list_update_middle(device_id, data)


def input_group_update(device_id, group_id, input_group_name, data):
    """
    输入分组修改
    :param device_id: 设备Id
    :param group_id: 分组Id
    :param input_group_name: 输入分组名称
    :param data: 组列表信息
    :return: 结果对象，见文档
    """
    group_data = {
        "deviceId": device_id,
        'groups': [
            {
                "inputGroupId": group_id,
                "inputGroupName": input_group_name
            }
        ]
    }
    open_input_middle.input_group_list_update_middle(
        device_id, group_data)
    rs = open_input_middle.input_group_node_update_middle(
        device_id, group_id, data)
    return rs


def input_group_delete(device_id, group_id, data):
    """
    删除输入分组
    :param device_id: 设备Id
    :param group_id: 组Id
    :param data: 分组数据
    :return: 结果对象
    """
    return open_input_middle.input_write_group_delete_middle(
        device_id, group_id, data)


def input_group_node_delete(device_id, group_id, input_id, data):
    """
    删除分组输入源
    :param device_id: 设备Id
    :param group_id: 组Id
    :param input_id: 输入源 Id
    :param data: 分组数据
    :return: 结果对象
    """
    return open_input_middle.input_write_group_node_delete_middle(
        device_id, group_id, input_id, data)


def _get_used_inputs_dict(device_id):
    """
    读取被使用的输入源dict
    :param device_id: dictId
    :return: {slotId:[interfaceId1, interfaceId2, ...]}
    """
    rs = open_device_middle.device_detail_without_cache_middle(device_id)
    if rs['status'] == 0:
        used_dict = {}
        input_slots = [x for x in rs['data']['slotList']
                       if x['cardType'] == CardType.In and x['interfaces']]

        for slot in input_slots:
            for interface in slot['interfaces']:
                if interface['isUsed'] == 1:
                    if slot['slotId'] in used_dict:
                        used_dict[slot['slotId']].append(
                            interface['interfaceId'])
                    else:
                        used_dict[slot['slotId']] = [interface['interfaceId']]
        rs['data'] = used_dict
    return rs


def input_write_resolution(
        device_id,
        input_id,
        refresh,
        width,
        height,
        conn_capacity):
    """
    设置输入源分辨率
    :param device_id: 设备Id
    :param input_id: 输入源Id
    :param refresh: 刷新率
    :param width: 宽度
    :param height: 高度
    :param conn_capacity: 连接容量
    """
    if refresh <= 0:
        return get_result_model(ResInfo.Input_RefreshRate_Err)
    rs = get_result_model(ResInfo.Input_Write_Timing_List_Error)
    max_band = Max_Band_Capacity_Map['input'].get(conn_capacity, 0)
    if max_band > 0:
        timing = outputResolution(width, height, refresh, max_band)
        if timing:
            timing['inputId'] = input_id
            timing['deviceId'] = device_id
            rs = open_input_middle.input_write_timing_middle(
                device_id, input_id, timing)
    return rs


def input_group_detail(device_id, group_id):
    return  open_input_middle.input_group_detail_middle(device_id, group_id)


def input_source_create(device_id, data):
    """
    创建轮巡源列表
    """
    source_type = data.get('sourceType')
    if source_type != SourceType.InputSourcePull:
        return get_result_model(ResInfo.Input_Support_Err)
    return open_input_middle.input_source_create(device_id, data)


def input_source_list(device_id):
    """
    创建轮巡源列表
    """
    return open_input_middle.input_source_list(device_id)

def input_source_delete(device_id, data):
    """
    删除轮巡源列表
    """
    return open_input_middle.input_source_delete(device_id, data)

def input_source_update(device_id, data):
    """
    更新轮巡源列表
    """
    return open_input_middle.input_source_update(device_id, data)

def input_source_enable(device_id, data):
    """
    开启,关闭轮巡源列表
    """
    return open_input_middle.input_source_enable(device_id, data)
















