"""
Created on 2021年9月14日
@author: 程声清
output服务类
"""
from copy import deepcopy

from H_9U.open.middle import open_device_middle, open_screen_middle, open_input_middle, open_output_middle
from H_9U.models.sysconst import CardCategory, InterfaceType, FunctionType, Max_Band_Capacity_Map, ModelId
from H_9U.models.result import get_result_model, ResInfo

from H_9U.util.log import logger
from H_9U.util.outputtiming import outputResolution


def output_read_detail(device_id, output_id):
    """
    读取输出详情并加入缓存
    :param device_id: 设备id
    :param output_id: 输出id
    :return: 输出详情
    """
    return open_output_middle.output_original_detail_middle(
        device_id, output_id)


def output_read_list(device_id):
    """
    读取列表
    :param device_id: 设备Id
    :return: 结果对象
    """
    return open_output_middle.output_read_list_middle(device_id)


def output_write_timing(device_id, output_id, params):
    """
    设置Timing.
    :param device_id: 设备Id.
    :param output_id: 输出Id.
    :param params:
    :return: 结果对象
    """
    if 'outputIdList' not in params:
        params['outputIdList'] = [output_id]
    rs = output_timing_is_support(device_id, output_id)
    if rs['status'] != 0:
        return rs
    screen_rs = _get_screen_list(device_id)
    if screen_rs['status'] != 0:
        return get_result_model(ResInfo.Middle_Data_Err)
    for screen in [s for s in screen_rs['data']
                   if s['presetPoll']['enable'] == 1]:
        if output_id in screen['outputMode']['screenInterfaces']:
            return get_result_model(ResInfo.Screen_Is_Preset_Polling)
        from H_9U.service.output import output_svc
        rs = output_svc.write_timing(device_id, [output_id], params)
    return rs


def output_write_4k_timing(device_id, output_id, params):
    """
    设置4kTiming.
    :param device_id: 设备Id.
    :param output_id: 输出Id.
    :param params:
    :return: 结果对象
    """
    rs = output_4k_is_support(device_id, output_id)
    if rs['status'] != 0:
        return rs
    screen_rs = _get_screen_list(device_id)
    if screen_rs['status'] != 0:
        return get_result_model(ResInfo.Middle_Data_Err)
    rs = open_output_middle.write_4k_timing_middle(device_id, output_id, params)
    return rs


def output_write_payload(device_id, output_id, data):
    """
    二合一卡设置带载面积
    :param device_id: 设备Id.
    :param output_id: 输出id.
    :param data: 输出列表.
    :return: 结果对象
    """
    rs = output_payload_is_support(device_id, output_id)
    if rs['status'] != 0:
        return rs
    screen_rs = _get_screen_list(device_id)
    output_list = []
    if screen_rs['status'] != 0:
        return get_result_model(ResInfo.Middle_Data_Err)
    output_list.append(output_id)
    rs = open_output_middle.output_write_payload_middle(
        device_id, output_list, data)
    return rs


def output_write_image_quality(device_id, output_id, params):
    """
    设置画质
    :param device_id: 设备Id
    :param output_id:  输出Id
    :param params:
    :return:  结果对象
    """
    rs = open_output_middle.output_write_image_quality_middle(
        device_id, output_id, params)
    if rs["status"] == 0:
        logger.debug("设置画质-成功! CODE:%i, params:%s" % (rs["status"], params))
    else:
        logger.error("设置画质-失败! CODE:%i, params:%s" % (rs["status"], params))

    return rs


def output_write_gamma(device_id, output_id, params):
    """
    output设置gamma数据
    :param device_id: 设备id
    :param output_id: output id
    :param params:  gamma数据
    :return: 结果对象
    """
    rs = open_output_middle.output_write_gamma_middle(
        device_id, output_id, params)
    if rs["status"] == 0:
        logger.debug("设置Gamma-成功! CODE:%i, params:%s" % (rs["status"], params))
    else:
        logger.error("设置Gamma-失败! CODE:%i, params:%s" % (rs["status"], params))

    return rs


def output_write_fixed_position(device_id, params):
    """
    设置输出定位
    :param device_id: 设备Id
    :param params: 输出定位参数
    :return: 结果对象
    """
    return open_output_middle.output_write_fixed_position_middle(
        device_id, params)


def _write_timing(device_id, output_id, params):
    """
    设置Timing.
    :param device_id: device_id
    :param output_id: output_id
    :param params: params
    :return: 结果对象
    """
    try:
        # 处理刷新率数据
        # params['refreshRate'] = int(params['refreshRate'] * 100)
        data = []
        time_ming = deepcopy(params)
        time_ming['outputId'] = output_id
        rs = open_output_middle.write_timing_middle(
                device_id, output_id, time_ming)
        if rs["status"] != 0:
            logger.debug("设置Timing-失败! CODE:%i, params:%s" % (rs["status"], output_id))
            success = 0
        else:
            success = 1
        data.append({
            'outputId': output_id,
            'success': success
        })
        rs['data'] = data
    except Exception as e:
        logger.error(e)
        rs = get_result_model(ResInfo.Output_Write_Timing_List_Error)
    return rs


# TODO: screen middle 中提供方法，找到当前正在轮巡的屏幕和output
def _get_screen_list(device_id):
    """
    获取屏幕列表数据
    :param device_id: 设备id.
    :return: 结果对象
    """
    rs = open_screen_middle.read_list_middle(device_id)
    if rs["status"] == 0:
        if rs['data']:
            screens = rs['data'].get("screens", None)
            detail_list = []
            if screens:
                # mode_rs = open_device_middle.device_ipc_mode_read_middle(
                #     device_id)
                # ipc_mode = mode_rs['data']['ipcMode']
                # ipc_list = montage_list = None
                # if ipc_mode == 0:
                #     ipc_list = _input_ipc_source_list(device_id)['data']
                # else:
                #     montage_list = _input_ipc_montage_list(device_id)['data']
                # input_list = _input_read_all_list(device_id)['data']
                for screen in screens:
                    screen_id = screen["screenId"]
                    detail_rs = open_screen_middle.screen_read_detail_middle(
                        device_id, screen_id)
                    if detail_rs['status'] == 0 and detail_rs['data']:
                        detail_list.append(detail_rs['data'])
                    else:
                        logger.error(
                            "获得screenId:%i,screenId:%i屏幕详情-失败!结果中将缺少该screen信息." %
                            (device_id, screen_id))

            rs['data'] = detail_list
        else:
            rs['data'] = []
    return rs


def _input_read_all_list(device_id):
    """
    获取所有输入列表
    :param device_id: 设备id.
    :return: 结果对象
    """
    rs = open_input_middle.input_list_middle(device_id)
    device_rs = open_device_middle.device_detail_middle(device_id)
    used_rs = open_input_middle.get_used_inputs_dict_middle(device_id)
    if rs["status"] == 0 and device_rs['status'] == 0 and used_rs['status'] == 0:
        input_list = []
        input_dict = used_rs['data']
        if rs['data']['inputs']:
            input_list = []

            # 循环输入信息
            for item in [x for x in rs['data']['inputs']]:
                # 获取输入详情
                try:
                    detail = open_input_middle.read_detail_middle(
                        device_id, item['inputId'])
                    if detail['status'] == 0:
                        detail['data']['iSignal'] = 1 if detail['data']['resolution']['refresh'] > 0 else 0
                        detail['data']['isUsed'] = 1 if detail['data']['slotId'] in input_dict and detail[
                            'data']['interfaceId'] in input_dict[detail['data']['slotId']] else 0
                        detail['data']['isEDIDSetting'] = InterfaceType.allow_edid_setting(
                            detail['data']['interfaceType'])
                        detail['data']['modelId'] = [x for x in device_rs['data']['slotList']
                                                     if x['slotId'] == detail['data']['slotId']][0]['modelId']
                        input_list.append(detail['data'])
                except Exception as e:
                    logger.error("输入源列表异常：" + str(e))
        rs['data'] = input_list

        k4_cards = [x for x in device_rs['data']['slotList']
                    if x['cardCategory'] == CardCategory.K4_HDMI_DP_Card]
        k4_card_slots = [s['slotId'] for s in k4_cards]
        # 含有4k卡
        if k4_cards:
            for i in input_list:
                try:
                    if i['online'] == 1 and i['slotId'] in k4_card_slots:
                        slot = [
                            x for x in k4_cards if x['slotId'] == i['slotId']][0]
                        functions = [x for x in slot['interfaces']
                                     if x['functionType'] != FunctionType.Disable]
                        func_type = functions[0]['functionType'] if functions else FunctionType.Disable
                        if func_type == FunctionType.HDMI20:
                            i['interfaceType'] = InterfaceType.HDMI20
                        elif func_type == FunctionType.DP12:
                            i['interfaceType'] = InterfaceType.DP12
                        i['isUsed'] = 1 if i['slotId'] in input_dict else 0
                except Exception as e:
                    logger.error("4k输入源列表异常：" + str(e))
    else:
        rs = get_result_model(ResInfo.Middle_Data_Err)

    return rs


def _input_ipc_montage_list(device_id):
    """
    获取所有ipc拼接输入列表
    :param device_id: 设备id.
    :return: 结果对象
    """
    slots = _ipc_card_list(device_id)
    mode_list = []
    for slot_id in slots:
        rs = _ipc_montage_detail(device_id, slot_id)
        if rs['status'] == 0 and rs['data'] and rs['data']['sourcePosList']:
            mode_list.append(rs['data'])
    rs = get_result_model()
    rs['data'] = mode_list
    return rs


def _ipc_montage_detail(device_id, slot_id):
    """
     读取ipc拼接详情
     :param device_id: 设备id
     :param slot_id: slotId
     :return: 拼接详情
     """
    rs = open_input_middle.read_ipc_source_size_pos_middle(device_id, slot_id)
    if rs['status'] == 0 and rs['data']['sourcePosList']:
        ipc_input_id = -1
        is_signal = 0

        for source in rs['data']['sourcePosList']:
            name = ''
            source_detail = open_input_middle.read_ipc_source_detail_middle(
                device_id, slot_id, source['ipcSourceId'])
            if source_detail['status'] == 0 and source_detail['data']:
                name = source_detail['data']['name']
                if ipc_input_id == -1:
                    ipc_input_id = source_detail['data']['inputId']
                if is_signal == 0:
                    is_signal = source_detail['data']['iSignal']
            source['ipcSourceName'] = name

        rs['data']['ipcInputId'] = ipc_input_id
        rs['data']['iSignal'] = is_signal
        crop_rs = open_input_middle.ipc_crop_list_middle(
            1, device_id, slot_id, 0)
        crops = crop_rs['data']['crops'] if rs['status'] == 0 else []
        rs['data']['crops'] = crops
    return rs


def _input_ipc_source_list(device_id):
    """
    获取所有ipc拼接输入源列表
    :param device_id: 设备id.
    :return: 结果对象
    """
    ipc_cards = _ipc_card_list(device_id)
    ipc_list = []
    if ipc_cards:
        for slot_id in ipc_cards:
            rs = _input_ipc_source_detail_list(device_id, slot_id)
            if rs['status'] == 0:
                ipc_list.append(rs['data'])

    rs = get_result_model()
    rs['data'] = {'deviceId': device_id, 'ipcList': ipc_list}
    return rs


def _input_ipc_source_detail_list(device_id, slot_id):
    """
    读取ipc源列表详情
    :param device_id: 设备id
    :param slot_id: slot id
    :return: ipc列表
    """
    rs = open_input_middle.input_read_ipc_source_list_middle(
        device_id, slot_id)
    if rs['status'] == 0:
        source_list = []
        for source in rs['data']['ipcSourceList']:
            source_id = source['ipcSourceId']
            source_info = _ipc_source_detail(0, device_id, slot_id, source_id)
            if source_info['status'] == 0 and source_info['data']:
                source_list.append(source_info['data'])
        rs['data'] = {'slotId': slot_id, 'ipcSourceList': source_list}

    return rs


def _ipc_source_detail(ipc_mode, device_id, slot_id, source_id):
    """
    获取ipc源详情
    :param ipc_mode: ipc_mode
    :param device_id: 设备id.
    :param slot_id: 槽位id.
    :param source_id: 源id.
    :return: 结果对象
    """
    rs = open_input_middle.read_ipc_source_detail_middle(
        device_id, slot_id, source_id)
    if rs['status'] == 0 and rs['data']:
        rs['data']['ipcInputId'] = rs['data']['inputId']
        crop_rs = _ipc_crop_list(ipc_mode, device_id, slot_id, source_id)
        crops = crop_rs['data']['crops'] if crop_rs['status'] == 0 else []
        rs['data']['crops'] = crops
    return rs


def _ipc_crop_list(ipc_mode, device_id, slot_id, source_id):
    """
    获取ipc截取列表
    :param ipc_mode: ipc_mode.
    :param device_id: 设备id.
    :param slot_id: 槽位id.
    :param source_id: 源id.
    :return: 结果对象
    """
    rs = open_input_middle.input_read_ipc_crop_list_middle(
        ipc_mode, device_id, slot_id, source_id)
    if rs['status'] == 0 and rs['data']:
        crops = []
        for c in rs['data']['crops']:
            crop_id = c['cropId']
            crop = _ipc_crop_detail(
                ipc_mode, device_id, slot_id, source_id, crop_id)
            if crop['status'] == 0 and crop['data']:
                crops.append(crop['data'])
        rs['data']['crops'] = crops
    return rs


def _ipc_crop_detail(ipc_mode, device_id, slot_id, source_id, crop_id):
    """
    获取ipc截取列表
    :param ipc_mode: ipc_mode.
    :param device_id: 设备id.
    :param slot_id: 槽位id.
    :param source_id: 源id.
    :param crop_id: 截取id.
    :return: 结果对象
    """
    return open_input_middle.input_read_ipc_crop_detail_middle(
        ipc_mode, device_id, slot_id, source_id, crop_id)


def _ipc_card_list(device_id):
    """
    获取ipc卡列表
    :param device_id:
    :return: ipc卡列表
    """
    rs = open_device_middle.device_detail_read_middle(device_id)
    if rs['status'] == 0:
        if rs['data']:
            ipc_list = [x['slotId'] for x in rs['data']['slotList']
                        if x['cardCategory'] == CardCategory.IPCCard]
            rs['data'] = ipc_list
        else:
            rs['data'] = []
    return rs


def output_rename(device_id, output_id, data):
    """
    设置基本信息
    :param device_id: 设备Id
    :param output_id:  输出Id
    :param data: 数据
    :return: 结果对象
    """
    rs = open_output_middle.output_rename_middle(device_id, output_id, data)
    if rs["status"] != 0:
        logger.error("设置重命名-失败! CODE:%i, params:%s" % (rs["status"], data))
    return rs


def output_write_resolution(device_id, output_id, refresh, width, height):
    """
    设置输出分辨率
    :param device_id: 设备Id
    :param output_id: output_id
    :param refresh: 刷新率
    :param width: 款
    :param height: 高
    """
    # rs = open_output_middle.output_original_detail_middle(device_id, output_id)
    # if rs['status'] == 0:
    #     model_id = rs['data']['modelId']
    #     if model_id in ModelId.Two_In_One_Card_List or model_id == ModelId.Output_2_HDMI20_Card:
    #         rs = get_result_model(ResInfo.Output_Timing_Err)
    rs = output_timing_is_support(device_id, output_id)
    if rs['status'] == 0 and rs['data']:
        conn = rs['data']['connectCapacity']
        max_band = Max_Band_Capacity_Map['output'].get(conn, 0)
        if max_band > 0:
            timing = outputResolution(width, height, refresh, max_band)
            if timing:
                timing['outputId'] = output_id
                timing['deviceId'] = device_id
                rs = open_output_middle.write_timing_middle(device_id, output_id, timing)
    return rs


def output_timing_is_support(device_id, output_id):
    """
    是否支持输出timing (二合一和4k不支持)
    :param device_id: 设备Id
    :param output_id:  输出Id
    :return: 结果对象
    """
    rs = open_output_middle.output_original_detail_middle(
        device_id, output_id)
    if rs['status'] == 0 and rs['data']:
        model_id = rs['data']['modelId']
        if model_id == ModelId.Output_Two_In_One_Card_20 or model_id == ModelId.Output_2_HDMI20_Card:
            rs = get_result_model(ResInfo.Output_Support_Err)
        else:
            rs = get_result_model()
    else:
        rs = get_result_model(ResInfo.Output_Read_Error)
    return rs


def output_payload_is_support(device_id, output_id):
    """
    二合一卡设置带载面积(二合一专用)
    :param device_id: 设备Id
    :param output_id: 输出Id
    :return: 结果对象
    """
    rs = open_output_middle.output_original_detail_middle(
        device_id, output_id)
    if rs['status'] == 0 and rs['data']:
        model_id = rs['data']['modelId']
        if model_id != ModelId.Output_Two_In_One_Card_20:
            rs = get_result_model(ResInfo.Output_Support_Err)
        else:
            rs = get_result_model()
    else:
        rs = get_result_model(ResInfo.Output_Read_Error)
    return rs


def output_4k_is_support(device_id, output_id):
    """
    设置指定输出的Timing信息(4k专用)
    :param device_id: 设备Id
    :param output_id: 输出Id
    :return: 结果对象
    """
    rs = open_output_middle.output_original_detail_middle(
        device_id, output_id)
    if rs['status'] == 0 and rs['data']:
        model_id = rs['data']['modelId']
        if model_id != ModelId.Output_2_HDMI20_Card:
            rs = get_result_model(ResInfo.Output_Support_Err)
        else:
            rs = get_result_model()
    else:
        rs = get_result_model(ResInfo.Output_Read_Error)
    return rs
