"""
Created on 2019年6月28日
@author: 刘益宗
bkg接口.
返回值统一为 {data:JSON结构体, msg:"", status:0}
"""
import time

from H_9U.models.result import get_result_model, ResInfo
from H_9U.models.sysconst import UploadFileType
from H_9U.service.common import upload_file
from H_9U.util.fileopt import remove_file
from H_9U.mao.bkg_mao import bkg_api
from H_9U.service.middlewarefilepath import bkg_manager


class BkgSvc:

    def read_list(self, device_id):
        """
        读取bkg 列表
        :param device_id: 设备Id
        :return:  列表数据，不含详细信息
        """
        return bkg_api.read_list(device_id)

    def read_all_list(self, device_id):
        """
        读取bkg详细信息列表
        :param device_id: 设备Id
        :return: 参考前端接口文档
        """
        rs = self.read_list(device_id)
        if rs['status'] == 0 and rs['data']:
            for bkg in rs['data']['bkgs']:
                self._bkg_data_build(device_id, bkg)
        rs['data'] = rs['data']['bkgs']
        return rs

    def _bkg_data_build(self, device_id, bkg):
        """
        读取bkg详细信息列表
        :param device_id: 设备Id
        :return: 参考前端接口文档
        """
        img_file = bkg['data']['image']['fileName']
        bkg['data']['image']['imgUrl'] = ''
        bkg['thumbnailData'] = {
                "deviceId": device_id,
                "bkgId": bkg['bkgId'],
                "image": {
                    "fileName": '',
                    "imgUrl":  '',
                    "fileLength":  '',
                    "hashSum": 0x55FF,
                    "width": 0,
                    "height": 0}}
        #  为bkg图片补充缩略图等信息
        if img_file:
            # bkg绝对路径，url； 缩略图绝对路径，url
            bkg_path, bkg_url, thumbnail_path, thumbnail_url = bkg_manager.get_bkg_path_info(img_file)
            bkg['data']['image']['imgUrl'] = bkg_url
            # 缩略图数据
            bkg['thumbnailData'] = {
                "deviceId": device_id,
                "bkgId": bkg['bkgId'],
                "image": {
                    "fileName": thumbnail_path,
                    "imgUrl": thumbnail_url,
                    "fileLength": 0,
                    "hashSum": 0x55FF,
                    "width": 0,
                    "height": 0}}

    def bkg_general_write(self, device_id, bkg_id, params):
        """
        设置bkg 名称
        :param params: 名称数据
        :return: 结果对象
        """
        # 已存在则不允许修改名称
        name = params.get('name')
        if name and self._check_bkg_name_exist(device_id, bkg_id, name):
            return get_result_model(ResInfo.BKG_Name_Exists)

        return bkg_api.write_general(device_id, bkg_id, params)

    def bkg_read_detail(self, device_id, bkg_id):
        """
        读取bkg详情
        :param device_id: 设备id
        :param bkg_id:  bkgid
        :return: 参考前端接口文档
        """
        rs = bkg_api.read_detail(device_id, bkg_id)
        if rs['status'] == 0 and rs['data']:
            img_file = rs['data']['data']['image']['fileName']

            rs['data']['data']['image']['imgUrl'] = ''
            rs['data']['thumbnailData'] = {
                "deviceId": device_id,
                "bkgId": bkg_id,
                "image": {
                    "fileName": '',
                    "imgUrl":  '',
                    "fileLength":  '',
                    "hashSum": 0x55FF,
                    "width": 0,
                    "height": 0}}
            #  为bkg图片补充缩略图等信息
            if img_file:
                # bkg绝对路径，url； 缩略图绝对路径，url
                bkg_path, bkg_url, thumbnail_path, thumbnail_url = bkg_manager.get_bkg_path_info(img_file)
                rs['data']['data']['image']['imgUrl'] = bkg_url
                # 缩略图数据
                rs['data']['thumbnailData'] = {
                    "deviceId": device_id,
                    "bkgId": bkg_id,
                    "image": {
                        "fileName": thumbnail_path,
                        "imgUrl": thumbnail_url,
                        "fileLength": 0,
                        "hashSum": 0x55FF,
                        "width": 0,
                        "height": 0}}
        return rs

    def bkg_create(self, device_id, name):
        """
        创建BKG
        :param device_id: 设备Id
        :param name:  BKG名称
        :return: BKGId
        """
        # 上传图片
        up_rs = upload_file(UploadFileType.BKG, name)
        if not up_rs["bln"]:
            return get_result_model(ResInfo.Upload_File_Fail)

        # 上传成功后：先判断资源是否够用，然后再将图片转成dta文件,最后再在设备上创建BKG
        path_nm = up_rs["pathNm"]
        # 获得图片尺寸，判断资源是否够用
        if not bkg_manager.valid_bkg_file_size(path_nm):
            return get_result_model(ResInfo.BKG_Max_Count_Limit_Error)
        # 发送中间件数据
        data = {
            "deviceId": device_id,
            "name": name,
            "image": {
                "fileName": path_nm,
                "fileLength": 0,
                "hashSum": 0x55FF,
                "width": 0,
                "height": 0,
                "createTime": str(time.time()),
            }}
        rs = bkg_api.write_create(device_id, data)
        # 失败删除源文件
        if rs['status'] != 0:
            remove_file(path_nm)
        
        return rs

    def bkg_delete(self, device_id, bkg_id):
        """
        删除bkg
        :param device_id: 设备Id
        :param bkg_id: bkg Id
        :return:  结果对象
        """
        # 读取详细信息
        rs = self.bkg_read_detail(device_id, bkg_id)
        # 删除中间件数据
        del_rs = bkg_api.write_delete(device_id, bkg_id, {"deviceId": device_id, "bkgId": bkg_id})
        # 原始图片中间件负责移除
        if del_rs['status'] == 0 and rs['status'] == 0:
            remove_file(rs['data']['thumbnailData']['image']['fileName'])
        return del_rs

    def bkg_delete_batch(self, device_id, model, bkg_ids, bkg_used_list):
        """
        批量删除bkg
        bkg列表与要删除的bkgid列表取交集
        删除未被屏幕使用的bkg
        model： 0全部删除，1 部分删除
        :return: 结果数据
        """
        # 读取bkg列表，无BKG返回成功
        rs = self.read_list(device_id)
        if rs['status'] != 0:
            return get_result_model(ResInfo.BKG_Query_Error)
        bkgs = rs['data']['bkgs']
        if not bkgs:
            return get_result_model()
        # 过滤要删除数据
        if model == 1:
            bkgs = [x['bkgId '] for x in bkgs if x['bkgId '] in bkg_ids]
        rs_data = []
        partial = False
        # 循环阐述
        for bkg in bkgs:
            if bkg['bkgId'] not in bkg_used_list:
                del_rs = self.bkg_delete(device_id, bkg['bkgId'])
                status = del_rs['status']
            else:
                status = 1
            rs_data.append({"deviceId": device_id, "bkgId": bkg['bkgId'], "status": status})
            # 是否有不成功删除
            if status != 0:
                partial = True

        if partial and model == 0:  # 部分成功
            rs = get_result_model(ResInfo.BKG_Batch_Delete_Partial_Success)
            rs['data'] = rs_data
        else:  # 全部成功
            rs = get_result_model()
            rs['data'] = rs_data
        return rs

    def _check_bkg_name_exist(self, device_id, bkg_id, name):
        """
        查看bkg名称是否已存在
        :param device_id: 设备id
        :param name: bkg
        :return: True, False
        """
        exist = True
        rs = self.read_list(device_id)
        if rs['status'] == 0 and rs['data']:
            if not rs['data']['bkgs']:
                exist = False
            bkgs = [x for x in rs['data']['bkgs'] if x['name'] == name]
            if len(bkgs) <= 0:
                exist = False
            if len([x for x in bkgs if x['bkgId'] != bkg_id]) <= 0:
                exist = False

        return exist


bkg_svc = BkgSvc()
