from H_9U.models.result import get_result_model
from H_9U.mao.impl import layer
from H_9U.util.cache import cacher
from H_9U.models.cachekey import CacheKey, CacheDuration


class LayerApi:

    def read_list(self, device_id, screen_id):
        """
        读取图层列表并加入缓存
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :return: 图层列表
        """
        return layer.layer_list_read(device_id, screen_id)

    def read_detail(self, device_id, screen_id, layer_id):
        """
        读取图层详情并加入缓存
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param layer_id: 图层id
        :return: 图层详情
        """
        key = CacheKey.layer_detail(device_id, screen_id, layer_id)
        return cacher.LayerCache.try_get_value(key, CacheDuration.Default, layer.layer_full_detail_read, device_id, screen_id, layer_id)

    def write_create(self, device_id, screen_id, data):
        """
        创建图层
        清除图层列表缓存
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param data: 数据对象
        :return: 结果对象
        """
        rs = layer.layer_create_write(device_id, screen_id, data)
        if rs['status'] == 0:
            cacher.clear_all()
        return rs

    def write_create_batch(self, device_id, screen_id, layers):
        """
        批量创建图层
        清除图层列表缓存
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param layers: 图层列表
        :return: 结果对选哪个
        """
        rs = get_result_model()
        if layers:
            for x in layers:
                x['deviceId'] = device_id
                x['screenId'] = screen_id
            rs = layer.layer_create_write(device_id, screen_id, layers)
            if rs['status'] == 0:
                cacher.LayerCache.delete(CacheKey.layer_list(device_id, screen_id))
                cacher.DeviceCache.clear()
        return rs

    def write_delete(self, device_id, screen_id, layer_id, data):
        """
        删除图层
        清除全部缓存
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param layer_id: 图层id
        :param data:
        :return: 结果对象
        """
        rs = layer.layer_delete_write(device_id, screen_id, layer_id, data)
        if rs['status'] == 0:
            cacher.clear_all()
        return rs

    def write_general(self, device_id, screen_id, layer_id, data):
        """
        设置图层基本信息
        清除图层列表缓存
        清除图层详情缓存
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param layer_id: 图层id
        :param data: 基本信息
        :return: 结果对象
        """
        rs = layer.layer_general_write(device_id, screen_id, layer_id, data)
        if rs['status'] == 0:
            cacher.LayerCache.delete(CacheKey.layer_list(device_id, screen_id))
            cacher.LayerCache.delete(CacheKey.layer_detail(device_id, screen_id, layer_id))
        return rs

    def read_general(self, device_id, screen_id, layer_id):
        """
        读取基本信息
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param layer_id: 图层id
        :return: 结果对象
        """
        return layer.layer_general_read(device_id, screen_id, layer_id)

    def write_layout_window(self, device_id, screen_id, layer_id, data):
        """
        设置图层布局信息
        清除图层详情缓存
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param layer_id:  图层id
        :param data: 布局信息
        :return: 结果对象
        """
        rs = layer.layer_layout_window_write(device_id, screen_id, layer_id, data)
        if rs['status'] == 0:
            cacher.LayerCache.delete(CacheKey.layer_list(device_id, screen_id))
            cacher.LayerCache.delete(CacheKey.layer_detail(device_id, screen_id, layer_id))
        return rs

    def read_layout_window(self, device_id, screen_id, layer_id):
        """
        读取图层布局信息
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param layer_id: 图层id
        :return: 结果对象
        """
        return layer.layer_layout_window_read(device_id, screen_id, layer_id)

    def write_source(self, device_id, screen_id, layer_id, data):
        """
        图层切源
        清除全部缓存
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param layer_id: 图层id
        :param data: 源信息
        :return: 结果对象
        """
        rs = layer.layer_source_write(device_id, screen_id, layer_id, data)
        if rs['status'] == 0:
            cacher.clear_all()
        return rs

    def read_source(self, device_id, screen_id, layer_id):
        """
        读取图层源信息
        :param device_id: 设备id
        :param screen_id: 屏幕信息
        :param layer_id: 图层信息
        :return: 原信息
        """
        return layer.layer_source_read(device_id, screen_id, layer_id)

    def read_zorder(self, device_id, screen_id):
        """
        读取图层z序
        :param device_id:设备id
        :param screen_id: 屏幕id
        :return: 结果对象
        """
        return layer.layer_z_order_read(device_id, screen_id)

    def read_detail_list(self, device_id, screen_id):
        """
        读取屏幕图层图层详情
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :return: 图层详细信息列表
        """
        return layer.layer_full_detail_list(device_id, screen_id)

    def write_zorder(self, device_id, screen_id, data):
        """
        设置z序
        清空缓存
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param data: z序
        :return: 结果对象
        """
        rs = layer.layer_z_order_write(device_id, screen_id, data)
        if rs['status'] == 0:
            cacher.clear_all()
        return rs

    def write_freeze(self, device_id, screen_id, layer_id, data):
        """
        冻结图层
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param data: 冻结信息
        :return: 结果对象
        """
        rs = layer.layer_freeze(device_id, screen_id, layer_id, data)
        if rs['status'] == 0:
            cacher.LayerCache.delete(CacheKey.layer_list(device_id, screen_id))
            cacher.LayerCache.delete(CacheKey.layer_detail(device_id, screen_id, layer_id))
        return rs

    def write_screen_layer_layout(self, device_id, screen_id, data):
        """
        屏幕图层自动布局（模板开窗）
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :param data: 图层数据
        :return: 结果对象
        """
        rs = layer.screen_layer_auto_play(device_id, screen_id, data)
        if rs['status'] == 0:
            cacher.LayerCache.clear()
            cacher.DeviceCache.clear()
        return rs

    def write_lock(self, device_id, screen_id, layer_id, data):
        rs = layer.layer_lock(device_id, screen_id, layer_id, data)
        if rs['status'] == 0:
            cacher.LayerCache.delete(CacheKey.layer_list(device_id, screen_id))
            cacher.LayerCache.delete(CacheKey.layer_detail(device_id, screen_id, layer_id))
        return rs

    def write_audio_status(self, device_id, screen_id, layer_id, data):
        """
        屏幕图层音频设置
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :param layer_id: 图层Id
        :param data: 图层数据
        :return: 结果对象
        """
        rs = layer.write_audio_status(device_id, screen_id, layer_id, data)
        if rs['status'] == 0:
            cacher.LayerCache.delete(CacheKey.layer_list(device_id, screen_id))
            cacher.LayerCache.delete(CacheKey.layer_detail(device_id, screen_id, layer_id))
            cacher.ScreenCache.delete(CacheKey.screen_list(device_id))
            cacher.ScreenCache.delete(CacheKey.screen_detail(device_id, screen_id))
        return rs

    def write_layer_lock_ratio(self, device_id, screen_id, layer_id, data):
        """
        设置屏幕图层纵横比
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :param layer_id: 图层Id
        :param data: 图层数据
        :return: 结果对象
        """
        rs = layer.layer_lock_ratio(device_id, screen_id, layer_id, data)
        if rs['status'] == 0:
            cacher.LayerCache.delete(CacheKey.layer_detail(device_id, screen_id, layer_id))
        return rs

    def read_detail_no_cache(self, device_id, screen_id, layer_id):
        """
        读取图层详情不走缓存
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param layer_id: 图层id
        :return: 图层详情
        """
        return layer.layer_full_detail_read(device_id, screen_id, layer_id)

    def write_layer_stream_rule(self, device_id, data):
        """
        设置ipc源图层使用主辅码流
        :param device_id: 设备id
        :param data: 图层id
        :return: 操作结果
        """
        return layer.write_layer_stream_rule(device_id, data)

    def write_layer_detail_batch(self, device_id, data):
        """
        批量获取图层详情
        :return: 操作结果
        """
        return layer.multi_write_layer_detail(device_id, data)



layer_api = LayerApi()
