"""
Created on 2019年6月28日
@author: 刘益宗
设备服务.
对应3.5代接口协议中的 Device
返回值统一为 {data:JSON结构体, msg:"", status:0}
"""
import json
import shutil
import time
import os
import traceback
import uuid
from copy import deepcopy

from flask import g
from flask.globals import request

from H_9U.dao.auth_code import auth_code_dao
from H_9U.dao.device_info_dao import deviceInfodao
from H_9U.models.result import get_result_model, ResInfo
from H_9U.models.sysconst import CardType, UpgradeItem, UpgradeFileModel, UpgradeConfig, CardCategory, \
    CardName, UpgradeCardType, UpgradeCardCode, UpgradeH9AuxReplaceDataConfig, \
    Upgrade2HDMI20OutputDataConfig, SN_Prefix, SafeConfig, DeviceType, HttpConfigType, UpgradeFuncCode
from H_9U.service.ipc import ipc_svc
from H_9U.service.preset import preset_svc
from H_9U.util.log import logger
from H_9U.util.extract import extract, pd_extract
from H_9U.util.extract import extract
from H_9U.service import common
from H_9U.models.sysconst import UploadFileType
from threading import Timer, Lock
from H_9U.conf.syssettings import SysSettings
from H_9U.mao.device_mao import device_api
from H_9U.util.fileopt import remove_file
from H_9U.service.common import upload_file
from H_9U.api.websender import push_data_org
from H_9U.models.syncdataname import SyncDataName
from H_9U.models.sysconst import ModelId, FunctionType
from H_9U.service.middlewarefilepath import nomark_manager
from H_9U.util.common import str_hash_code, check_password, password_verifier
from H_9U.util.nginx_ctrl import set_nginx_and_reload


class DeviceSvc:
    """
    设备服务类
    """
    def __init__(self):
        self.is_upgrade = False  # 当前是否正在升级 标志
        self.is_check = False  # 是否在自检 标志
        self.upgrade_time = None   # 升级开始时间
        self.upgrade_all_success = True
        self.is_setting_file_operating = False
        self.setting_lock = Lock()
        self.is_upgrade_send = False
        self.upgrade_retry_times = 2
        self.upgrade_sections = None
        self.upgrade_from_usb = False

    def set_upgrade_from_usb(self):
        self.upgrade_from_usb = True

    def read_upgrade_status(self):
        """
        读取当前升级状态
        :return:
        """
        return self.is_upgrade

    def read_self_check_status(self):
        return self.is_check

    def read_setting_file_operate_status(self):
        return self.is_setting_file_operating

    def begin_upgrade(self):
        """
        开始升级
        升级标志设为True
        记录升级开始时间
        :return: 成功失败
        """
        success = False
        if not self.is_upgrade:
            self.upgrade_time = time.monotonic()
            self.is_upgrade = True
            self.upgrade_sections = None
            success = True
        return success

    def end_upgrade(self):
        """
        结束升级
        升级标识设为False
        升级开始时间设备None
        :return:
        """
        if self.is_upgrade:
            self.upgrade_time = None
            self.is_upgrade = False

    def begin_check(self):
        self.is_check = True

    def end_check(self):
        self.is_check = False

    def begin_settings(self):
        success = False
        with self.setting_lock:
            if not self.is_setting_file_operating:
                self.is_setting_file_operating = True
                success = True
        return success

    def end_settings(self):
        self.is_setting_file_operating = False

    def device_detail_read(self, device_id):
        """
        读取设备详细信息
        :param device_id:设备id
        :return: 参考协议文档：0x00 - key: deviceDetail
        """
        return device_api.read_detail(device_id)

    def device_detail_without_cache(self, device_id):
        return device_api.read_detail_without_cache(device_id)

    def device_general_read(self, device_id):
        """
        读取设备基本信息
        :param device_id:设备Id
        :return: 参考协议文档：0x01 - key: deviceGeneral
        """
        return device_api.read_general(device_id)

    def device_general_write(self, device_id, data):
        """
        写入设备基本信息
        :param device_id:设备Id
        :param data: 参考协议文档：0x01 - key: deviceGeneral
        :return: 参考协议文档：0x01 - key: deviceGeneral
        """
        return device_api.write_general(device_id, data)

    def device_slot_read(self, device_id, slot_id):
        """
        读取设备槽信息
        :param device_id:设备Id
        :param slot_id: 卡槽Id
        :return: 参考前端接口协议文档：readSlot
        """
        rs = device_api.read_slot(device_id, slot_id)
        if rs['status'] == 0:
            # 设置默认值
            rs['data']['sn'] = ''
            rs['data']['hardwareVersion'] = ''
            rs['data']['mcuVersion'] = ''
            rs['data']['fpgaVersion'] = ''
        return rs

    def device_slot_write(self, device_id, slot_id, data):
        """
        设置slot参数
        :param device_id: 设备Id
        :param slot_id:  slotId
        :param data:  参数对象
        :return: 结果对象
        """
        rs_http = self.switch_http_read(device_id, HttpConfigType.DeviceHttpConfig)
        # 当前程序是https 时候 我们才重新处理ip
        if rs_http['status'] == 0 and rs_http['data'] and rs_http['data']['is_open'] == 1:
            rs = device_api.read_detail(device_id)
            if rs["status"] == 0:
                slots = [x for x in rs['data']['slotList'] if x['slotId'] == slot_id]
                if slots:
                    if slots[0]['cardType'] == CardType.Mvr:
                        echo_urls = data['encoding']['echoUrl'].split(":")
                        mvr_urls = data['encoding']['mvrUrl'].split(":")
                        if len(echo_urls) != 3:
                            return get_result_model(ResInfo.Params_Error)
                        ip = echo_urls[1].replace("/","")
                        echo_path = echo_urls[2]
                        if len(mvr_urls) != 3:
                            return get_result_model(ResInfo.Params_Error)
                        mvr_path = mvr_urls[2]
                        set_nginx_and_reload(ip, echo_path, mvr_path)

                    rs = device_api.write_slot(device_id, slot_id, data)
                else:
                    rs = get_result_model(ResInfo.Device_Slot_Not_Exist)
            return rs
        return device_api.write_slot(device_id, slot_id, data)

    def device_interface_read(self, device_id, slot_id, interface_id):
        """
        读取设备的接口信息
        :param device_id: 设备Id
        :param slot_id: 卡槽Id
        :param interface_id: 接口Id
        :return: 参考协议文档：0x03 - key: deviceInterface
        """
        return device_api.read_interface(device_id, slot_id, interface_id)

    def device_power_read(self, device_id, power_id):
        """
        读取设备电源信息
        :param device_id:设备id
        :param power_id: 电源id 0：主电源，1：备份电源
        :return:参考协议文档：0x04 - key: devicePower
        """
        return device_api.read_power(device_id, power_id)  # 未实现

    def device_genlock_read(self, device_id, genlock_id):
        """
        读取设备Genlock
        :param deviceId:设备Id
        :return: 参考协议文档：0x05 - key: deviceGenlock
        """
        return device_api.read_genlock(device_id, genlock_id)

    def device_genlock_write(self, device_id, genlock_id, data):
        """
        读取设备Genlock数据
        :param device_id: 设备Id
        :param genlock_id: GenlockId
        :param data: {参考协议文档：0x05 - key: deviceGenlock
        :return: 参考协议文档：0x05 - key: deviceGenlock
        """
        return device_api.write_genlock(device_id, genlock_id, data)

    def slot_function_read(self, device_id, slot_id):
        """
        读取子卡功能类型
        :param device_id: 设备id
        :param slot_id: 槽Id
        :return: 参考协议文档：0x06 - key: deviceSlotFunction
        """
        return device_api.read_slot_function(device_id, slot_id)

    def device_ip_read(self, device_id):
        """
        设备设备IP地址和DNS
        :param device_id: 设备Id
        :return:参考协议文档：0x07 - key: deviceIp
        """
        return device_api.read_ip(device_id)  # 未实现

    def device_ip_write(self, device_id, data):
        """
        设置设备IP和DNS
        :param device_id: 设备号
        :param data: 参考协议文档：0x07 - key: deviceIp
        :return: 参考协议文档：0x07 - key: deviceIp
        """
        return device_api.write_ip(device_id, data)

    def device_com_read(self, device_id):
        """
        设置设备COM信息
        :param device_id: 设备Id
        :return: 参考协议文档：0x08 - key: deviceCom
        """
        rs = device_api.read_com(device_id)
        return rs

    def device_com_write(self, device_id, data):
        """
        设置设备COM信息
        :param device_id: 设备号
        :param data:  参考协议文档：0x08 - key: deviceCom
        :return: 参考协议文档：0x08 - key: deviceCom
        """
        return device_api.write_com(device_id, data)

    def device_save_write(self, device_id):
        """
        保存当前设备信息
        :param device_id: 设备号
        :return: 参考协议文档：0x09 - key: deviceSave
        """
        data = {'deviceId': device_id}
        return device_api.write_save(device_id, data)

    def device_shutdown_write(self, device_id, opt_type):
        """
        设备关机
        :param device_id:设备号
        :param opt_type: 0：设备重启；1：设备关机；2：开机
        :return: 参考协议文档：0x0A - key: deviceShutDown
        """
        data = {"deviceId": device_id, "type": opt_type}

        rs = device_api.write_shutdown(device_id, data)
        if rs['status'] == 0:
            push_data_org(SyncDataName.Device_Shutdown, data)
        return rs

    def device_end_process(self, device_id, opt_type):
        """
        结束任务，设备重启
        :param device_id: 设备号
        :param opt_type: 1：升级重启，2：自检重启
        :return: 参考协议文档：0x0A - key: deviceShutDown
        """
        rs = get_result_model()
        if opt_type == 1:
            if self.read_upgrade_status() or not self.upgrade_all_success:
                rs = self.device_shutdown_write(device_id, 0)

        elif opt_type == 2:  # 自检重启
            if self.read_self_check_status():
                rs = self.device_shutdown_write(device_id, 0)
        return rs

    def device_restore_factory_write(self, device_id, opt_type):
        """
        设备恢复出厂设置。
        :param device_id: 设备id
        :param opt_type: 0：保存ip恢复；1：保存ip、edid恢复；2：保存ip、edid、id恢复；3：默认恢复出厂设置
        :return: 参考协议文档：0x0B - key: deviceRestoreFactory
        """
        data = {"deviceId": device_id, "type": opt_type}
        rs = device_api.write_restore_factory(device_id, data)
        if rs['status'] == 0:
            self.device_shutdown_write(device_id, 0)
        return rs

    def device_firware_version_read(self, device_id):
        """
        获取程序版本号
        :param device_id: 设备号
        :return: 参考协议文档：0x0C - key: deviceFirwareVersion
        """
        return device_api.read_firware_version(device_id)

    def ipc_mode_read(self, device_id, slot_id):
        """
        读取设备IPC mode
        :param device_id: 设备id
        :return: 结果对象
        """
        return device_api.ipc_mode_info(device_id, slot_id)

    def device_upgrade_write(self, device_id, data):
        """
        固件升级(一)
        :param device_id: 设备号
        :param data:JSON对象
        :return:  参考协议文档：0x0D - key: deviceUpgrade
        """
        try:
            rs = device_api.write_upgrade(device_id, data)
            if rs['status'] == 0:
                self.is_upgrade_send = True
            return rs
        except:
            self.is_upgrade_send = False
            raise

    def device_upgrade_sts_read(self, device_id):
        """
        固件升级(二)读取固件升级状态
        :param device_id: 设备号
        :return:  参考协议文档：0x0D - key: deviceUpgrade
        """
        if not self.is_upgrade_send and not self.upgrade_from_usb:
            return get_result_model(ResInfo.Device_Is_Initing)
        rs = device_api.read_upgrade_sts(device_id)
        if rs['status'] == ResInfo.Device_Upgrade_Fail.value:
            self.is_upgrade_send = False
            self.end_upgrade()
        if rs['status'] == 0:
            data = rs['data']
            # 升级结果
            process_status = data["processStatus"]
            rs['data']['isUpgrading'] = 1 if self.read_upgrade_status() else 0
            rs['data']['retryTimes'] = self.upgrade_retry_times
            # 升级结束
            if process_status == 0:
                success, sections = self._valid_upgrade_result(data)
                if success:  # 升级成功
                    self.upgrade_retry_times = 0
        return rs

    def mvr_is_online(self, device_id):
        """
        预监卡是否在线
        :param device_id:
        :return: 1 在线，0 不在线
        """
        rs = self.device_detail_read(device_id)
        on_line = 0
        slot_id = -1
        if rs['status'] == 0 and rs['data']:
            # 读取预监卡
            slots = [s for s in rs['data']['slotList'] if s['cardType'] == CardType.Mvr]
            if slots:
                # 预监卡接口是否有接口
                if slots[0]['cardType'] != 0:
                    on_line = 1
                    slot_id = slots[0]['slotId']
        rs = get_result_model()
        rs['data'] = {'online': on_line, 'deviceId': device_id, 'slotId': slot_id}
        return rs

    def device_upgrade_info_read(self, device_id):
        """
        读取固件升级信息
        :param device_id: 设备Id
        :return: 固件升级信息
        """
        # 读取版本信息
        rs = self.device_firware_version_read(device_id)
        if rs['status'] != 0 or not rs['data']:
            return rs
        version_info = rs['data']
        # 构建升级信息
        data = dict()
        data['processStatus'] = 1 if self.read_upgrade_status() else 0
        data['deviceId'] = device_id
        data['version'] = version_info['version']
        data['brief'] = version_info['brief']

        main_model_id = version_info['mainControl']['modelId']
        # 主控信息
        data['mainControl'] = self._build_upgrade_info(version_info['mainControl'])
        data['mainControl']['sections'].append(
            {"code": UpgradeItem.MCU.value,
             "name": UpgradeItem.MCU.name,
             "version": version_info['mainControl']['ARM']['version'],
             "filename": version_info['mainControl']['ARM']['filename'],
             "brief": version_info['mainControl']['ARM']['brief'],
             "folderPath":version_info['mainControl']['folderPath']}
        )
        # aux信息
        if 'auxcard' in version_info:
            data['auxCard'] = self._build_upgrade_info(version_info['auxcard'], main_model_id,'auxcard')

        if 'backboard' in version_info:
            data['backboard'] = self._build_upgrade_info(version_info['backboard'])

        # 输入卡信息
        if 'inputCard' in version_info and version_info['inputCard']:
            data['inputCard'] = []
            for card in version_info['inputCard']:
                input_card = self._build_upgrade_info(card)
                data['inputCard'].append(input_card)

        # 输出卡信息
        if 'outputCard' in version_info and version_info['outputCard']:
            data['outputCard'] = []
            for card in version_info['outputCard']:
                output_card = self._build_upgrade_info(card)
                data['outputCard'].append(output_card)

        # mvr卡信息
        if 'mvrCard' in version_info:
            data['mvrCard'] = self._build_upgrade_info(version_info['mvrCard'])

        rs['data'] = data
        return rs

    def _build_upgrade_info(self, card, main_model_id=None, card_name=None):
        """
        升级mcu和fpga信息
        :param card:  升级卡内容
        :return: 升级数据对象
        """
        info = {
            "modelId": card['modelId'],
            "version": card['version'],
            "brief": card['brief'],
            "hardver": card['hardver'] if 'hardver' in card else '',
            "folderPath": card['folderPath'],
            "sections": []
        }
        if 'funcs' in card:
            info['funcs'] = card['funcs']
        if 'funcCode' in card:
            info['funcCode'] = card['funcCode']

        if UpgradeItem.MCU.name in card:
            info['sections'].append(
                {"code": card['MCU']['code'],
                 "name": UpgradeItem.MCU.name,
                 "version": card['MCU']['version'],
                 "brief": card['MCU']['brief'],
                 "filename": card['MCU']['filename']}
            )

        if 'slotId' in card:
            info["slotId"] = card['slotId']

        if UpgradeItem.FPGA_A.name in card:
            info['sections'].append(
                {"code": card['FPGA_A']['code'],
                 "name": UpgradeItem.FPGA_A.name,
                 "version": card['FPGA_A']['version'],
                 "brief": card['FPGA_A']['brief'],
                 "filename": card['FPGA_A']['filename']}
            )
            if UpgradeItem.FPGA_B.name in card:
                info['sections'].append(
                    {"code": card['FPGA_B']['code'],
                     "name": UpgradeItem.FPGA_B.name,
                     "version": card['FPGA_B']['version'],
                     "brief": card['FPGA_B']['brief'],
                     "filename": card['FPGA_B']['filename']}
                )
        elif UpgradeItem.FPGA.name in card:
            info['sections'].append(
                {"code": card['FPGA']['code'],
                 "name": UpgradeItem.FPGA.name,
                 "version": card['FPGA']['version'],
                 "brief": card['FPGA']['brief'],
                 "filename": card['FPGA']['filename']}
            )

        if UpgradeItem.ASIC.name in card:
            info['sections'].append(
                {"code": card['ASIC']['code'],
                 "name": UpgradeItem.ASIC.name,
                 "version": card['ASIC']['version'],
                 "brief": card['ASIC']['brief'],
                 "filename": card['ASIC']['filename']}
            )
        return info

    def device_upgrade(self, device_id, upgrade_type, parts, upgrade_file):
        """
        升级
        :param device_id: 设备Id
        :param upgrade_type:  升级类型
        :param parts:  升级部分
        :return:  结果对象
        """
        # 检查系统是否存在core文件 存在则删除
        if os.path.exists(SysSettings.Middleware_Core_File_Path):
            os.remove(SysSettings.Middleware_Core_File_Path)
        if os.path.exists(SysSettings.Middleware_CoreDump_File_Path):
            os.remove(SysSettings.Middleware_CoreDump_File_Path)

        st = time.monotonic()  # 返回当前时间的时间戳（1970纪元后经过的浮点秒数）
        path_nm = upgrade_file
        device_rs = self.device_detail_read(device_id)
        if device_rs['status'] != 0:
            return get_result_model(ResInfo.Middle_Data_Err)
        main_model_id = device_rs['data']['modelId']
        try:
            extract_path = os.path.join(SysSettings.Upload_Upgrade_File_Path, 'firmwareUpgradeFile')
            if main_model_id in DeviceType.P_Main_All:
                package_pwd_rs = deviceInfodao.get_package_pwd()
                if package_pwd_rs['status'] != 0:
                    return package_pwd_rs
                package_pwd = package_pwd_rs['data']['packagePwd']
                ex_rs = pd_extract(path_nm, extract_path, package_pwd)
            else:
                ex_rs = extract(upgrade_file, extract_path)
            # 文件上传成功后还要调用中间件进行设备升级
            if ex_rs is not True:
                shutil.rmtree(extract_path)
                logger.error("固件升级-解压文件-失败!")
                return get_result_model(ResInfo.Extract_File_Err)
        except Exception as e:
            logger.error("固件升级-解压文件-失败: %s" % e)
            return get_result_model(ResInfo.Extract_File_Err)
        upgrade_file_path = extract_path
        upgrade_forder = os.listdir(upgrade_file_path)[0]
        if main_model_id in DeviceType.P_Main_All:
            if SysSettings.Version_Type != 0:
                if not os.path.exists(os.path.join(upgrade_file_path, upgrade_forder, 'version.conf')):
                    logger.error('升级包未找到版本类型文件-升级失败')
                    return get_result_model(ResInfo.Device_Upgrade_File_Error)
                with open(os.path.join(upgrade_file_path, upgrade_forder, 'version.conf'), 'r') as f:
                    version_type = json.load(f)
                if version_type['versionType'] != SysSettings.Version_Type:
                    logger.error('升级包版本不符-升级失败')
                    return get_result_model(ResInfo.Device_Upgrade_File_Error)


        upgrade_ctrl_path = ""
        if main_model_id == DeviceType.Alita_20:
            upgrade_ctrl_path = os.path.join(os.path.join(upgrade_file_path, upgrade_forder), 'Alita_Ctrl')
        if main_model_id == DeviceType.H_20U:
            upgrade_ctrl_path = os.path.join(os.path.join(upgrade_file_path, upgrade_forder), 'H_Ctrl')
        if os.path.exists(upgrade_ctrl_path):
            files = os.listdir(upgrade_ctrl_path)
            flag = True
            index = -1
            file_h_index = -1
            file_a_index = -1
            for file in files:
                if file == SysSettings.Safe_Valid_File:
                    index = 0
                if file == SysSettings.H20_Valid_File:
                    file_h_index = 0
                if file == SysSettings.A20_Valid_File:
                    file_a_index = 0
            # 安全版本且找不到该文件
            is_safe = g.safe
            if is_safe == SafeConfig.IsSafe and index == -1:
                flag = False
            # H20 不准向下升级
            if main_model_id == DeviceType.H_20U and file_h_index == -1:
                flag = False
            # A20 不准向下升级
            if main_model_id == DeviceType.Alita_20 and file_a_index == -1:
                flag = False
            if not flag:
                logger.info("当前系统版本不准升级为不匹配系统")
                return get_result_model(ResInfo.Device_Upgrade_File_Error)
        upgrade_folder = os.listdir(upgrade_file_path)[0]
        # 确定升级内容/选项
        upgrade_parts = None
        logger.info('设备升级，组装升级请求包')
        if upgrade_type == 0:  # 整体升级
            fireware_version = self.device_upgrade_info_read(device_id)
            if fireware_version['status'] == 0 and fireware_version['data']:
                upgrade_parts = fireware_version['data']
        else:  # 部分升级
            upgrade_parts = parts

        # 调用中间件接口以进行升级操作
        if upgrade_parts is None:
            logger.error("固件升级-失败:upgradeParts为空，停止升级")
            return get_result_model(ResInfo.Device_Upgrade_Parts_Empty)

        # 升级json
        try:
            # 将parts中upgradeFile、filePath字段进行赋值
            logger.info('设备升级，验证升级文件是否正确')
            self.build_upgrade_parts(device_id, path_nm, upgrade_parts,
                                     os.path.join(upgrade_file_path, upgrade_folder), main_model_id)
        except Exception as e:
            logger.info('设备升级，验证升级包异常' + str(e))
            logger.exception(e)
            return get_result_model(ResInfo.Device_Upgrade_File_Error)

        logger.info("开始固件升级.deviceId:%i,upgradeType:%i,parts:%s" % (device_id, upgrade_type, upgrade_parts))
        # rs = get_result_model()
        rs = self.device_upgrade_write(device_id, data=upgrade_parts)
        # 记录升级状态
        if rs['status'] != 0:
            logger.info("固件升级-失败:%s" % str(rs))
            return rs
        else:
            self.upgrade_sections = upgrade_parts
            logger.info("固件升级-标记:%s, 开始轮询升级状态" % self.is_upgrade)
            from threading import Thread
            # Timer(5, self._query_upgrade_status, (device_id, g.token, g.user)).start()
            Thread(target=self._query_upgrade_status, args=(device_id, g.token, g.user)).start()

            et =  time.monotonic()
            lu = et - st
            logger.info("固件升级(一)耗时：%i秒" % lu)
            return rs

    def build_upgrade_parts(self, device_id, path_nm, parts, path, main_model_id):
        """
        拼接升级包json
        :param device_id: 设备id
        :param path_nm: 压缩包文件路径
        :param parts: 升级part
        :param path: 解压缩包路径
        :return: None
        """
        parts["deviceId"] = device_id
        parts["upgradeFile"] = path_nm
        parts["processStatus"] = 0
        parts["version"] = '1.0.0.0'

        main_control = parts.get("mainControl")
        aux_card = parts.get("auxCard")
        input_cards = parts.get("inputCard")  # List集合
        output_cards = parts.get("outputCard")  # List集合
        mvr_card = parts.get("mvrCard")
        back_card = parts.get("backboard")
        try:
            has_section = False
            file_name = 'CtrlCardExec.sh'
            if main_control and main_control['sections']:
                self._add_upgrade_file_path(main_control, path, valid_xml=False)
                if main_control['sections']:
                    has_section = True

            if aux_card and aux_card['sections']:
                self._add_upgrade_file_path(aux_card, path)
                if aux_card['sections']:
                    has_section = True

            if back_card and back_card['sections']:
                self._add_upgrade_file_path(back_card, path)
                if back_card['sections']:
                    has_section = True

            if output_cards:
                for output_card in output_cards:
                    if output_card['sections']:
                        self._add_upgrade_file_path(output_card, path)
                        if output_card['sections']:
                            has_section = True
            if input_cards:
                for input_card in input_cards:
                    if input_card['sections']:
                        self._add_upgrade_file_path(input_card, path)
                        if input_card['sections']:
                            has_section = True
            if mvr_card:
                self._add_upgrade_file_path(mvr_card, path)
                if mvr_card['sections']:
                    has_section = True

            if not has_section:
                raise RuntimeError('文件section为空')
        except Exception as e:
            msg = traceback.format_exc()
            logger.error('固件升级，查找升级文件异常：%s' % msg)
            raise e

    def _add_upgrade_file_path(self, item, file_path, valid_xml=True):
        """
        升级文件路径校验
        :param item:  升级项目
        :param file_path:  升级文件路径
        :return:  none
        """
        if 'funcs' in item:
            item = self.advanced_data_processing(item, file_path)

        # model_id = item['modelId']
        # if model_id not in UpgradeFileModel:
        #     item['sections'] = []
        #     return
        dirs = item['folderPath']
        folder = None
        if isinstance(dirs, tuple):
            for d in dirs:
                if os.path.exists(os.path.join(file_path, d)):
                    folder = d
                    break
            if not folder:
                folder = dirs[0]
        else:
            folder = dirs
        file_path = os.path.join(file_path, folder)
        if valid_xml:
            if not os.path.exists(os.path.join(file_path, 'Config.xml')):
                logger.error('upgrade file not find: %s:%s' % (file_path, 'Config.xml'))
                item['sections'] = []
                return
        item['version'] = '1.0.0.0'
        for section in item['sections'][::-1]:
            filename = section['filename']
            if not os.path.exists(os.path.join(file_path, filename)):
                logger.error('upgrade file not find: %s:%s' % (file_path, filename))
                item['sections'].remove(section)
            else:
                section['filePath'] = os.path.join(file_path, filename)
                section['xmlPath'] = os.path.join(file_path, 'Config.xml')

    def advanced_data_processing(self, card, path):
        """
        对高级功能数据的处理
        :param card:  升级项目
        :param path:  升级文件路径
        :return:  none
        """
        model_id = card['modelId']
        funcs = card['funcs']
        for section in card['sections']:
            if card['funcCode'] != UpgradeFuncCode.func_code_common:
                for func in funcs:
                    if func['type'] == section['code']:
                        if func['funcEnable'] in [UpgradeFuncCode.func_dynamic_subPixe_code, UpgradeFuncCode.func_code_enable]:
                            # 判断是否能找到对应的文件
                            folder = card['folderPath']
                            file_path = os.path.join(path, folder)
                            if os.path.exists(os.path.join(file_path, func['filename'])):
                                section['code'] = func['code']
                                section['filename'] = func['filename']
        return card

    def _query_upgrade_status(self, device_id, token, user):
        """
        查询固件升级状态
        :param device_id: 设备Id
        :return: None
        """
        # 正在升级
        while True:
            time.sleep(5)
            if self.read_upgrade_status() and time.monotonic() - self.upgrade_time > SysSettings.Upgrade_Time:
                self.end_upgrade()
                return
            from H_9U import app
            with app.app_context():
                # 获取token
                g.token = token
                g.user = user
                logger.info("固件升级，查询升级结果...")
                # 查询结果成功
                upgrade_rs = self.device_upgrade_sts_read(device_id)
                if upgrade_rs["status"] == ResInfo.Device_Upgrade_Fail.value:
                    self.end_upgrade()
                    return
                elif upgrade_rs["status"] != 0:
                    continue

                # 查询升级结果成功
                data = upgrade_rs['data']
                # 升级结果
                process_status = data["processStatus"]
                if process_status != 0:
                    logger.info("固件升级-升级中,稍后查询")

                    continue
                logger.info("固件升级-结束")
                success, sections = self._valid_upgrade_result(data)
                if success:  # 升级成功
                    self.upgrade_retry_times = 0
                    logger.info("固件升级-成功,开始重启设备")
                    time.sleep(5)
                    self.device_shutdown_write(device_id, 0)
                else:  # 升级失败
                    retried = False
                    while not retried and self.upgrade_retry_times > 0:
                        self.upgrade_retry_times -= 1
                        try:
                            up_sections = self._build_retry_sections(sections)
                            logger.info(
                                "重试升级.times:%i,parts:%s" % (self.upgrade_retry_times, up_sections))
                            rs = self.device_upgrade_write(device_id, up_sections)
                        except Exception as e:
                            logger.error(e)
                            rs = get_result_model(ResInfo.Midware_Inner_Error)
                        if rs['status'] == 0:
                            retried = True
                    if retried:
                        logger.info("固件升级-重试成功,重试：" + str(self.upgrade_retry_times))
                        Timer(5, self._query_upgrade_status, (device_id, token, user)).start()
                    else:
                        logger.info("固件升级失败,开始重启设备")
                        time.sleep(5)
                        self.device_shutdown_write(device_id, 0)

    def _build_retry_sections(self, err_sections):
        if not self.upgrade_sections:
            logger.info("重试升级。self section 为空")
            raise RuntimeError("重试升级。self section 为空")

        sections = deepcopy(self.upgrade_sections)
        # 主控
        if 'mainControl' not in err_sections:
            sections['mainControl']['sections'] = []
        else:
            new_sections = []
            self.__build_retry_new_sections(err_sections['mainControl'], sections['mainControl'], new_sections)
            sections['mainControl']['sections'] = new_sections

        # aux
        if 'auxCard' not in err_sections and 'auxCard' in sections:
            sections['auxCard']['sections'] = []
        else:
            new_sections = []
            self.__build_retry_new_sections(err_sections['auxCard'], sections['auxCard'], new_sections)
            sections['auxCard']['sections'] = new_sections
        # input
        for input in sections['inputCard']:
            if 'inputCard' in err_sections:
                new_sections = []
                for err_card in err_sections['inputCard']:
                    if err_card['slotId'] == input['slotId']:
                        self.__build_retry_new_sections(err_card, input, new_sections)
                input['sections'] = new_sections
            else:
                input['sections'] = []
        # output
        for output in sections['outputCard']:
            if 'outputCard' in err_sections:
                new_sections = []
                for err_card in err_sections['outputCard']:
                    if err_card['slotId'] == output['slotId']:
                        self.__build_retry_new_sections(err_card, output, new_sections)
                output['sections'] = new_sections
            else:
                output['sections'] = []
        # mvr
        if 'mvrCard' not in err_sections:
            if 'mvrCard' in sections:
                sections['mvrCard']['sections'] = []
        else:
            new_sections = []
            self.__build_retry_new_sections(err_sections['mvrCard'], sections['mvrCard'], new_sections)
            sections['mvrCard']['sections'] = new_sections

        # backboard
        if 'backboard' not in err_sections:
            if 'backboard' in sections:
                sections['backboard']['sections'] = []
        else:
            new_sections = []
            self.__build_retry_new_sections(err_sections['backboard'], sections['backboard'], new_sections)
            sections['backboard']['sections'] = new_sections
        return sections

    def __build_retry_new_sections(self, err_card, card, new_sections):
        """
        :param err_card: 升级失败信息
        :param card: 原升级信息
        :param new_sections: 新的升级信息
        """
        for err_section in err_card['sections']:
            new_section = [card_section for card_section in card['sections']
                           if err_section['name'] == card_section['name']]
            new_sections.extend(new_section)

    def _valid_upgrade_result(self, upgrade_data):
        """
        校验升级结果，全部section的result 为0 返回True
        :param upgrade_data: 升级结果数据
        :return: True 成功
        """
        # 主控
        err_sec = {}
        all_success = True
        main_control = upgrade_data.get('mainControl', None)
        if main_control:
            success, sections = self._valid_upgrade_sections(main_control['sections'])
            if not success:
                err_sec['mainControl'] = deepcopy(main_control)
                err_sec['mainControl']['sections'] = sections

                all_success = False
        # aux
        aux_card = upgrade_data.get('auxCard', None)
        if aux_card:
            success, sections = self._valid_upgrade_sections(aux_card['sections'])
            if not success:
                err_sec['auxCard'] = deepcopy(aux_card)
                err_sec['auxCard']['sections'] = sections
                all_success = False
        # input
        input_card = upgrade_data.get('inputCard', None)
        if input_card:
            for input in input_card:
                success, sections = self._valid_upgrade_sections(input['sections'])
                if not success:
                    if 'inputCard' not in err_sec:
                        err_sec['inputCard'] = []
                    err_input = deepcopy(input)
                    err_input['sections'] = sections
                    err_sec['inputCard'].append(err_input)

                    all_success = False
        # output
        output_card = upgrade_data.get('outputCard', None)
        if output_card:
            for output in output_card:
                success, sections = self._valid_upgrade_sections(output['sections'])
                if not success:
                    if 'outputCard' not in err_sec:
                        err_sec['outputCard'] = []
                    err_output = deepcopy(output)
                    err_output['sections'] = sections
                    err_sec['outputCard'].append(err_output)
                    all_success = False
        # mvr
        mvr_card = upgrade_data.get('mvrCard', None)
        if mvr_card:
            success, sections = self._valid_upgrade_sections(mvr_card['sections'])
            if not success:
                err_sec['mvrCard'] = deepcopy(mvr_card)
                err_sec['mvrCard']['sections'] = sections
                all_success = False

        # 背板
        back_card = upgrade_data.get('backboard', None)
        if back_card:
            success, sections = self._valid_upgrade_sections(back_card['sections'])
            if not success:
                err_sec['backboard'] = deepcopy(back_card)
                err_sec['backboard']['sections'] = sections
                all_success = False

        return all_success, err_sec

    def _valid_upgrade_sections(self, sections):
        """
        校验升级结果，全部section的result为0则返回成功
        :param sections: 升级section
        :return: True 成功
        """
        if sections:
            s = [section for section in sections if section['result']!= 0]
            if s:
                logger.info("升级校验结果不成功："+ str(s))
                return False, s
        return True, None

    def device_main_control_read(self, device_id, slot_id=None):
        """
        读取主控版本信息
        :param device_id: 设备Id
        :return: 主控版本信息
        """
        rs = self.device_firware_version_read(device_id)
        if rs['status'] != 0:
            return rs
        if 'inputCard' in rs['data']:
            rs['data'].pop('inputCard')
        if 'outputCard' in rs['data']:
            rs['data'].pop('outputCard')
        if 'mvrCard' in rs['data']:
            rs['data'].pop('mvrCard')

        rs['data']['mainControl']['status'] = 1
        rs['data']['auxcard']['status'] = 1
        if 'backboard' in rs['data']:
            rs['data']['backboard']['status'] = 1
        if slot_id is None:
            return rs
        device_rs = self.device_detail_read(device_id)
        if device_rs['status'] == 0 and 'ctrlCard' in device_rs['data']:
            ctrl_card = device_rs['data']['ctrlCard']
            if slot_id == ctrl_card['mainSlotId']:
                rs['data']['mainControl']['backupStatus'] = 1 if ctrl_card['mainState'] == 1 else 0
                rs['data']['mainControl']['status'] = 1 if ctrl_card['mainState'] in [1, 2] else 0
            elif slot_id == ctrl_card['deputySlotId']:
                rs['data']['mainControl']['backupStatus'] = 1 if ctrl_card['deputyState'] == 1 else 0
                rs['data']['mainControl']['status'] = 1 if ctrl_card['deputyState'] in [1, 2] else 0
        return rs

    def import_settings(self, device_id):
        """
        导入配置文件
        :param device_id: 设备Id
        :return: 文件对象
        """
        # 上传图片
        up_rs = upload_file(UploadFileType.Device_Settings)
        if not up_rs["bln"]:
            return get_result_model(ResInfo.Upload_File_Fail)

        # 上传成功，下发中间件
        path_nm = up_rs["pathNm"]
        data = {'deviceId': device_id, 'type': 0, 'filePath': path_nm}
        rs = device_api.write_device_settings(device_id, data)
        if rs['status'] != 0:
            # 删除文件
            remove_file(path_nm)
        else:
            # opt_type: 0：设备重启；1：设备关机；2：开机
            # 5秒后设备重启
            self.device_shutdown_write(device_id, 0)
        return rs

    def export_settings(self, device_id):
        """
        导出配置文件
        :param device_id:设备Id
        :return:文件信息
        """
        path = SysSettings.Device_Download_Path
        setting_file = os.path.join(path, 'settings.zip')
        if os.path.exists(path):
            if os.path.exists(setting_file):
                os.remove(setting_file)
        else:
            os.mkdir(path)

        data = {
            "deviceId": device_id,
            "type": 1,  # 1 导出
            "filePath": setting_file
        }
        rs = device_api.write_device_settings(device_id, data)
        if rs['status'] == 0:
            rs['data'] = {'filePath': setting_file}
        return rs

    def device_self_check(self, device_id):
        """
        设备自检
        :param device_id:
        :return:
        """
        rs = device_api.write_self_check(device_id, {'deviceId': device_id})
        if rs['status'] == ResInfo.Repeat_Err.value:
            rs = get_result_model()
        return rs

    def device_self_check_status(self, device_id):
        """
        读取设备自检状态
        :param device_id: 设备id
        :return: 结果对象
        """
        return device_api.read_self_check_status(device_id)

    def nomark_mode(self, device_id):
        """
        读取设备中性标志
        :param device_id:  设备id
        :return: 中性标志
        """
        rs = device_api.read_nomark_mode(device_id)
        if rs['status'] == 0 and rs['data']:
            logo, icon, bkg = nomark_manager.get_default_image_urls()
            if rs['data']['nomarkMode'] == 0:
                rs['data']['customList'] = []
                rs['data']['logo'] = logo
                rs['data']['icon'] = icon
                rs['data']['bkg'] = bkg
                return rs
            if 'logo' not in rs['data']:
                rs['data']['logo'] = logo
            else:
                rs['data']['logo'] = nomark_manager.format_image_url(rs['data']['logo'])

            if 'icon' not in rs['data']:
                rs['data']['icon'] = icon
            else:
                rs['data']['icon'] = nomark_manager.format_image_url(rs['data']['icon'])

            if 'bkg' not in rs['data']:
                rs['data']['bkg'] = bkg
            else:
                rs['data']['bkg'] = nomark_manager.format_image_url(rs['data']['bkg'])
        return rs

    def nomark_mode_write(self, device_id, nomark_mode, custom_list, lang_list):
        """
        设置设备中性标志
        :param device_id:  设备id
        :return: 中性标志
        """
        data = {
            'deviceId': device_id,
            'nomarkMode': nomark_mode
        }
        if nomark_mode == 2:  # 定制商标
            # 公司信息
            if custom_list:
                data['customList'] = custom_list
            if lang_list:
                data['langList'] = lang_list

            # 读取原中性设置
            rs = device_api.read_nomark_mode(device_id)
            if rs['status'] != 0:
                return get_result_model(ResInfo.Middle_Data_Err)
            logo_default, icon_default, bkg_default = nomark_manager.get_default_image_paths()
            # logo信息
            # 如有文件上传，则上传文件并发送给中间件
            # 如没有文件上传，则判断之前是否有logo数据，没有则使用默认图片
            # icon、bkg 逻辑同上
            if 'logo' in request.files:
                up_rs = common.upload_file(UploadFileType.NomarkMode, file_key='logo')
                if not up_rs["bln"]:
                    return get_result_model(ResInfo.Upload_File_Fail)
                path_nm = up_rs["pathNm"]
                data['logo'] = path_nm
            elif 'logo' not in rs['data']:
                data['logo'] = logo_default
            if 'icon' in request.files:
                up_rs = common.upload_file(UploadFileType.NomarkMode, file_key='icon')
                if not up_rs["bln"]:
                    return get_result_model(ResInfo.Upload_File_Fail)
                path_nm = up_rs["pathNm"]
                data['icon'] = path_nm
            elif 'icon' not in rs['data']:
                data['icon'] = icon_default
            if 'bkg' in request.files:
                up_rs = common.upload_file(UploadFileType.NomarkMode, file_key='bkg')
                if not up_rs["bln"]:
                    return get_result_model(ResInfo.Upload_File_Fail)
                path_nm = up_rs["pathNm"]
                data['bkg'] = path_nm
            elif 'bkg' not in rs['data']:
                data['bkg'] = bkg_default

        rs = device_api.write_nomark_mode(device_id, data)

        return rs

    def effective_time_write(self, device_id, data):
        """
        设备授权时间
        :param device_id: 设备Id
        :param data: 授权时间
        :return: 结果对象
        """

        return device_api.write_effective_time(device_id, data)

    def local_time_write(self, device_id, data):
        """
        设备时间
        :param device_id: 设备Id
        :param data: 时间
        :return: 结果对象
        """

        return device_api.write_local_time(device_id, data)

    def effective_time_read(self, device_id):
        """
        设备授权时间
        :param device_id: 设备Id
        :return: 结果对象
        """
        rs = device_api.read_effective_time(device_id)
        if rs['status'] == 0:
            local_time_rs = self.local_time_read(device_id)
            rs['loaclTime'] = {}
            if local_time_rs['status'] == 0:
                rs['data']['loaclTime'] = local_time_rs['data']
        return rs

    def local_time_read(self, device_id):
        """
        读取设备当前时间
        :param device_id: 设备Id
        :return:  结果对象
        """
        return device_api.read_local_time(device_id)

    def slot_function_write(self, device_id, slot_id, functions):
        """
        设置子卡接口功能
        :param device_id: 设备Id
        :param slot_id: slotId
        :param functions: 子卡功能设置
        :return: 结果对象
        """
        slot_rs = device_api.read_slot(device_id, slot_id)
        if slot_rs['status'] != 0:
            return get_result_model(ResInfo.Device_Detail_Error)

        data = []
        if slot_rs['data']['modelId'] in ModelId.Input_SL_DL_List:
            for f in functions:
                i_id = int(f['id'])
                func = int(f['functionType'])
                if func not in [FunctionType.SL, FunctionType.DL]:
                    return get_result_model(ResInfo.Params_Error)
                data.append({
                    'interfaceId': 2 * i_id,
                    'functionType': FunctionType.SL if func == FunctionType.SL else FunctionType.Disable,
                })
                data.append({
                    'interfaceId': 2 * i_id + 1,
                    'functionType': func
                })
        elif slot_rs['data']['modelId'] in [ModelId.Input_HDMI20_DP12, ModelId.Pd_Input_HDMI20_DP12]:
            f = functions[0]
            func = int(f['functionType'])
            if func not in [FunctionType.HDMI20, FunctionType.DP12]:
                return get_result_model(ResInfo.Params_Error)
            data.append({
                'interfaceId': 0,
                'functionType': func
            })
            # 4选2输入卡
        elif slot_rs['data']['modelId'] in [ModelId.H_2xHDMI20_2xDP12_I, ModelId.H_2xHDMI_DP_I, ModelId.Pd_H_2xHDMI20_2xDP12_I, ModelId.Pd_H_2xHDMI_DP_I, ModelId.H_2xHDMI21_2xDP14_AUDIO_I]:

            f = functions[0]
            func = int(f['functionType'])
            func_id = int(f['id'])
            if func not in [2,3,4,5]:
                return get_result_model(ResInfo.Params_Error)
            if func not in [FunctionType.HDMI20, FunctionType.DP12, FunctionType.DP14, FunctionType.HDMI21]:
                return get_result_model(ResInfo.Params_Error)
            if func_id == 0:
                if func in [3, 5]:
                    data.append(
                    {
                    'interfaceId': 0,
                    'functionType': 255
                })
                    data.append(
                    {'interfaceId': 1,
                    'functionType': func
                     }),
                if func in [2, 4]:
                    data.append(
                    {
                        'interfaceId': 0,
                        'functionType': func
                    })
                    data.append(
                    {'interfaceId': 1,
                     'functionType': 255
                     }),
            if func_id == 1:
                if func in [3,5]:
                    data.append(
                    {
                    'interfaceId': 2,
                    'functionType': 255
                })
                    data.append(
                    {'interfaceId': 3,
                    'functionType': func
                     }),
                if func in [2, 4]:
                    data.append(
                    {
                        'interfaceId': 2,
                        'functionType': func
                    })
                    data.append(
                    {'interfaceId': 3,
                     'functionType': 255
                     }),
        rs = get_result_model()
        if data:
            rs = device_api.write_slot_function(device_id, slot_id,
                                           {'deviceId': device_id, 'slotId': slot_id, 'interfaces': data})

        return rs

    def read_device_language(self, device_id):
        """
        读取语言模式
        :param device_id: 设备Id
        :return: 结果对象
        """
        return device_api.read_device_language(device_id)

    def write_device_language(self, device_id, data):
        """
        设置语言模式
        :param device_id: 设备Id
        :param data: 语言模式
        :return: 结果对象
        """
        return device_api.write_device_language(device_id, data)

    def ipc_mode_write(self, device_id, slot_id, data):
        """
        设置设备IPC mode
        :param device_id: 设备id
        :param data: mode
        :return: 结果对象
        """
        return device_api.write_ipc_mode(device_id, slot_id, data)

    def resource_mode_read(self, device_id):
        """读取设备资源状态，跨接口是否减图层"""
        return device_api.read_resource_mode(device_id)

    def resource_mode_write(self, device_id, data):
        """设置图层资源状态“跨接口是否减图层"""
        return device_api.write_resource_mode(device_id, data)

    def backup_mode_read(self, device_id):
        """读取设备备份状态"""
        return device_api.backup_mode_read(device_id)

    def backup_mode_write(self, device_id, data):
        """设置设备备份状态"""
        return device_api.backup_mode_write(device_id, data)

    def device_init_status_read(self, device_id):
        """读取设备初始化状态"""
        return device_api.init_status_read(device_id)

    def device_type_read(self, device_id):
        return device_api.device_type_read(device_id)

    def device_read_card_info(self, device_id, card_type):
        """
        综合读取卡信息
        :param device_id: 设备id
        :param card_type: 卡类型
        :return: 结果对象
        """
        # 判断卡的类型是输入或者输出 1 = 输入; 2 = 输出
        rs = get_result_model()
        interfaces_list = []
        if card_type == 1:
            input_list = []
            from H_9U.service.input import input_svc
            input_rs = input_svc.read_list(device_id)
            device_rs = device_svc.device_detail_read(device_id)
            slot_ids = []
            if device_rs['status'] == 0 and device_rs['data']:
                slot_lists = device_rs['data']['slotList']
                for slot_list in slot_lists:
                    if len(slot_list['interfaces']) == 0:
                        slot_ids.append(slot_list['slotId'])
            if input_rs["status"] == 0 and  input_rs['data']:
                if input_rs['data']['inputs']:
                    input_list = []
                for item in [x for x in input_rs['data']['inputs']]:
                    try:
                        detail = input_svc.read_detail(device_id, item['inputId'])
                        if detail['status'] == 0:
                            # 未插卡的筛掉
                            if detail['data']['slotId'] in slot_ids:
                                continue
                            if detail['data']['general']['name'] == "":
                                continue
                            interface_type = detail['data']['interfaceType']
                            if interface_type != 0:
                                interface_rs = {'interfaceType': interface_type}
                                interfaces_list.append(interface_rs)
                            if detail['data']['resolution']['refresh'] > 0:
                                is_signal = 1
                            else:
                                is_signal = 0
                            resolution = detail['data']['resolution']
                            color_depth = detail['data']['general']['colorDepth']
                            color_space = detail['data']['general']['colorSpace']
                            csc_mode = detail['data']['general']['cscMode']
                            sample_rate = detail['data']['general']['sampleRate']
                            name = detail['data']['general']['name']
                            detail_rs = {'resolution': resolution, 'colorDepth' : color_depth, 'colorSpace': color_space, 'cscMode' : csc_mode, 'sampleRate': sample_rate,
                                         'name': name, 'id': item['inputId'], 'iSignal': is_signal }
                            input_list.append(detail_rs)
                    except Exception as e:
                        logger.error("输入源列表异常：" + str(e))
                rs['data']['inputs'] = input_list
        elif card_type == 2:
            freeze_list = []
            from H_9U.service.output import output_svc
            output_rs = output_svc.read_list(device_id)
            from H_9U.service.screen import screen_svc
            screen_rs = screen_svc.freezed_output(device_id)
            if screen_rs['status'] == 0 and screen_rs['data']:
                freeze_list = screen_rs['data']['freeze']
            if output_rs["status"] == 0 and output_rs['data']:
                output_list = []
                for item in output_rs['data']['outputs']:
                    try:
                        detail = output_svc.output_read_detail(device_id, item['outputId'])
                        if detail['status'] == 0:
                            from H_9U.api.device import device_detail_without_cache
                            device = device_detail_without_cache(device_id)
                            slots = []
                            if device['status'] == 0:
                                slots = device['data']['slotList']
                            slot_id = detail['data']['slotId']
                            interface_id = detail['data']['interfaceId']
                            interface_type = detail['data']['interfaceType']
                            interface_rs = {'interfaceType': interface_type}
                            interfaces_list.append(interface_rs)
                            i_signal = 0
                            if  item['outputId'] in freeze_list:
                                freeze = 1
                            else:
                                freeze = 0
                            slot = [s for s in slots if s['slotId'] == slot_id][0]
                            if slot['cardType'] != CardType.NoCard:
                                model_id = slot['modelId']
                                detail['data']['cardCategory'] = CardCategory.get_card_category(model_id)
                                interface = [i for i in slot['interfaces'] if i['interfaceId'] == interface_id][0]
                                i_signal = interface['iSignal']
                            resolution = detail['data']['resolution']
                            color_depth = detail['data']['general']['colorDepth']
                            color_space = detail['data']['general']['colorSpace']
                            sample_rate = detail['data']['general']['sampleRate']
                            name = detail['data']['general']['name']
                            detail_rs = {'resolution': resolution, 'colorDepth': color_depth, 'colorSpace': color_space, 'sampleRate': sample_rate,
                                     'name': name, 'id': item['outputId'], 'iSignal': i_signal, 'freeze': freeze}
                            output_list.append(detail_rs)
                    except Exception as e:
                        logger.error("输出源列表异常：" + str(e))
                rs['data']['outputs'] = output_list
        else:
            return get_result_model(ResInfo.Params_Error)
        rs['data']['cardType'] = card_type
        rs['data']['deviceId'] = device_id
        rs['data']['interfaces'] = interfaces_list
        return rs

    def device_read_device_hk_fire_ware_version(self, device_id):
        """
        读取设备详细信息
        :param device_id: 设备id
        :return: 结果对象
        """
        rs = get_result_model()
        fire_rs = self.device_firware_version_read(device_id)
        if fire_rs['status'] == 0 and fire_rs['data']:
            version =  fire_rs['data']['version']
            brief = fire_rs['data']['brief']
            rs['data']['deviceId'] = device_id
            rs['data']['version'] = version
            rs['data']['brief'] = brief
            # mcu
            aux_card = fire_rs['data']['auxcard']
            rs['data']['auxcard'] = aux_card
            # 主控
            main_control = fire_rs['data']['mainControl']
            rs['data']['mainControl'] = main_control
            # inputCard
            input_cards = fire_rs['data']['inputCard']
            for input_card in input_cards:
                model_id = input_card['modelId']
                card_name =  CardName.get_card_name(model_id)
                hardware_version = input_card['hardver']
                input_card['hardwareVersion'] = hardware_version
                input_card['cardName'] = card_name
            rs['data']['inputCard'] = input_cards
            # outputCard
            output_cards = fire_rs['data']['outputCard']
            for output_card in output_cards:
                model_id = output_card['modelId']
                category = CardCategory.get_card_category(model_id)
                if category != 0:
                    card_name = CardName.get_card_name(model_id)
                    hardware_version = output_card['hardver']
                    output_card['hardwareVersion'] = hardware_version
                    output_card['cardName'] = card_name
            rs['data']['outputCard'] = output_cards
        return rs

    def read_device_background_color(self, device_id):
        """读取设备背景颜色"""
        return device_api.device_background_color_read(device_id)

    def write_device_background_color(self, device_id, data):
        """设置设备背景颜色"""
        return device_api.device_background_color_write(device_id, data)

    def read_device_font_list(self, device_id):
        """
        读取设备字体列表
        :param device_id: 设备id
        :return: 结果对象
        """
        return device_api.read_device_font_list(device_id)

    def write_device_font_delete(self, device_id, data):
        """
       读取设备字体列表
       :param device_id: 设备id
       :param data: 设备删除参数
       :return: 结果对象
       """
        return device_api.device_font_delete(device_id, data)

    def device_font_create(self, device_id,):
        # 上传图片
        up_rs = upload_file(UploadFileType.Font,)
        if not up_rs["bln"]:
            return get_result_model(ResInfo.Upload_File_Fail)

        # 上传成功后：先判断资源是否够用，然后再将图片转成dta文件,最后再在设备上创建BKG
        path_nm = up_rs["pathNm"]
        # 获得图片尺寸，判断资源是否够用

        # 发送中间件数据
        data = {
            "deviceId": device_id,
            "path": path_nm}
        rs = device_api.device_font_create(device_id, data)
        # 失败删除源文件
        if rs['status'] != 0:
            remove_file(path_nm)

        return rs

    def device_version_comparison_data_read(self, device_id):
        """
        获取程序版本号对比数据
        :param device_id: 设备号
        :return: 参考协议文档：0x0C - key: deviceFirwareVersion
        """
        v_rs = self.read_card_version()
        if v_rs['status'] != 0:
            return v_rs
        card_info = []
        rs = device_api.read_firware_version(device_id)
        if rs['status'] == 0 and rs['data']:
            version_info = rs['data']
            if 'mainControl' in version_info:
                 card_info.append(self._build_card_info(version_info['mainControl'], v_rs['data']))
            if 'auxcard' in version_info:
                card_info.append(self._build_card_info(version_info['auxcard'], v_rs['data']))
            if 'backboard' in version_info:
                card_info.append(self._build_card_info(version_info['backboard'], v_rs['data']))
            if 'inputCard' in version_info and version_info['inputCard']:
                for card in version_info['inputCard']:
                    card_info.append(self._build_card_info(card, v_rs['data']))
            if 'outputCard' in version_info and version_info['outputCard']:
                for card in version_info['outputCard']:
                    card_info.append(self._build_card_info(card, v_rs['data']))
            if 'mvrCard' in version_info:
                card_info.append(self._build_card_info(version_info['mvrCard'], v_rs['data']))
        rs['data'] = card_info
        return rs

    def device_fiber_mode_write(self, device_id, data):
        """
        设置4光纤卡的模式
        :param device_id: 设备号
        :param data: data
        :return:
        """
        return device_api.device_fiber_mode_write(device_id, data)

    def _build_card_info(self, card, array):
        model_id = card['modelId']
        version = card['version']
        # inter = {
        #     "ARM": "",
        #     "FPGA": "",
        #     "MCU": "",
        #     "FPGA_A": "",
        #     "FPGA_B": ""
        # }
        # if 'ARM' in card:
        #    inter['ARM'] = card['ARM']['internalver']
        # if 'FPGA' in card:
        #     inter['FPGA'] = card['FPGA']['internalver']
        # if 'MCU' in card:
        #     inter['MCU'] = card['MCU']['internalver']
        # if 'FPGA_A' in card:
        #     inter['FPGA_A'] = card['FPGA_A']['internalver']
        # if 'FPGA_B' in card:
        #     inter['FPGA_B'] = card['FPGA_B']['internalver']

        if 'slotId' in card:
            slot_id = card['slotId']
        else:
            slot_id = None
        card_file_name = ""
        support_version = ""
        flag = False
        if array:
            for card_info in array:
                if model_id in card_info:
                    support_version = card_info[model_id][1]
                    card_file_name = card_info[model_id][0]
                if version in support_version:
                    flag = True
                else:
                    flag = False
        card_info_dict = {
            "slotName": card_file_name,
            "modelId": model_id,
            "slotId": slot_id,
            "version": support_version,
            "slotVersion": version,
            "flag": flag,
            # "internalver":inter
        }

        return card_info_dict

    def read_card_version(self):
        """
        读取版本信息
        :return: 版本信息
        dict = {
            modleId: [name, version],
            modleId2:[name, version]
        }
        """
        rs = get_result_model()
        card_version_array = []
        try:
            f = open(SysSettings.Middleware_All_Slot_Version_Path, 'r')  # 以读方式打开文件
            result = list()
            for line in f.readlines():  # 依次读取每行
                line = line.strip()  # 去掉每行头尾空白
                if not len(line) or line.startswith('#'):  # 判断是否是空行或注释行
                    continue  # 是的话，跳过不处理
                result.append(line)  # 保存
            f.close()
            for i in result:
                value = i.strip().split(",")
                model_id = int(value[1].strip())
                file_name = value[0].strip()
                version = value[2].strip()
                dict_info = {model_id: [file_name, version]}
                card_version_array.append(dict_info)
        except Exception as e:
            logger.error("读取文件异常：" + str(e))
            return get_result_model(ResInfo.File_Not_Exists)
        rs['data'] = card_version_array
        return rs

    def input_backup_list_read(self, device_id):
        """
        获取输入源备份列表
        :param device_id:  设备Id
        :return: 结果对象
        """
        # 1. 获取当前输入源列表
        # 2. 删除backup 节点中 type =0 的相关的
        # rs = device_api.device_config_read(ModelConfKey.InputBackup)
        rs = device_api.input_backup_list_read(device_id)
        return rs

    def input_backup_write(self, device_id, enable, type, data):
        """
        输入源开启/关闭备份
        :param device_id:  设备Id
        :param enable:  开启/关闭
        :param data:  数据
        :return: 结果对象
        """
        return device_api.input_backup_write(device_id, enable, type, data)

    def input_backup_read(self, device_id):
        """
        输入源开启/关闭备份 回读
        :param device_id:  设备Id
        :param input_backup_enable:  开启/关闭
        :param data:  数据
        :return: 结果对象
        """
        return device_api.input_backup_read(device_id)

    def input_backup_update(self, device_id, data):
        """
        输入源备份添加/更新
        :param device_id:  设备Id
        :param input_backup_enable:  开启/关闭
        :param data:  数据
        :return: 结果对象
        """
        return device_api.input_backup_update(device_id, data)

    def input_backup_delete(self, device_id, data):
        """
        输入源备份删除
        :param device_id:  设备Id
        :param input_backup_enable:  开启/关闭
        :param data:  数据
        :return: 结果对象
        """
        return device_api.input_backup_delete(device_id, data)

    def check_auth_code(self, device_id, auth_code):
        device_rs = self.device_firware_version_read(device_id)
        if device_rs['status'] != 0:
            return device_rs
        sn = device_rs['data']['mainControl']['sn']
        hash_code = str_hash_code(SN_Prefix + sn)
        rs = auth_code_dao.get_auth_data(auth_code)
        if rs['status'] == 0 and rs['data']:
            return get_result_model(ResInfo.Auth_Code_Add_Fail)

        if check_password(str(hash_code), auth_code):
            id_code = ''.join(str(uuid.uuid1()).split('-'))
            rs = auth_code_dao.set_auth_code(sn, id_code, auth_code)
            if rs['status'] == 0:
                rs['data']['idCode'] = id_code
        else:
            rs = get_result_model(ResInfo.User_Verify_Code_Err)
        return rs

    def switch_http_read(self, device_id, key):
        return deviceInfodao.http_enable_read(device_id, key)

    def switch_http_write(self, device_id, key, enable):
        return deviceInfodao.http_enable_write(device_id, key, enable)

    def central_cmd_list_read(self, device_id):
        """
        获取中控命令列表
        :param device_id: 设备id
        :return: 结果对象
        """
        rs = device_api.central_cmd_list_read(device_id)
        if rs['status'] == 0:
            cmd_list = []
            for cmd in rs['data']['cmdList']:
                drs = device_api.central_cmd_detail_read(device_id, cmd['id'])
                if drs['status'] == 0 and drs['data']:
                    cmd_list.append(drs['data'])
            rs['data']['cmdList'] = cmd_list
        return rs

    def central_cmd_detail_read(self, device_id, cmd_id):
        """
        获取中控命令列表
        :param device_id: 设备id
        :return: 结果对象
        """
        return device_api.central_cmd_detail_read(device_id, cmd_id)

    def central_cmd_create_write(self, device_id, data):
        """
        中控命令创建
        """
        return device_api.central_cmd_create_write(device_id, data)

    def central_cmd_modify_write(self, device_id, data):
        """
        中控命令修改
        """
        return device_api.central_cmd_modify_write(device_id, data)

    def central_cmd_delete_write(self, device_id, data):
        """
        中控命令删除
        """
        return device_api.central_cmd_delete_write(device_id, data)

    def central_cmd_apply_write(self, device_id, data):
        """
        中控命令应用
        """
        return device_api.central_cmd_apply_write(device_id, data)

    def write_device_output_interface_mode(self, device_id, data):
        """
        设置输出接口的模式切换
        :param device_id: 设备id
        :param data: 设备参数
        :return: 结果对象
        """
        return device_api.write_device_output_interface_mode(device_id, data)

    def audio_card_detail_read(self, device_id, slot_id):
        """
        读取音频卡详情
        """
        return device_api.read_audio_card_detail(device_id, slot_id)

    def audio_card_attribute_write(self, device_id, slot_id, data):
        """
         设置音频卡模式
        """
        return device_api.write_audio_card_attribute(device_id, slot_id, data)

    def audio_interface_delay_time_write(self, device_id, slot_id, data):
        """
         设计音频接口延时
        """
        return device_api.write_audio_interface_delay_time( device_id, slot_id, data)

    def audio_input_audio_card_list(self, device_id):
        """
        读取所有音频卡的属性
        通道与源的属性
        :return: 结果对象
        """
        audio_list = []
        rs = device_api.read_audio_card_list(device_id)
        if rs['status'] != 0 or not rs['data']:
            rs = get_result_model()
            rs['data']['List'] = audio_list
            return rs
        for audio in rs['data']['audioList']:
            slot_id = audio['slotId']
            audio['inputId'] = 255
            detail_rs = self.audio_card_detail_read(device_id, slot_id)
            if detail_rs['status'] != 0 or not detail_rs['data']:
                continue
            audio['interfaces'] = detail_rs['data']['interfaces']
        from H_9U.service.input import input_svc
        input_rs = input_svc.read_list(device_id)
        if rs['status'] != 0 or not rs['data']:
            rs = get_result_model()
            rs['data']['List'] = audio_list
            return rs
        for item in  input_rs['data']['inputs']:
            if item['isSupportAudio'] == 1 and item['online'] == 1:
                input_id = item['inputId']
                audio_dict = {
                    "channelMode": 0,
                    "interfaces": [],
                    "slotId": 255,
                    "inputId": input_id,
                    "name": item['general']['name']
                }
                rs['data']['audioList'].append(audio_dict)
        return rs


    def genLock_status_read(self, device_id):
        """
        读取genlock的同步状态
        :return: 结果对象
        """
        screen_list = []
        screen_sdi_list = []
        rs = device_api.genLock_status_read(device_id)
        if rs['status'] != 0 or not rs['data']:
            return rs
        genLock_status = rs['data']['status']
        input_id = rs['data']['inputId']
        from H_9U.service.input import input_svc
        input_rs = input_svc.read_detail(device_id, input_id)
        if input_rs['status'] != 0 or not input_rs['data']:
            return input_rs
        input_name = ""
        if rs['data']['type'] == 1:
            input_name = input_rs['data']['general']['name']
        if genLock_status != 1 and genLock_status != 4:
            rs['data']['screenList'] = screen_list
            rs['data']['screenSdiList'] = screen_sdi_list
            rs['data']['name'] = input_name
            return rs
        # 查询当前屏幕列表是否 为空；为空则给出对应的
        from H_9U.service.screen import screen_svc
        screen_rs = screen_svc.read_list(device_id)
        if screen_rs['data'] and screen_rs['data'].get("screens", None):
            for screen in screen_rs['data'].get("screens", None):
                screen_id = screen["screenId"]
                detail_rs = screen_svc.read_detail(device_id, screen_id)
                if detail_rs['status'] != 0 or not detail_rs['data']:
                    continue
                # 同步源帧频与以下屏幕帧频差值过大 导致失败的屏幕
                if detail_rs['data']['genlock']['status'] == 1:
                    screen_dict = {
                        "name": detail_rs['data']['general']['name'],
                        "screenId":detail_rs['data']['screenId']
                    }
                    screen_list.append(screen_dict)
                if detail_rs['data']['genlock']['status'] == 4:
                    screen_dict = {
                        "name": detail_rs['data']['general']['name'],
                        "screenId": detail_rs['data']['screenId']
                    }
                    screen_sdi_list.append(screen_dict)
        if len(screen_list) > 0:
            screen_list.extend(screen_sdi_list)
            screen_sdi_list = []
        rs['data']['screenList'] = screen_list
        rs['data']['screenSdiList'] = screen_sdi_list
        rs['data']['name'] = input_name
        return rs

    def device_slot_eth_distance(self, device_id, slot_id, data):
        """
        设置4网口卡模式
        :param device_id: 设备id
        :param slotId: slotId
        :param data: 参数
        :return: 结果对象
        """
        return device_api.device_slot_eth_distance_write(device_id, slot_id, data)

    def write_port_mode(self, device_id, data):
        """
        读取光口卡模式
        :param device_id: 设备id
        :param slotId: slotId
        :return: 结果对象
        """
        return device_api.write_port_mode(device_id, data)


    def read_fiber_detail(self, device_id, slot_id):
        """
        读取光口卡详情
        :param device_id: 设备id
        :param slotId: slotId
        :return: 结果对象
        """
        return device_api.read_fiber_detail(device_id, slot_id)

    def import_sdp(self, device_id, data):
        up_rs = upload_file(UploadFileType.Sdp)
        # 判断文件上传是否成功
        if not up_rs["bln"]:
            return get_result_model(ResInfo.Upload_File_Fail)
        path_nm = up_rs["pathNm"]
        # 发送中间件数据
        param = {
            "deviceId": device_id,
            "filePath": path_nm,
            "type": data['type'],
            "parameter": data['parameter']
        }
        rs = device_api.sdp_create(device_id, param)
        if rs['status'] == 0:
            remove_file(path_nm)
        return rs

    def test(self):
        return device_api.test()

    def test_write(self,count):
        return device_api.test_write(count)

    def ip_displayed(self, device_id, is_displayed):
        return deviceInfodao.ipDisplayed(device_id, is_displayed)

    def ip_displayed_read(self, device_id):
        return deviceInfodao.ipDisplayedRead(device_id)

    def write_lcd_config(self, device_id, data):
        """
        读取光口卡模式
        :param device_id: 设备id
        :param slotId: slotId
        :return: 结果对象
        """
        rs = device_api.write_lcd_config(device_id, data)
        if rs['status'] != 0:
            return get_result_model(ResInfo.Device_Lcd_Setting_Err)
        return rs

    def read_lcd_config(self, device_id):
        """
		读取光口卡模式
		:param device_id: 设备id
		:param slotId: slotId
		:return: 结果对象
		"""
        rs = device_api.read_lcd_config(device_id)
        return rs


    def bright_mode_write(self, device_id, data):
        """
        设置亮度调节模式
        :param device_id:  设备Id
        :param data:  数据
        :return: 结果对象
        """
        return device_api.bright_mode_write(device_id, data)

    def bright_mode_read(self, device_id):
        """
        读取亮度调节模式
        :param device_id:  设备Id
        :return: 结果对象
        """
        return device_api.bright_mode_read(device_id)

    def read_test_pattern_border_status(self, device_id):
        """
		读取亮度调节模式
		:param device_id:  设备Id
		:return: 结果对象
		"""
        return device_api.read_test_pattern_border_status(device_id)

    def write_test_pattern_border_status(self, device_id, data):
        """
        读取亮度调节模式
        :param device_id:  设备Id
        :return: 结果对象
        """
        return device_api.write_test_pattern_border_status(device_id, data)

    def is_preset_id_write(self, is_preset_id):
        """
		读取光口卡模式
		:param device_id: 设备id
		:param slotId: slotId
		:return: 结果对象
		"""
        return deviceInfodao.is_preset_id_write(is_preset_id)

    def is_preset_id_read(self, device_id):
        """
		读取光口卡模式
		:param device_id: 设备id
		:param slotId: slotId
		:return: 结果对象
		"""
        return deviceInfodao.is_preset_id_read(device_id)

    def interface_capacity_write(self, device_id, slot_id, interface_id, data):
        """
        设置子卡接口的容量
        """

        # 读取卡槽信息
        rs = device_svc.device_slot_read(device_id, slot_id)
        if rs['status'] != 0 or not rs['data']:
            return rs
        slot_model_id = rs['data']['modelId']
        interface_list = rs['data']['interfaces']
        card_category = CardCategory.get_card_category(slot_model_id)
        if card_category in [CardCategory.H_2xHDMI20_2xDP12_I, CardCategory.H2xHDMIDPICard] and len(interface_list) == 4:
            if interface_id == 0:
                if interface_list[0]['functionType'] == FunctionType.Disable:
                    data['interfaceId'] = 1
            if interface_id == 1:
                if interface_list[2]['functionType'] != FunctionType.Disable:
                    data['interfaceId'] = 2
                else:
                    data['interfaceId'] = 3
        return device_api.write_device_interface_capacity(device_id, slot_id, interface_id, data)


    def read_conf(self, device_id, param):
        rs = get_result_model()
        data = {
            # 预监卡是否在线
            "mvrCardOnline"  : -1,
            # 是否展示场景Id
            "presetIdEnable" : -1,
            # 是否展示IPC列表的IP
            "ipcIpEnable"    : -1,
            # 是否拉取主辅码流
            "ipcStreamRule"  : -1,
            # 网络源分组数量
            "networkGroupCount" : 0,
            # 输入源分组数量
            "inputGroupCount" : 0,
            # 场景切换特效
            "presetEffect" : {},
            # 语言模式 0:英文 1:中文
            "languageMode" :-1,
        }
        mvr_is_online_rs = self.mvr_is_online(device_id)
        if mvr_is_online_rs['status'] == 0:
            data['mvrCardOnline'] = mvr_is_online_rs['data']['online']

        preset_id_rs  = self.is_preset_id_read(device_id)
        if preset_id_rs['status'] == 0:
            data["presetIdEnable"] = preset_id_rs['data']['is_preset_id']

        ip_displayed_rs = self.ip_displayed_read(device_id)
        if ip_displayed_rs['status'] == 0:
            data['ipcIpEnable'] = ip_displayed_rs['data']

        ipc_stream_rule_rs = ipc_svc.read_ipc_stream_rule(device_id)
        if ipc_stream_rule_rs['status'] == 0:
            data['ipcStreamRule'] = ipc_stream_rule_rs['data']

        preset_config_read_rs = preset_svc.preset_config_read(device_id)
        if preset_config_read_rs['status'] == 0:
            data['presetEffect'] = preset_config_read_rs['data']

        language_rs = self.read_device_language(device_id)
        if language_rs['status'] == 0 :
            data['languageMode'] = language_rs['data']['languageMode']
        rs['data'] = data
        return rs

device_svc = DeviceSvc()
