"""
Created on 2019年6月28日
@author: 刘益宗
对应接口文档output.
返回值统一为 {data:JSON结构体, msg:"", status:0}
"""

import os

from flask import Blueprint, send_from_directory, g
from flask.globals import request
from flask.json import jsonify

from H_9U.api.filter.filter import _filter_output_data, role_filter_data_func
from H_9U.service.output import output_svc
from H_9U.util.common import valid_json, valid_params
from H_9U.models.result import get_result_model, ResInfo
from H_9U.models.sysconst import CardType, InterfaceType, ModelId, CardCategory, DataType, PermissionCode
from H_9U.util.logwrapper import api_log
from H_9U.api.websender import push_data, push_data_org
from H_9U.models.syncdataname import SyncDataName
from H_9U.util.permission_valid import valid_function_permission

output_bp = Blueprint('output', __name__)


@output_bp.route('/readAllList', methods=['GET', 'POST'])
@api_log
def read_all_list():
    """
    读取输出枚举信息
    :return: 参考文档readALLList
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    user = g.user
    role_id = user['roleId']
    rs = output_read_all_list(device_id)
    if rs['status'] == 0:
        role_rs, role_type = role_filter_data_func(role_id)
        rs['data'] = _filter_output_data(role_type, role_rs['data']['outputList'], rs['data'])
    return jsonify(rs)


@output_bp.route('/writeGeneral', methods=['GET', 'POST'])
@api_log
def output_write_general():
    """
    设置常规信息
    :return: 参考文档writeGeneral
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'outputId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, output_id = values
    # 接口校验
    from H_9U.util.permission_valid import valid_function_permission
    rs = valid_function_permission(PermissionCode.Screen_Configuration, DataType.Output_Type, output_id)
    if rs['status'] != 0:
        return jsonify(rs)

    rs = output_svc.write_general(device_id, output_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Output_Write_General, params)
    return jsonify(rs)


@output_bp.route('/writeTestPattern', methods=['GET', 'POST'])
@api_log
def output_test_pattern_write():
    """
    设置指定输出的测试画面信息
    :return:参考文档writeTestPattern
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'outputId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, output_id = values

    # 接口校验
    rs = valid_function_permission(PermissionCode.Screen_Configuration, DataType.Output_Type, output_id)
    if rs['status'] != 0:
        return jsonify(rs)

    rs = output_write_test_pattern(device_id, output_id, params)
    if rs['status'] == 0:
        push_data(output_read_all_list, SyncDataName.Output_Read_List, device_id)
    return jsonify(rs)


@output_bp.route('/writeImageQuality', methods=['GET', 'POST'])
@api_log
def output_image_quality_write():
    """
    设置画质
    :return:参考文档writeImageQuality
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'outputId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, output_id = values

    # 接口校验
    rs = valid_function_permission(PermissionCode.Screen_Configuration, DataType.Output_Type, output_id)
    if rs['status'] != 0:
        return jsonify(rs)

    rs = output_write_image_quality(device_id, output_id, params)
    if rs['status'] == 0:
        push_data(output_read_all_list, SyncDataName.Output_Read_List, device_id)
    return jsonify(rs)


@output_bp.route('/writeGamma', methods=['GET', 'POST'])
@api_log
def output_gamma_write():
    """
    设置Gamma
    :return:参考文档writeGamma
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'outputId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, output_id = values

    # 接口校验
    rs = valid_function_permission(PermissionCode.Screen_Configuration, DataType.Output_Type, output_id)
    if rs['status'] != 0:
        return jsonify(rs)

    rs = output_write_gamma(device_id, output_id, params)
    if rs['status'] == 0:
        push_data(output_read_all_list, SyncDataName.Output_Read_List, device_id)
    return jsonify(rs)


@output_bp.route('/writeTiming', methods=['GET', 'POST'])
@api_log
def output_write_timing():
    """
    设置timing
    :return:  参考文档writeTiming
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'outputIdList')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, output_list = values

    for output_id in output_list:
        # 接口校验
        rs = valid_function_permission(PermissionCode.EDID_Edit, DataType.Output_Type, output_id)
        if rs['status'] != 0:
            return jsonify(rs)

    from H_9U.api.screen import screen_list
    screen_rs = screen_list(device_id)
    if screen_rs['status'] != 0:
        return jsonify(get_result_model(ResInfo.Middle_Data_Err))

    for screen in [s for s in screen_rs['data'] if s['presetPoll']['enable'] == 1]:
        if list(set([x['outputId'] for x in screen['outputMode']['screenInterfaces']]) & set(output_list)):
            return jsonify(get_result_model(ResInfo.Screen_Is_Preset_Polling))

    rs = output_svc.write_timing(device_id, output_list, params)
    if rs['data']:
        push_data(output_read_all_list, SyncDataName.Output_Read_List, device_id)
        push_data(screen_list, SyncDataName.Screen_Read_All_List, device_id)
        push_data(screen_list, SyncDataName.Output_Timing_Changed, device_id)
    return jsonify(rs)


@output_bp.route('/readFixedPosition', methods=['GET', 'POST'])
@api_log
def output_read_fixed_position():
    """
    读取输出定位
    :return: 参考文档readFixedPosition
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    return jsonify(output_svc.read_fixed_position(device_id))


@output_bp.route('/writeFixedPosition', methods=['GET', 'POST'])
@api_log
def output_write_fixed_position():
    """
    读取最佳输出定位
    :return: 参考文档readFixedPosition
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    return jsonify(output_svc.write_fixed_position(device_id, params))


@output_bp.route('/readList', methods=['GET', 'POST'])
@api_log
def output_read_list():
    """
    读取输出列表，简化字段
    :return: 参考文档 readList
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    rs = output_read_all_list(device_id)
    outputs = []
    if rs['status'] == 0 and rs['data']:
        for output_info in rs['data']:
            outputs.append(
                {
                    'deviceId': device_id,
                    'outputId': output_info['outputId'],
                    'slotId': output_info['slotId'],
                    'interfaceId': output_info['interfaceId'],
                    'interfaceType': output_info['interfaceType'],
                    'iSignal': output_info['iSignal'],
                    'general': output_info['general'],
                    'resolution': output_info['resolution']
                }
            )
    rs['data'] = outputs
    return jsonify(rs)


@output_bp.route('/exportEDID', methods=['GET', 'POST'])
def output_edid_export():
    """
    导入edie文件
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'outputId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))

    device_id, output_id = values

    # 接口校验
    rs = valid_function_permission(PermissionCode.EDID_Edit, DataType.Output_Type, output_id)
    if rs['status'] != 0:
        return jsonify(rs)

    rs = output_svc.output_edid_export(int(device_id), int(output_id))
    if rs['status'] == 0 and rs['data']:
        file_name = rs['data']['file']['fileName']
        if not os.path.exists(file_name):
            return jsonify(get_result_model(ResInfo.File_Not_Exists))
        path, name = os.path.split(file_name)
        return send_from_directory(path, name, as_attachment=True)
    return jsonify(rs)


@output_bp.route('/writePayload', methods=['GET', 'POST'])
def output_write_payload():
    """
    设置二合一卡带载面积
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'outputIds')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))

    device_id, output_ids = values

    for output_id in output_ids:
        # 接口校验
        rs = valid_function_permission(PermissionCode.EDID_Edit, DataType.Output_Type, output_id)
        if rs['status'] != 0:
            return jsonify(rs)

    from H_9U.api.screen import screen_list
    screen_rs = screen_list(device_id)
    if screen_rs['status'] != 0:
        return jsonify(get_result_model(ResInfo.Middle_Data_Err))

    rs = output_svc.output_write_payload(device_id, output_ids, params)
    if rs['status'] == 0:
        push_data(output_read_all_list, SyncDataName.Output_Read_List, device_id)
        push_data(screen_list, SyncDataName.Screen_Read_All_List, device_id)
        push_data(screen_list, SyncDataName.Output_Timing_Changed, device_id)

    return jsonify(rs)


@output_bp.route('/write4KTiming', methods=['GET', 'POST'])
def output_write_4k_iming():
    """
    设置4K输出卡带载面积
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'outputIds')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))

    device_id, output_ids = values

    for output_id in output_ids:
        # 接口校验
        rs = valid_function_permission(PermissionCode.EDID_Edit, DataType.Output_Type, output_id)
        if rs['status'] != 0:
            return jsonify(rs)

    from H_9U.api.screen import screen_list
    screen_rs = screen_list(device_id)
    if screen_rs['status'] != 0:
        return jsonify(get_result_model(ResInfo.Middle_Data_Err))

    rs = output_svc.output_write_4k_timing(device_id, output_ids, params)
    if rs['status'] == 0:
        push_data(output_read_all_list, SyncDataName.Output_Read_List, device_id)
        push_data(screen_list, SyncDataName.Screen_Read_All_List, device_id)
        push_data(screen_list, SyncDataName.Output_Timing_Changed, device_id)

    return jsonify(rs)


@output_bp.route('/writeSdiTiming', methods=['GET', 'POST'])
def output_write_sdi_timing():
    """
    设置光口卡timing
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'outputIdList')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))

    device_id, output_ids = values

    for output_id in output_ids:
        # 接口校验
        rs = valid_function_permission(PermissionCode.EDID_Edit, DataType.Output_Type, output_id)
        if rs['status'] != 0:
            return jsonify(rs)

    from H_9U.api.screen import screen_list
    screen_rs = screen_list(device_id)
    if screen_rs['status'] != 0:
        return jsonify(get_result_model(ResInfo.Middle_Data_Err))

    rs = output_svc.output_write_sdi_timing(device_id, output_ids, params)
    if rs['status'] == 0:
        push_data(output_read_all_list, SyncDataName.Output_Read_List, device_id)
        push_data(screen_list, SyncDataName.Screen_Read_All_List, device_id)
        push_data(screen_list, SyncDataName.Output_Timing_Changed, device_id)

    return jsonify(rs)


def output_read_all_list(device_id):
    """
    读取列表
    :param device_id: 设备Id
    :return: 结果对象
    """
    rs = output_svc.read_list(device_id)
    if rs["status"] != 0 or not rs['data']:
        return rs
    from H_9U.api.device import device_detail_without_cache
    device = device_detail_without_cache(device_id)
    slots = []
    if device['status'] == 0:
        slots = device['data']['slotList']
    # 循环输入信息
    from H_9U.api.screen import screen_freezed_output
    screen_data = screen_freezed_output(device_id)
    freeze_list = screen_data['data']['freeze']
    ftb_list = screen_data['data']['ftb']
    poll_list = screen_data['data']['presetPoll']
    for item in rs['data']['outputs']:
        slot_id = item['slotId']
        interface_id = item['interfaceId']
        output_svc.build_output_data(device_id, item)
        i_signal = 0
        is_used = 0
        is_edid = 0
        model_id = ModelId.Not_Online
        slot = [s for s in slots if s['slotId'] == slot_id][0]
        if slot['cardType'] != CardType.NoCard:
            model_id = slot['modelId']
            item['cardCategory'] = CardCategory.get_card_category(model_id)
            if model_id == ModelId.Output_4_Fiber_Card:
                item['SenderColorDepth'] = slot['SenderColorDepth']
            interface = [i for i in slot['interfaces'] if i['interfaceId'] == interface_id][0]
            i_signal = interface['iSignal']
            is_used = interface['isUsed']
            is_edid = InterfaceType.allow_edid_setting(interface['interfaceType'])
        item['iSignal'] = i_signal
        item['isUsed'] = is_used
        item['isEDIDSetting'] = is_edid
        item['modelId'] = model_id
        item['freeze'] = 1 if item['outputId'] in freeze_list else 0
        item['ftb'] = 1 if item['outputId'] in ftb_list else 0
        item['presetPoll'] = 1 if item['outputId'] in poll_list else 0
    rs['data'] = rs['data']['outputs']
    return rs


def output_write_image_quality(device_id, output_id, params):
    """
    写输出画质
    :param device_id: 设备Id
    :param output_id:  输出Id
    :param params:  画质参数
    :return:  结果对象
    """
    rs = output_svc.write_image_quality(device_id, output_id, params)
    return rs


def output_write_gamma(device_id, output_id, data):
    """
    output设置gamma数据
    :param device_id: 设备id
    :param output_id: output id
    :param data:  gamma数据
    :return: 结果对象
    """
    return output_svc.write_gamma(device_id, output_id, data)


def output_write_test_pattern(device_id, output_id, data):
    """
    设置output测试画面
    :param device_id: 设备Id
    :param output_id: 输出id
    :param data: 测试画面数据
    :return: 结果对象
    """
    return output_svc.write_test_pattern(device_id, output_id, data)


def output_original_detail(device_id, output_id):
    """
    读取输出详细信息
    :param device_id: 设备Id
    :param output_id:  输出Id
    :return: 输出详细信息
    """
    return output_svc.output_read_detail(device_id, output_id)


def output_write_timing_original(device_id, output_id, data):
    return output_svc.write_timing_original(device_id, output_id, data)


def output_write_general_two(device_id, output_id, data):
    return output_svc.write_general(device_id, output_id, data)


def get_slot_by_output_id(input_id):
    """
    根据输出源id获取槽位号
    :return:
    """
    slot_list = []
    device_id = 0
    detail_rs = output_svc.output_read_detail(device_id, input_id)
    if detail_rs['status'] == 0 and detail_rs['data']:
        detail_slot_id = detail_rs['data']['slotId']
        slot_list = [detail_slot_id]
    return slot_list