"""
Created on 2019年6月28日
@author: 刘益宗
场景接口.
参考前端接口文档中Preset部分
返回值统一为 {data:JSON结构体, msg:"", status:0}
"""
from flask import Blueprint
from flask.globals import request, g
from flask.json import jsonify
import time

from H_9U.service.device import device_svc
from H_9U.service.preset import preset_svc
from H_9U.util.common import valid_json, valid_params
from H_9U.models.result import get_result_model, ResInfo
from H_9U.util.logwrapper import api_log
from H_9U.api.websender import push_data, push_data_org
from H_9U.models.syncdataname import SyncDataName
from H_9U.models.sysconst import LayerType, ConnectCapacity, PermissionCode
from H_9U.api.input import input_get_connect_capacity, input_original_detail
from H_9U.util.permission_valid import get_code_list

preset_bp = Blueprint('preset', __name__)
# _show_preset_id = 0


@preset_bp.route('/readDetail', methods=['GET', 'POST'])
@api_log
def preset_read_detail():
    """
    读取指定场景的详细信息
    :return:
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId', 'presetId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, preset_id = values

    return jsonify(preset_svc.preset_detail_read(device_id, screen_id, preset_id))


@preset_bp.route('/createPreset', methods=['GET', 'POST'])
@api_log
def preset_create():
    """
    创建场景
    :return: 参考接口文档createPreset
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    params["createTime"] = str(time.time())

    rs = preset_svc.preset_create(device_id, screen_id, params)
    if rs['status'] == 0:
        preset_id = rs['data']['id']
        detail_rs = preset_svc.preset_detail_read(device_id, screen_id, preset_id)
        if detail_rs['status'] == 0:
            push_data_org(SyncDataName.Preset_Add, detail_rs['data'])
            rs = detail_rs
    return jsonify(rs)


@preset_bp.route('/overWrite', methods=['GET', 'POST'])
@api_log
def preset_overwrite():
    """
    创建场景
    :return: 参考接口文档createPreset
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId', "presetId")
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, preset_id = values
    params["createTime"] = str(time.time())
    params["overWrite"] = 1

    rs = preset_svc.preset_create(device_id, screen_id, params)
    if rs['status'] == 0:
        detail_rs = preset_svc.preset_detail_read(device_id, screen_id, preset_id)
        if detail_rs['status'] == 0:
            push_data_org(SyncDataName.Preset_Over_Write, detail_rs['data'])
            rs = detail_rs
    return jsonify(rs)


@preset_bp.route('/copy', methods=['GET', 'POST'])
@api_log
def preset_copy():
    """
    创建场景
    :return: 参考接口文档createPreset
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId', 'presetId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, preset_id = values

    params["createTime"] = str(time.time())
    rs = preset_svc.preset_copy(device_id, screen_id, params)
    if rs['status'] == 0:
        preset_id = rs['data']['id']
        rs = preset_svc.preset_detail_read(device_id, screen_id, preset_id)
        if rs['status'] == 0:
            push_data_org(SyncDataName.Preset_Add, rs['data'])
    return jsonify(rs)


@preset_bp.route('/deletePreset', methods=['GET', 'POST'])
@api_log
def preset_delete():
    """
    删除场景
    :return: 结果对象，status==0表示成功
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId', 'presetId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, preset_id = values

    rs = preset_svc.preset_delete(device_id, screen_id, preset_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Preset_Delete, {'deviceId': device_id, 'screenId': screen_id, 'presetId': preset_id})
        push_data(preset_group_list, SyncDataName.Preset_Group_List, device_id, screen_id)
    return jsonify(rs)


@preset_bp.route('/deleteAll', methods=['GET', 'POST'])
@api_log
def preset_delete_all():
    """
    清空场景
    :return: 删除场景列表数据
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values

    rs = preset_svc.preset_delete_all(device_id, screen_id)
    if rs['status'] == 0:
        push_data(preset_read_list, SyncDataName.Preset_Read_List, device_id, screen_id)
        push_data(preset_group_list, SyncDataName.Preset_Group_List, device_id, screen_id)
        push_data_org(SyncDataName.Preset_Clear, params)
    return jsonify(rs)


@preset_bp.route('/writeGeneral', methods=['GET', 'POST'])
@api_log
def preset_write_general():
    """
    设置基本信息
    :return:参考前端接口文档writeGeneral
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId', 'presetId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, preset_id = values

    rs = preset_svc.preset_general_write(device_id, screen_id, preset_id, params)
    if rs['status'] == 0:
        push_data(preset_svc.preset_detail_read, SyncDataName.Preset_Write_General, device_id, screen_id, preset_id)
        push_data_org(SyncDataName.Preset_Group_Write_General, params)
    return jsonify(rs)


@preset_bp.route('/play', methods=['GET', 'POST'])
@api_log
def preset_play():
    """
    播放场景
    :return: 参考前端接口文档play
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId', 'presetId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, preset_id = values

    rs = preset_svc.preset_play_write(device_id, screen_id, preset_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Preset_Play, params)
        from H_9U.api.screen import screen_detail
        detail = screen_detail(device_id, screen_id)
        if detail['status'] == 0 and detail['data']:
            osd_data = {
                "deviceId": device_id,
                "screenId": screen_id,
                "Osd": detail['data']['Osd'],
                "OsdImage": detail['data']['OsdImage']
            }
            push_data_org(SyncDataName.Screen_Write_OSD, osd_data)
            bkg_data = {
                "deviceId": device_id,
                "imageUrl": detail['data']['Bkg']['imgUrl'],
                "bkgId": detail['data']['Bkg']['bkgId'],
                "screenId": screen_id,
                "enable": detail['data']['Bkg']['enable']
            }
            push_data_org(SyncDataName.Screen_Write_BKG, bkg_data)
    return jsonify(rs)


@preset_bp.route('/readPlay', methods=['GET', 'POST'])
@api_log
def preset_play_read():
    """
    读取屏幕当前播放场景
    :return: 场景Id
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values

    return jsonify(preset_svc.preset_play_screen(device_id, screen_id))


@preset_bp.route('/groupList', methods=['GET', 'POST'])
@api_log
def group_list():
    """
    获取全部场景分组列表
    :return:参考前端接口文档场景分组信息
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values

    rs = preset_group_list(device_id, screen_id)
    return jsonify(rs)


def preset_group_list(device_id, screen_id):
    """
    场景分组下所有场景的详细信息
    :param device_id:  设备Id
    :param screen_id:  屏幕Id
    :return: 参考接口文档groupList
    """
    # 读取全部场景分组
    rs = preset_svc.preset_group_list(device_id, screen_id)
    if rs['status'] != 0 or not rs['data']:
        return rs
    group_list = []
    # 循环分组， 读取分组详细信息
    for group in rs['data']['presetGroups']:
        group_id = group['presetGroupId']
        group_name = group['name']
        # 读取分组详细信息
        group_rs = preset_svc.preset_group_node_list(device_id, screen_id, group_id)
        if group_rs['status'] != 0:
            continue
        group_rs['data']['name'] = group_name
        # 循环场景
        preset_list = []
        # 循环分组下场景
        __build_preset_data(device_id, group_rs, preset_list, screen_id)
        group_rs['data']['presets'] = preset_list
        group_list.append(group_rs['data'])
    rs['data']['presetGroups'] = group_list
    return rs


def __build_preset_data(device_id, group_rs, preset_list, screen_id):
    for preset in group_rs['data']['presets']:
        preset_id = preset['presetId']
        # 读取场景详细信息
        detail_rs = preset_svc.preset_detail_read(device_id, screen_id, preset_id)
        if detail_rs['status'] != 0 or not detail_rs['data']:
            continue
            # 循环场景下图层
        detail = detail_rs['data']
        detail['zorder'] = preset['zorder']
        detail['time'] = preset['time']
        # 读取场景下图层信息
        for layer in detail['layers']:
            input_id = layer['source']['inputId']
            crop_id = layer['source']['cropId']
            input_rs = input_original_detail(device_id, input_id)
            if input_rs['status'] != 0 or not input_rs['data']:
                continue
            connect_capacity = input_rs['data']['general']['connectCapacity']
            name = input_rs['data']['general']['name']
            if 0 <= crop_id < 255:
                crop = [x for x in input_rs['data']['crops'] if x['cropId'] == crop_id]
                if crop:
                    name = crop[0]['name']
            layer['source']['connectCapacity'] = connect_capacity
            layer['source']['name'] = name
        preset_list.append(detail)


@preset_bp.route('/groupCreate', methods=['GET', 'POST'])
@api_log
def preset_group_create():
    """
    创建场景分组
    设备id， 屏幕id，groupname, presest_id_list
    :return: 屏幕分组创建结果Id
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values

    if not params.get('presets') :
        return jsonify(get_result_model(ResInfo.Params_Error))

    rs = preset_svc.preset_group_create(device_id, screen_id, params)
    if rs['status'] == 0 and rs['data']:
        params['presetGroupId'] = rs['data']['id']
        group_name = params['name']
        # 查询场景分组下的场景列表
        group_rs = preset_svc.preset_group_node_list(device_id, screen_id, params['presetGroupId'])
        if group_rs['status'] == 0:
            group_rs['data']['name'] = group_name
            push_data_org(SyncDataName.Preset_Group_Add, group_rs['data'])
    return jsonify(rs)


@preset_bp.route('/groupDelete', methods=['GET', 'POST'])
@api_log
def preset_group_delete():
    """
    删除场景分组
    :return: 参考结果文档groupDelete
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId', 'presetGroupId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, group_id = values

    rs = preset_svc.preset_group_delete(device_id, screen_id, group_id)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Preset_Group_Delete, params)
    return jsonify(rs)


@preset_bp.route('/groupNodeList', methods=['GET', 'POST'])
@api_log
def preset_group_node_list():
    """
    读取场景分组节点列表
    :return: 参考前端接口文档groupNodeList
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId', 'presetGroupId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, group_id = values

    return jsonify(preset_svc.preset_group_node_list(device_id, screen_id, group_id))


@preset_bp.route('/groupUpdate', methods=['GET', 'POST'])
@api_log
def preset_group_update():
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId', 'presetGroupId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, preset_group_id = values

    if not params.get('presets') :
        return jsonify(get_result_model(ResInfo.Params_Error))

    rs = preset_svc.preset_group_update(device_id, screen_id, params)
    if rs['status'] == 0:
        group_name = params['name']
        # 查询场景分组下的场景列表
        group_rs = preset_svc.preset_group_node_list(device_id, screen_id, preset_group_id)
        if group_rs['status'] == 0:
            group_rs['data']['name'] = group_name
        push_data_org(SyncDataName.Preset_Group_Update, group_rs['data'])
    return jsonify(rs)


@preset_bp.route('/presetPoll', methods=['GET', 'POST'])
@api_log
def preset_poll():
    """
    场景轮询
    :return: 参考前端接口文档 presetPoll
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    rs = preset_svc.preset_poll(device_id, screen_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Preset_Poll, params)
        from H_9U.api.output import output_read_all_list
        push_data(output_read_all_list, SyncDataName.Output_Read_List, device_id)
    return jsonify(rs)


@preset_bp.route('/readList', methods=['GET', 'POST'])
@api_log
def read_list():
    """
    读取场景列表不含分组信息
    :return: 参考前端接口文档 readList
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values

    rs = preset_read_list(device_id, screen_id)
    return jsonify(rs)


@preset_bp.route('/readPresetPoll', methods=['GET', 'POST'])
@api_log
def preset_poll_read():
    """
    场景轮询读取信息
    :return: 参考前端接口文档 readPresetPoll
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values

    return jsonify(read_preset_pool(device_id, screen_id))


def read_preset_pool(device_id, screen_id):
    return preset_svc.preset_poll_read(device_id, screen_id)


@preset_bp.route('/nontimeSave', methods=['GET', 'POST'])
@api_log
def preset_nontime_create():
    """
    非实时上屏保存场景
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    preset_id = None
    if 'layers' in params:
        for layer in params['layers']:
            # 图层类型
            layer['general']['type'] = LayerType.Default
            # 输入接口容量：
            input_id = layer['source']['inputId']
            rs = input_get_connect_capacity(device_id, input_id)
            layer['general']['sizeType'] = ConnectCapacity.Default
            if rs['status'] == 0:
                layer['general']['sizeType'] = rs['data']['connectCapacity']
    if 'presetId' in params:
        preset_id = params['presetId']
        update = True
    else:
        update = False
    rs = preset_svc.preset_nontime_save(device_id, screen_id, params)
    if rs['status'] != 0:
        return jsonify(rs)
    if update:
        sync_name = SyncDataName.Preset_Over_Write
    else:
        preset_id = rs['data']['id']
        sync_name = SyncDataName.Preset_Add
    detail_rs = preset_svc.preset_detail_read(device_id, screen_id, preset_id)
    if detail_rs['status'] == 0:
        push_data_org(sync_name, detail_rs['data'])
        rs = detail_rs
    return jsonify(rs)


@preset_bp.route('/readEffect', methods=['GET', 'POST'])
@api_log
def preset_config_read():
    """
    读取场景切换特效
    :return:
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    return jsonify(preset_svc.preset_config_read(device_id))


@preset_bp.route('/writeEffect', methods=['GET', 'POST'])
@api_log
def preset_config_write():
    """
    读取场景切换特效
    :return:
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    user = g.user
    role_id = user['roleId']
    # 增加接口校验
    role_rs = get_code_list(PermissionCode.Device_Other, role_id)
    if role_rs['status'] != 0:
        return jsonify(role_rs)
    rs = preset_svc.preset_config_write(device_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Preset_Effect, params)
    return jsonify(rs)


@preset_bp.route('/readShowId', methods=['GET', 'POST'])
@api_log
def preset_show_id_read():
    """读取是否显示场景Id"""
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    preset_id_rs = device_svc.is_preset_id_read(device_id)
    rs = get_result_model()
    if preset_id_rs['status'] == 0:
        rs['data'] = {
            "deviceId": device_id,
            "enable": preset_id_rs['data']['is_preset_id']
        }
    return jsonify(rs)


@preset_bp.route('/writeShowId', methods=['GET', 'POST'])
@api_log
def preset_show_id_write():
    """设置是否显示场景Id"""
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params,'enable')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    enable = values
    # global _show_preset_id
    # _show_preset_id = enable
    # print(_show_preset_id)
    rs = device_svc.is_preset_id_write(enable)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Preset_Show_Id, params)
    return jsonify(rs)


def preset_read_list(device_id, screen_id):
    """
    读取屏幕下场景
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :return: 场景列表
    """
    rs = preset_svc.preset_list_read(device_id, screen_id)
    return rs


def preset_origin_list(device_id, screen_id):
    return preset_svc.preset_origin_list_read(device_id, screen_id)


def preset_detail(device_id, screen_id, preset_id):
    return preset_svc.preset_detail_read(device_id, screen_id, preset_id)


@preset_bp.route('/collectionCreate', methods=['GET', 'POST'])
@api_log
def preset_collection_create():
    """
    创建场景组合
    http://yapi.novatools.vip/project/613/interface/api/157210
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId','screenList','name')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))

    device_id = params.get('deviceId')
    rs = preset_svc.preset_collection_create(device_id, params)
    if rs['status'] == 0 and rs['data']:
        params['presetCollectionId'] = rs['data']['id']
        push_data_org(SyncDataName.Preset_Collection_Create, params)
        rs['data'] = params
    return jsonify(rs)


@preset_bp.route('/collectionUpdate', methods=['GET', 'POST'])
@api_log
def preset_collection_update():
    """
    编辑场景组合
    http://yapi.novatools.vip/project/613/interface/api/157222
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    if not params.get('screenList'):
        return jsonify(get_result_model(ResInfo.Params_Error))

    rs = preset_svc.preset_collection_update(device_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Preset_Collection_Update, params)
    return jsonify(rs)


@preset_bp.route('/collectionDel', methods=['GET', 'POST'])
@api_log
def preset_collection_delete():
    """
    删除场景组合
    http://yapi.novatools.vip/project/613/interface/api/157228
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'presetCollectionId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, preset_collection_id = values

    rs = preset_svc.preset_collection_delete(device_id, preset_collection_id)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Preset_Group_Delete, params)
    return jsonify(rs)


@preset_bp.route('/collectionDelAll', methods=['GET', 'POST'])
@api_log
def preset_collection_del_all():
    """
    清空场景组合
    http://yapi.novatools.vip/project/613/interface/api/157216
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    rs = preset_svc.preset_collection_del_all(device_id,params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Preset_Collection_DelAll, params)
    return jsonify(rs)

@preset_bp.route('/collectionPlay', methods=['GET', 'POST'])
@api_log
def preset_collection_play():
    """
    播放场景组合
    http://yapi.novatools.vip/project/613/interface/api/157246
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    rs = preset_svc.preset_collection_play(device_id,params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Preset_Collection_Play, params)
    return jsonify(rs)


@preset_bp.route('/collectionRead', methods=['GET', 'POST'])
@api_log
def preset_collection_read():
    """
    读取场景组合详情
    http://yapi.novatools.vip/project/613/interface/api/157234
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    return jsonify(preset_svc.preset_collection_read(device_id, params))


@preset_bp.route('/collectionList', methods=['GET', 'POST'])
@api_log
def preset_collection_list():
    """
    读取场景组合列表
    http://yapi.novatools.vip/project/613/interface/api/157240
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values

    return jsonify(preset_svc.preset_collection_list(device_id))


@preset_bp.route('/readMultiDetail', methods=['GET', 'POST'])
@api_log
def read_multi_Detail():
    """
    批量查询多个场景详情
    http://yapi.novatools.vip/project/613/interface/api/157726
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'deviceId', 'presetList')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))

    device_id = params.get('deviceId')
    return jsonify(preset_svc.read_multi_detail(device_id, params))