"""
Created on 2022年3月18日
@author: 程声清
Role接口.
对应前端接口Role
返回值统一为 {data:JSON结构体, msg:"", status:0}
"""
from flask.globals import request
from flask.json import jsonify
from flask import Blueprint, g

from H_9U.api.websender import push_data_org
from H_9U.models.syncdataname import SyncDataName
from H_9U.models.result import get_result_model, ResInfo
from H_9U.models.sysconst import PermissionCode, DataType
from H_9U.service.role import rolesvc
from H_9U.service.user import usersvc
from H_9U.util.common import valid_json, valid_params
from H_9U.util.logwrapper import api_log
from H_9U.util import logutil
from H_9U.util.permission_valid import get_code_list, valid_permission_role_user

role_bp = Blueprint('role', __name__)


@role_bp.route('/add', methods=['GET', 'POST'])
@api_log
def add():
    """
    添加角色
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'rolename', 'parentId', 'funcId', 'inputId', 'outputId', 'screenId', 'mvrId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    role_name = params.get('rolename')
    funcs = []
    funcs.append(params.get('funcId'))
    funcs.append(params.get('inputId'))
    funcs.append(params.get('outputId'))
    funcs.append(params.get('screenId'))
    funcs.append(params.get('mvrId'))

    user = g.user
    role_id = user['roleId']
    # 增加接口校验
    role_rs = get_code_list(PermissionCode.Role_Operation, role_id)
    if role_rs['status'] != 0:
        return jsonify(role_rs)

    rs = rolesvc.add(params)
    if rs['status'] == 0:
        logutil.add_opt_log(logutil.LogModule.Role,
                            logutil.LogFunction.Role,
                            logutil.LogOperation.Add,
                            "add role: {0}, roleId:{1}, funcs:{2}".format(role_name, rs['data']['id'], funcs))
        role_id = rs['data']['id']
        rs = rolesvc.get_role_by_id(role_id)
        if rs['status'] == 0:
            push_data_org(SyncDataName.Role_Add, rs['data'])
    return jsonify(rs)


@role_bp.route('/delete', methods=['GET', 'POST'])
@api_log
def delete():
    """
    删除角色
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'id')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    role_id = values
    # 接口校验
    role_id_list = [role_id]
    role_rs = valid_permission_role_user(PermissionCode.Role_Operation, DataType.Role_Type, role_id_list)
    if role_rs['status'] != 0:
        return jsonify(role_rs)

    role_user_rs = rolesvc.get_user_list_by_role_id(role_id)
    user_list = []
    role_rs = rolesvc.get_role_by_id(role_id)
    rs = rolesvc.delete(role_id)
    if rs['status'] == 0:
        logutil.add_opt_log(logutil.LogModule.Role,
                            logutil.LogFunction.Role,
                            logutil.LogOperation.Delete,
                            "Delete roleId: {0}, role:{1}".format(role_id, role_rs['data']))
        push_data_org(SyncDataName.Role_Delete, {'id': role_id})
    if role_user_rs['status'] != 0 or not role_user_rs['data']:
        return jsonify(rs)
    for u_id in role_user_rs['data']:
        user_rs = usersvc.get(u_id)
        # 获取角色信息
        if user_rs['status'] == 0 and user_rs['data']:
            role_rs = rolesvc.get_role_by_user_id(u_id)
            if role_rs['status'] == 0 and role_rs['data']:
                role_id = role_rs['data']['roleId']
                role_message = rolesvc.get_role_by_id(role_id)
                user_rs['data']['role'] = role_message['data']
            user_list.append(user_rs['data'])
    push_data_org(SyncDataName.Role_Change, user_list)
    return jsonify(rs)


@role_bp.route('/update', methods=['GET', 'POST'])
@api_log
def update():
    """
    更新角色
    :return: 结果对象
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'id', 'rolename', 'parentId', 'funcId', 'inputId', 'outputId', 'screenId', 'mvrId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    role_id = params.get('id')
    role_name = params.get('rolename')
    permission_list = []
    func_ids = params.get('funcId')
    input_ids = params.get('inputId')
    output_ids = params.get('outputId')
    screen_ids = params.get('screenId')
    mvr_ids = params.get('mvrId')
    permission_list.append(func_ids)
    permission_list.append(input_ids)
    permission_list.append(output_ids)
    permission_list.append(screen_ids)
    permission_list.append(mvr_ids)
    # 接口校验
    role_id_list = [role_id]
    role_rs = valid_permission_role_user(PermissionCode.Role_Operation, DataType.Role_Type, role_id_list)
    if role_rs['status'] != 0:
        return jsonify(role_rs)
    role_rs = rolesvc.get_role_by_id(role_id)
    funcs_rs = rolesvc.get_menu_data_all(role_id)
    rs = rolesvc.update(params)
    if rs['status'] == 0:
        logutil.add_opt_log(logutil.LogModule.Role,
                            logutil.LogFunction.Role,
                            logutil.LogOperation.Edit,
                            "Update roleId: {0}, before:{1},funcs:{2}, after name:{3}, funcs:{4}".format(
                                role_id,
                                role_rs['data'],
                                funcs_rs['data'] if funcs_rs else '',
                                role_name,
                                permission_list
                            ))
        role_id = params.get('id')
        rs = rolesvc.get_role_by_id(role_id)
        if rs['status'] == 0:
            push_data_org(SyncDataName.Role_Update, rs['data'])
    return jsonify(rs)


@role_bp.route('/all', methods=['GET', 'POST'])
@api_log
def role_all():
    """
    获取全部角色
    :return:
    """
    user = g.user
    rs = rolesvc.all_role(user)
    return jsonify(rs)


@role_bp.route('/detail', methods=['GET', 'POST'])
@api_log
def detail():
    """
    获取角色详情
    :return:
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'roleId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    role_id = values
    rs = rolesvc.get_role_detail(role_id)
    return jsonify(rs)


@role_bp.route('/menu/all', methods=['GET', 'POST'])
@api_log
def menu_all():
    """
    获取角色菜单列表
    :return:
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'roleId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    role_id = values
    # 当编辑时候传来的 roleId = 管理员的parentId = 0,此时得知
    if role_id == 0:
        role_type = 1
        role_rs = rolesvc.get_role_id_by_role_type(role_type)
        if role_rs['status'] == 0 and role_rs['data']:
            role_id = role_rs['data']['id']
    rs = rolesvc.get_menu_all(role_id)
    return jsonify(rs)


@role_bp.route('/menu/data/all', methods=['GET', 'POST'])
@api_log
def menu_data_all():
    """
    获取菜单-数据列表
    :return:
    """
    params = valid_json(request.get_data().decode())
    if not params:
        return jsonify(get_result_model(ResInfo.Request_Json_Error))

    values = valid_params(params, 'roleId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    role_id = values
    # 1.点击编辑时,因为要回显数据,所以此时获取的菜单数据为当前角色的上级角色的数据,这样保证当前角色的编辑能处理的权限数据只能小于或等于其父角色的数据;
    #   所以传来的roleId = parentId的值;如果传递的是超管的 则将roleId = 1
    # 2.点击新增的时候, 获取菜单数据为 roleId = 选择角色的id的值
    if role_id == 0:
        role_type = 1
        role_rs = rolesvc.get_role_id_by_role_type(role_type)
        if role_rs['status'] == 0 and role_rs['data']:
            role_id = role_rs['data']['id']
    rs = rolesvc.get_menu_data_all(role_id)
    return jsonify(rs)
