"""
Created on 2019年6月28日
@author: 刘益宗
设备接口.
对应3.5代接口协议中的 Device
返回值统一为 {data:JSON结构体, msg:"", status:0}
"""

from H_9U.protocol.fncode import FnCode
from H_9U.util.log import logger
from H_9U.util.logwrapper import log_wrapper
from H_9U.models.result import transfer_midware_resp
from H_9U import controller as ctrl
from H_9U.models.result import MidwareErr, ResInfo, get_result_model
from H_9U.util.formatdata import format_client_data, format_server_data
from H_9U.models.sysconst import ModelId, FunctionType
from H_9U.models.sysconst import InterfaceType


@log_wrapper
def _device_detail_read(device_id):
    """
    读取设备详细信息
    :param device_id:设备id
    :return: 参考协议文档：0x00 - key: deviceDetail
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_DETAIL, device_id))


def device_detail_read(device_id):
    """
    读取设备详细信息
    :param device_id:设备id
    :return: 参考协议文档：0x00 - key: deviceDetail
    """
    rs = _device_detail_read(device_id)
    if rs['status'] == 0:
        # 4 dvi卡
        for slot in [x for x in rs['data']['slotList'] if x['modelId'] in ModelId.DL_Output_Card_List]:
            func_type_0 = slot['functions']['id0']['functionType']
            func_type_1 = slot['functions']['id1']['functionType']
            if func_type_0 == FunctionType.DL or func_type_1 == FunctionType.DL:
                for interface in slot['interfaces']:
                    func_type = func_type_0 if interface['interfaceId'] // 2 == 0 else func_type_1
                    if func_type == FunctionType.DL:
                        if interface['interfaceId'] % 2 == 0:
                            interface['functionType'] = FunctionType.Disable
                        else:
                            interface['functionType'] = FunctionType.DL
                            if slot['modelId'] in [ModelId.Output_4_DVI_Card, ModelId.Pd_Output_4_DVI_Card] :
                                interface['interfaceType'] = InterfaceType.Dual_Link_DVI
                            elif slot['modelId'] in [ModelId.Output_4_HDMI_Card, ModelId.Pd_Output_4_HDMI_Card]:
                                interface['interfaceType'] = InterfaceType.HDMI14
    return rs


@log_wrapper
def device_general_read(device_id):
    """
    读取设备基本信息
    :param device_id:设备Id
    :return: 参考协议文档：0x01 - key: deviceGeneral
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_GENERAL, device_id))


@log_wrapper
def device_general_write(device_id, data):
    """
    写入设备基本信息
    :param device_id:设备Id
    :param data: 参考协议文档：0x01 - key: deviceGeneral
    :return: 参考协议文档：0x01 - key: deviceGeneral
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_GENERAL, [data], device_id))


@log_wrapper
def _device_slot_read(device_id, slot_id):
    """
    读取设备槽信息
    :param device_id:设备Id
    :param slot_id: 卡槽Id
    :return: 参考协议文档：0x02 - key: deviceSlot
    """
    rs = transfer_midware_resp(ctrl.read(FnCode.DEVICE_SLOT, device_id, slot_id))
    format_server_data(rs['data'])
    return rs


def device_slot_read(device_id, slot_id):
    """
    读取设备槽信息
    :param device_id:设备Id
    :param slot_id: 卡槽Id
    :return: 参考协议文档：0x02 - key: deviceSlot
    """
    rs = _device_slot_read(device_id, slot_id)
    if rs['status'] == 0 and rs['data']:
        if rs['data']['modelId'] in ModelId.DL_Output_Card_List \
                and 'functions' in rs['data']:
            slot = rs['data']
            func_type_0 = slot['functions']['id0']['functionType']
            func_type_1 = slot['functions']['id1']['functionType']

            if func_type_0 == FunctionType.DL or func_type_1 == FunctionType.DL:
                for interface in slot['interfaces']:
                    func_type = func_type_0 if interface['interfaceId'] // 2 == 0 else func_type_1
                    if func_type == FunctionType.DL:
                        if interface['interfaceId'] % 2 == 0:
                            interface['functionType'] = FunctionType.Disable
                        else:
                            interface['functionType'] = FunctionType.DL
                            if slot['modelId'] in [ModelId.Output_4_DVI_Card, ModelId.Pd_Output_4_DVI_Card]:
                                interface['interfaceType'] = InterfaceType.Dual_Link_DVI
                            elif slot['modelId'] in [ModelId.Output_4_HDMI_Card, ModelId.Pd_Output_4_HDMI_Card]:
                                interface['interfaceType'] = InterfaceType.HDMI14
                            elif slot['modelId'] == ModelId.Output_4_HDBaseT_Card:
                                interface['interfaceType'] = InterfaceType.DB9_MALE
                            elif slot['modelId'] in [ModelId.Output_4_RJ45_Card, ModelId.Pd_Output_4_RJ45_Card]:
                                interface['interfaceType'] = InterfaceType.RJ45DL

            rs['data']['functions'] = [
                {
                    'id': 0,
                    'functionType': func_type_0
                },
                {
                    'id': 1,
                    'functionType': func_type_1
                }
            ]

            device_rs = _device_detail_read(device_id)
            if device_rs['status'] == 0:
                slot = [x for x in device_rs['data']['slotList'] if x['slotId'] == slot_id][0]
                for func in rs['data']['functions']:
                    func['isUsed'] = 1 \
                        if [x for x in slot['interfaces'] if x['isUsed'] == 1 and x['interfaceId'] // 2 == func['id']] \
                        else 0
            else:
                logger.info("read slot fill function is used err slotId:" + str(slot_id))
                for func in rs['data']['functions']:
                    func['isUsed'] = 1
    return rs


@log_wrapper
def device_slot_write(device_id, slot_id, data):
    """
    设置slot 参数
    :param device_id: 设备Id
    :param slot_id:  slotid
    :param data:  参数对象
    :return:  结果对象
    """
    format_client_data(data)
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_SLOT, [data], device_id, slot_id))


@log_wrapper
def device_interface_read(device_id, slot_id, interface_id):
    """
    读取设备的接口信息
    :param device_id: 设备Id
    :param slot_id: 卡槽Id
    :param interface_id: 接口Id
    :return: 参考协议文档：0x03 - key: deviceInterface
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_INTERFACE, device_id, slot_id, interface_id))


@log_wrapper
def device_power_read(device_id, power_id):
    """
    读取设备电源信息
    :param device_id:设备id
    :param power_id: 电源id 0：主电源，1：备份电源
    :return:参考协议文档：0x04 - key: devicePower
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_POWER, device_id, power_id))


@log_wrapper
def device_genlock_read(device_id, genlock_id):
    """
    读取设备Genlock
    :param device_id:设备Id
    :return: 参考协议文档：0x05 - key: deviceGenlock
    """
    rs = transfer_midware_resp(ctrl.read(FnCode.DEVICE_GENLOCK, device_id, genlock_id))
    format_server_data(rs['data'])
    return rs


@log_wrapper
def device_genlock_write(device_id, genlock_id, data):
    """
    读取设备Genlock数据
    :param device_id: 设备Id
    :param genlock_id: GenlockId
    :param data: {参考协议文档：0x05 - key: deviceGenlock
    :return: 参考协议文档：0x05 - key: deviceGenlock
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_GENLOCK, [data], device_id, genlock_id))


@log_wrapper
def device_slot_function_write(device_id, slot_id, data):
    """
    写入子卡功能类型
    :param device_id: 设备Id
    :param slot_id: 槽Id
    :param data: jsondata
    :param 参考协议文档：0x06  deviceSlotFunction
    :return: 参考协议文档：0x06: deviceSlotFunction
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_SLOTFUNCTION, [data], device_id, slot_id))


@log_wrapper
def device_slot_function_read(device_id, slot_id):
    """
    读取子卡功能类型
    :param device_id: 设备id
    :param slot_id: 槽Id
    :param interface_id: 接口Id
    :return: 参考协议文档：0x06 - key: deviceSlotFunction
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_SLOTFUNCTION, device_id, slot_id))


@log_wrapper
def device_ip_read(device_id):
    """
    设备设备IP地址和DNS
    :param device_id: 设备Id
    :return:参考协议文档：0x07 - key: deviceIp
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_IP, device_id))


@log_wrapper
def device_ip_write(device_id, data):
    """
    设置设备IP和DNS
    :param device_id: 设备号
    :param data: 参考协议文档：0x07 - key: deviceIp
    :return: 参考协议文档：0x07 - key: deviceIp
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_IP, [data], device_id))


@log_wrapper
def device_com_read(device_id):
    """
    设置设备COM信息
    :param device_id: 设备Id
    :return: 参考协议文档：0x08 - key: deviceCom
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_COM, device_id))


@log_wrapper
def device_com_write(device_id, data):
    """
    设置设备COM信息
    :param device_id: 设备号
    :param data:  参考协议文档：0x08 - key: deviceCom
    :return: 参考协议文档：0x08 - key: deviceCom
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_COM, [data], device_id))


@log_wrapper
def device_save_write(device_id, data):
    """
    保存当前设备信息
    :param device_id: 设备号
    :return: 参考协议文档：0x09 - key: deviceSave
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_SAVE, [data], device_id))


@log_wrapper
def device_shutdown_write(device_id, data):
    """
    设备关机
    :param device_id:设备号
    :param data: 0：设备重启；1：设备关机；2：开机
    :return: 参考协议文档：0x0A - key: deviceShutDown
    """
    logger.debug("设关机/重启, data:%s" % data)

    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_SHUTDOWN, [data], device_id))


@log_wrapper
def device_restore_factory_write(device_id, data):
    """
    设备恢复出厂设置。
    :param device_id: 设备id
    :param data: 设备操作
    :return: 参考协议文档：0x0B - key: deviceRestoreFactory
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_RESTOREFACTORY, [data], device_id))


@log_wrapper
def device_firware_version_read(device_id):
    """
    获取程序版本号
    :param device_id: 设备号
    :return: 参考协议文档：0x0C - key: deviceFirwareVersion
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_FIRWAREVERSION, device_id))


@log_wrapper
def device_upgrade_write(device_id, data):
    """
    固件升级(一)
    :param device_id: 设备号
    :param data:JSON对象
    :return:  参考协议文档：0x0D - key: deviceUpgrade
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_UPGRADE, [data], device_id))


@log_wrapper
def device_upgrade_sts_read(device_id):
    """
    固件升级(二)读取固件升级状态
    :param device_id: 设备号
    :return:  参考协议文档：0x0D - key: deviceUpgrade
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_UPGRADESTS, device_id))


@log_wrapper
def device_self_check(device_id, data):
    """
    设备自检
    :param device_id: 设备Id
    :param data: 自检
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_SELF_CHECK, [data], device_id))


@log_wrapper
def device_self_check_status(device_id):
    """
    设备自检
    :param device_id: 设备Id
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_SELF_CHECK_STS, device_id))


@log_wrapper
def device_guide_data_write(device_id, data):
    """
    设置设备配置信息
    :param device_id: 设备id
    :param data: 文件路径
    :return: 结果对象
    """
    res = ctrl.write(FnCode.DEVICE_GUIDE_DATA, [data], device_id)
    status = res['status']
    if status == 0:
        if res['data']:
            res['data'] = res['data'][0]

    elif status == MidwareErr.VERIFYANSERROR:
        res = get_result_model(ResInfo.Device_Setting_File_Err)
    elif status in MidwareErr.Midware_Server_Status_Map:
        res = get_result_model(MidwareErr.Midware_Server_Status_Map[status])
        data = res.get('data')
        res['data'] = data
    else:
        msg = str(res)
        res = get_result_model(ResInfo.Unknow_Error)
        res['data'] = msg
    return res


@log_wrapper
def device_nomark_mode_read(device_id):
    """
    读取中性设置标志
    :param device_id: 设备Id
    :return: 中性标志
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_NOMARK_MODE, device_id))


@log_wrapper
def device_nomark_mode_write(device_id, data):
    """
    设置中性模式
    :param device_id: 设备Id
    :param data: 中性模式，参考接口文档
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_NOMARK_MODE, [data], device_id))

@log_wrapper
def device_effective_time_write(device_id, data):
    """
    设备授权时间
    :param device_id: 设备Id
    :param data: 授权信息
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_EFFECTIVE_TIME, [data], device_id))

@log_wrapper
def device_local_time_write(device_id, data):
    """
    设备时间
    :param device_id: 设备Id
    :param data: 时间
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_LOCAL_TIME, [data], device_id))


@log_wrapper
def device_effective_time_read(device_id):
    """
    设备授权时间
    :param device_id: 设备Id
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_EFFECTIVE_TIME, device_id))


@log_wrapper
def device_local_time_read(device_id):
    """
    设备授权时间
    :param device_id: 设备Id
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_LOCAL_TIME, device_id))


@log_wrapper
def device_hdr_support_read(device_id):
    """
    设备授权时间
    :param device_id: 设备Id
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_HDR_SUPPORT, device_id))


@log_wrapper
def device_language_read(device_id):
    """
    读取设备语言
    :param device_id: 设备Id
    :return: 语言,0:英文；1：中文
    """
    rs = transfer_midware_resp(ctrl.read(FnCode.DEVICE_LANGUAGE, device_id))
    if rs['status'] == 0:
        rs['data'] = {
            'deviceId': device_id,
            'languageMode': 1 if rs['data']['lang'] == 'zh' else 0 if rs['data']['lang'] == 'en' else 2
        }
    else:
        rs = get_result_model()
        rs['data'] = {
            'deviceId': device_id,
            'languageMode': 1
        }
    return rs


@log_wrapper
def device_language_write(device_id, data):
    """
    读取设备语言
    :param device_id: 设备Id
    :param data: 语言数据
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_LANGUAGE, [data], device_id))


@log_wrapper
def device_ipc_mode_read(device_id, slot_id):
    """
    读取ipc mode
    :param device_id:  设备Id
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_IPC_OUT_MODE, device_id, slot_id))


@log_wrapper
def device_ipc_mode_write(device_id, slot_id, data):
    """
    设置ipcmode
    :param device_id: 设备Id
    :param data: mode数据
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_IPC_OUT_MODE, [data], device_id, slot_id))


@log_wrapper
def device_resource_mode_read(device_id):
    """
    读取动态资源类型
    :param device_id: 设备Id
    :param data: mode数据
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_DYNAMIC_RESOURCES, device_id))


@log_wrapper
def device_resource_mode_write(device_id, data):
    """
    设置动态资源类型
    :param device_id: 设备Id
    :param data: mode数据
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_DYNAMIC_RESOURCES, [data], device_id))


def device_backup_mode_read(device_id):
    """
    读取备份类型
    :param device_id: 设备Id
    :param data: mode数据
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_WORK_MODE, device_id))


def device_backup_mode_write(device_id, data):
    """
    设置备份类型
    :param device_id: 设备Id
    :param data: mode数据
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_WORK_MODE, [data], device_id))


@log_wrapper
def device_init_status_read(device_id):
    """读取设备初始化状态"""
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_INIT_STATUS, device_id))


@log_wrapper
def device_type_read(device_id):
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_TYPE, device_id))

@log_wrapper
def device_background_color_read(device_id):
    """
    读取设备背景颜色
    :param device_id: 设备Id
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_BG_COLOR, device_id))


@log_wrapper
def device_background_color_write(device_id, data):
    """
    设置设备背景颜色
    :param device_id: 设备Id
    :param data: background_color数据
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_BG_COLOR, [data], device_id))


###### osd接口 #####
@log_wrapper
def read_device_font_list(device_id):
    """
    读取设备字体列表
    :param device_id: 设备id
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_FONT_LIST, device_id))


@log_wrapper
def write_device_font_delete(device_id, data):
    """
    删除设备字体
    :param device_id: 设备id
    :param data: 设备删除参数
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_FONT_DELETE, [data], device_id))


@log_wrapper
def write_device_font_create(device_id, data):
    """
    读取设备字体列表
    :param device_id: 设备id
    :param data: 设备删除参数
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_FONT_CREATE, [data], device_id))

@log_wrapper
def write_device_fiber_mode(device_id, data):
    """
    设置4光纤卡的模式
    :param device_id: 设备号
    :param params: params
    :return:
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_SLOT, [data], device_id))

###### osd接口 end #####



@log_wrapper
def input_backup_write(device_id, enable, type, data):
    """
    输入源开启/关闭备份
    :param device_id:  设备Id
    :param enable:  开启/关闭
    :param data:  数据
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_INPUT_BACKUP_SWITCH, [data], device_id, enable, type))


@log_wrapper
def input_backup_read(device_id):
    """
    输入源开启/关闭备份状态回读
    :param device_id:  设备Id
    :param input_backup_enable:  开启/关闭
    :param data:  数据
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_INPUT_BACKUP_SWITCH, device_id))


@log_wrapper
def input_backup_update(device_id, data):
    """
    输入源备份添加/更新
    :param device_id:  设备Id
    :param data:  数据
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_INPUT_BACKUP_UPDATE, [data], device_id))

@log_wrapper
def input_backup_list_read(device_id):
    """
    输入源备份添加/更新
    :param device_id:  设备Id
    :param data:  数据
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_INPUT_BACKUP_UPDATE, device_id))


@log_wrapper
def input_backup_delete(device_id, data):
    """
    输入源备份删除
    :param device_id:  设备Id
    :param data:  数据
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_INPUT_BACKUP_DELETE, [data], device_id))


@log_wrapper
def write_device_central_cmd_create(device_id, data):
    """
    创建设备中控命令
    :param device_id: 设备id
    :param data: 命令对象
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_CENTRAL_CMD_CREATE, [data], device_id))


@log_wrapper
def write_device_central_cmd_modify(device_id, data):
    """
    修改中控命令
    :param device_id: 设备id
    :param data: 命令对象
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_CENTRAL_CMD_MODIFY, [data], device_id))


@log_wrapper
def write_device_central_cmd_delete(device_id, data):
    """
    删除中控命令
    :param device_id: 设备id
    :param data: 中控命令列表
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_CENTRAL_CMD_DELETE, [data], device_id))


@log_wrapper
def write_device_central_cmd_apply(device_id, data):
    """
    应用中控命令
    :param device_id: 设备id
    :param data: 命令id数据
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_CENTRAL_CMD_APPLY, [data], device_id))


@log_wrapper
def read_device_central_cmd_list(device_id):
    """
    读取设备中控命令列表
    :param device_id: 设备id
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_CENTRAL_CMD_LIST, device_id))


@log_wrapper
def read_device_central_cmd_detail(device_id, cmd_id):
    """
    读取设备中控命令详情
    :param device_id: 设备id
    :param cmd_id: 设备命令id
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_CENTRAL_CMD_FULL_DETAIL, device_id, cmd_id))


@log_wrapper
def write_device_output_interface_mode(device_id, data):
    """
    设置输出接口的模式切换
    :param device_id: 设备id
    :param data: 设备参数
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_INTERFACE, [data], device_id))

@log_wrapper
def write_device_interface_capacity(device_id, slot_id, interface_id, data):
    """
    设置输出接口的模式切换
    :param device_id: 设备id
    :param data: 设备参数
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_INTERFACE, [data], device_id, slot_id, interface_id))


@log_wrapper
def read_audio_card_detail(device_id, slot_id):
    """
     读取音频卡详情
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_AUDIO_CARD_DETAIL, device_id, slot_id))


@log_wrapper
def write_audio_card_attribute(device_id,slot_id, data):
    """
     读取音频卡详情
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_AUDIO_CARD_MODE, [data], device_id, slot_id))


@log_wrapper
def write_audio_interface_delay_time(device_id, slot_id, data):
    """
     读取音频卡详情
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_AUDIO_INTERFACE_DELAY_TIME, [data], device_id, slot_id))


@log_wrapper
def read_audio_card_list(device_id):
    """
     读取音频卡列表
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_AUDIO_CARD_LIST,  device_id))


@log_wrapper
def write_port_mode(device_id, data):
    """
    读取光口卡模式
    :param device_id: 设备id
    :param slotId: slotId
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_SLOT, [data], device_id))


@log_wrapper
def read_fiber_detail(device_id, slot_id):
    """
    读取光口卡详情
    :param device_id: 设备id
    :param slotId: slotId
    :return: 结果对象
    """
    rs = transfer_midware_resp(ctrl.read(FnCode.DEVICE_FIBER_CARD_DETAIL, device_id, slot_id))
    format_server_data(rs['data'])
    return rs

@log_wrapper
def sdp_create(device_id, data):
    """
    读取光口卡详情
    :param device_id: 设备id
    :param slotId: slotId
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_GUIDE_DATA, [data], device_id))


@log_wrapper
def write_lcd_config(device_id, data):
    """
    读取光口卡模式
    :param device_id: 设备id
    :param slotId: slotId
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_LCD_ENABLE, [data], device_id))


@log_wrapper
def read_lcd_config(device_id):
    """
    读取光口卡模式
    :param device_id: 设备id
    :param slotId: slotId
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_LCD_ENABLE, device_id))




@log_wrapper
def bright_mode_write(device_id, data):
    """
    设置亮度调节模式
    :param device_id:  设备Id
    :param mode:  模式
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.DEVICE_BRIGHT_MODE_SWITCH, [data], device_id))


@log_wrapper
def bright_mode_read(device_id):
    """
    设置亮度调节模式
    :param device_id:  设备Id
    :param mode:  模式
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.read(FnCode.DEVICE_BRIGHT_MODE_SWITCH, device_id))



def test():
    return transfer_midware_resp(ctrl.read(0x01F0, 0))


def test_write(count):
    data = {'list':[]}
    for x in range(count):
        data['list'].append({
            "range": x
        })
    print('#'*100)
    print(len(str(data)))
    rs = transfer_midware_resp(ctrl.write(0x01F0,[data], 0))
    print(rs)
    return rs
