"""
Created on 2019年11月12日
@author: 刘益宗
输入接口.
对应3.5代接口协议中的 ipc
返回值统一为 {data:JSON结构体, msg:"", status:0}
"""
from H_9U.protocol.fncode import FnCode
from H_9U.models.result import transfer_midware_resp
from H_9U import controller as ctrl
from H_9U.util.logwrapper import log_wrapper
from H_9U.models.sysconst import ConnectCapacity, InterfaceType, SourceType, IpcCropType


@log_wrapper
def ipc_source_list(device_id, data):
    """
    获取ipc源列表
    :param device_id: 设备id
    :param data: data
    :return: ipc源列表
    """
    # data['online'] = 1
    rs = transfer_midware_resp(ctrl.write(FnCode.IPC_SOURCE_LIST, [data], device_id))
    return rs


@log_wrapper
def ipc_channel_list(device_id, source_id):
    """
    读取信号源通道列表
    :param device_id: 设备Id
    :param source_id: 源Id
    :return 结果数据
    """
    source_id_low = source_id % 256
    source_id_high = source_id // 256
    return transfer_midware_resp(ctrl.read(FnCode.IPC_CHANNEL_LIST, device_id,  source_id_low, source_id_high))


@log_wrapper
def ipc_channel_detail(device_id, source_id, channel_id):
    """
     读取信号源通道详情
     :param device_id: 设备Id
     :param source_id: 源Id
     :param channel_id: 源Id
     :return 结果数据
     """
    data = {"deviceId": device_id, "sourceId": source_id, "channelId": channel_id}
    return transfer_midware_resp(ctrl.write(FnCode.IPC_SOURCE_CHANNEL_DETAIL, [data], device_id))


@log_wrapper
def ipc_camera_list(device_id, decode_capacity):
    """
    获取ipc摄像头列表
    :param device_id: 设备id
    :param decode_capacity: 规格
    :return: ipc摄像头列表
    """
    return transfer_midware_resp(ctrl.read(FnCode.IPC_CAMERA_LIST, device_id, decode_capacity))


@log_wrapper
def ipc_channel_edit(device_id, data):
    """
    修改通道数据
    :param device_id: 设备Id
    :param data: data
    :return 结果数据
    """
    return transfer_midware_resp(ctrl.write(FnCode.IPC_CHANNEL_GENERAL, [data], device_id))


@log_wrapper
def ipc_source_create(device_id, slot_id, data):
    """
    创建ipc源
    :param device_id: 设备id
    :param slot_id: slot id
    :param data: 源信息
    :return: 源id
    """
    if not isinstance(data, list):
        data = [data]
    return transfer_midware_resp(ctrl.write(FnCode.IPC_SOURCE_CREATE, data, device_id, slot_id))


@log_wrapper
def ipc_channel_delete(device_id, data):
    """
    删除通道
    :param device_id: 设备id
    :param data: 源信息
    :return: 源id
    """
    return transfer_midware_resp(ctrl.write(FnCode.IPC_CHANNEL_DELETE, [data], device_id))


@log_wrapper
def ipc_source_append(device_id, data):
    """
    追加ipc源的通道数据
    :param device_id: 设备id
    :param data: 源信息
    :return: 源id
    """
    return transfer_midware_resp(ctrl.write(FnCode.IPC_CHANNEL_APPEND, [data], device_id))


@log_wrapper
def ipc_source_update_Info(device_id, data):
    """
    追加ipc源的通道数据
    :param device_id: 设备id
    :param data: 源信息
    :return: 源id
    """
    return transfer_midware_resp(ctrl.write(FnCode.IPC_SOURCE_GENERAL,[data], device_id))


@log_wrapper
def ipc_source_delete(device_id, data):
    """
    删除信号源
    :param device_id: 设备id
    :param data:
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.IPC_SOURCE_DELETE, [data], device_id))


@log_wrapper
def ipc_source_general_read(device_id, source_id):
    """
    查询ipc源详情
    :param device_id: 设备id
    :param source_id: source id
    :return: 结果对象
    """
    source_id_low = source_id % 256
    source_id_high = source_id // 256
    return transfer_midware_resp(ctrl.read(FnCode.IPC_SOURCE_GENERAL, device_id, source_id_low, source_id_high))


@log_wrapper
def ipc_source_general(device_id, source_id, data):
    """
    设置ipc拼接模式
    :param device_id: 设备id
    :param source_id: source_id
    :param data: 拼接模式
    :return: 结果对象
    """
    source_id_low = source_id % 256
    source_id_high = source_id // 256
    return transfer_midware_resp(ctrl.write(FnCode.IPC_SOURCE_GENERAL, [data], device_id, source_id_low, source_id_high))


@log_wrapper
def ipc_crop_list(device_id, source_id, source_type):
    """
    ipc截取列表
    :param device_id: 设备id
    :param source_id: 源 id
    :return: 结果对象
    """
    source_id_low = source_id % 256
    source_id_high = source_id // 256
    return transfer_midware_resp(ctrl.read(FnCode.IPC_CROP_LIST, device_id, source_id_low, source_id_high, source_type))


@log_wrapper
def ipc_crop_create(device_id, source_id, data):
    """
    ipc截取创建
    :param device_id: 设备id
    :param source_id: 源 id
    :param data:
    :return: 结果对象
    """
    source_id_low = source_id % 256
    source_id_high = source_id // 256
    return transfer_midware_resp(ctrl.write(FnCode.IPC_CROP_CREATE, [data], device_id, source_id_low, source_id_high))


@log_wrapper
def ipc_crop_delete(device_id, source_id, data):
    """
    ipc截取删除
    :param device_id: 设备id
    :param source_id: 源 id
    :param data:
    :return: 结果对象
    """
    source_id_low = source_id % 256
    source_id_high = source_id // 256
    return transfer_midware_resp(ctrl.write(FnCode.IPC_CROP_DELETE, [data], device_id, source_id_low, source_id_high))


@log_wrapper
def ipc_crop_general_read(device_id, source_id, crop_id):
    """
    读取ipc截取信息
    :param device_id: 设备id
    :param source_id: 源 id
    :param crop_id: 截取 id
    :return: 结果对象
    """
    source_id_low = source_id % 256
    source_id_high = source_id // 256
    data = {
        "deviceId":device_id,
        "sourceId": source_id,
        "sourceType": SourceType.IPCSource,
        "cropId": crop_id,
        "codeType": IpcCropType.ReadCropType
    }
    return transfer_midware_resp(ctrl.write(FnCode.IPC_CROP_GENERAL, [data], device_id, source_id_low, source_id_high, crop_id))


@log_wrapper
def ipc_crop_general_write(device_id,  data):
    """
    修改ipc截取信息
    :param device_id: 设备id
    :param source_id: 源 id
    :param crop_id: 截取 id
    :param data:
    :return: 结果对象
    """
    data['codeType'] = IpcCropType.WriteCropType
    return transfer_midware_resp(ctrl.write(FnCode.IPC_CROP_GENERAL, [data], device_id))


@log_wrapper
def ipc_group_list(device_id):
    """
    ipc分组列表
    :param device_id: 设备id
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.read(FnCode.IPC_GROUP_LIST, device_id))


@log_wrapper
def ipc_group_create(device_id, data):
    """
    创建ipc分组
    :param device_id: 设备id
    :param data: 分组数据
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.IPC_GROUP_CREATE, [data], device_id))


@log_wrapper
def ipc_group_delete(device_id, data):
    """
    删除ipc分组
    :param device_id: 设备id
    :param data: 分组数据
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.IPC_GROUP_DELETE, [data], device_id))


@log_wrapper
def ipc_group_node_list(device_id, data):
    """
    ipc分组列表
    :param device_id: 设备id
    :param data: data
    :return: 结果对象
    """
    data['online'] = 1
    data['isMontage'] = 0
    return transfer_midware_resp(ctrl.write(FnCode.IPC_GROUP_NODE_LIST, [data], device_id))


@log_wrapper
def ipc_group_node_add(device_id, data):
    """
     ipc分组添加节点
    :param device_id: 设备id
    :param data: 分组
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.IPC_GROUP_NODE_ADD, [data], device_id))


@log_wrapper
def ipc_group_node_delete(device_id, data):
    """
     ipc分组删除节点
    :param device_id: 设备id
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.IPC_GROUP_NODE_DELETE, [data], device_id))


@log_wrapper
def ipc_group_node_clear(device_id, group_id, data):
    """
     ipc分组删除节点
    :param device_id: 设备id
    :param group_id: 分组id
    :param data:
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.IPC_GROUP_NODE_CLEAR, [data], device_id, group_id))


@log_wrapper
def ipc_slot_source_count_read(device_id, slot_id):
    """
    读取系统初始化状态
    :param device_id 设备号
    :param slot_id 槽位好
    :return: ipc源数量
    """
    return transfer_midware_resp(ctrl.read(FnCode.IPC_SLOT_SOURCE_COUNT, device_id, slot_id))


@log_wrapper
def read_ipc_stream_detail(device_id, data):
    """
    读取ipc码流详情，以写的方式读取
    :param device_id: 设备Id
    :param data: 码流Id列表
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.IPC_STREAM_DETAIL, [data], device_id))


# ipc 拼接 开始
@log_wrapper
def ipc_montage_list(device_id, slot_id):
    """
    ipc 拼接列表
    :param device_id: 设备Id
    :param slot_id: slot id
    :return: slot下拼接列表
    """
    rs = transfer_midware_resp(ctrl.read(FnCode.IPC_MONTAGE_LIST, device_id, slot_id))
    if rs['status'] == 0 and rs['data']:
        for x in rs['data']['montageList']:
            x['connectionCapacity'] = ConnectCapacity.DL
            x['interfaceType'] = InterfaceType.RJ45
    return rs


@log_wrapper
def ipc_montage_group_list(device_id, slot_id):
    """
    ipc 拼接列表
    :param device_id: 设备Id
    :param slot_id: slot id
    :return: slot下拼接列表
    """
    rs = transfer_midware_resp(ctrl.read(FnCode.IPC_MONTAGE_GROUP_LIST, device_id, slot_id))
    return rs


@log_wrapper
def ipc_montage_create(device_id, data):
    """
    ipc 拼接列表
    :param device_id: 设备Id
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.IPC_MONTAGE_CREATE, [data], device_id))


@log_wrapper
def ipc_montage_template_detail(device_id, template_id):
    """
    读取ipc拼接模板详情
    :param device_id: 设备Id
    :param slot_id: slot_id
    :param template_id: template_id
    :param montage_id: montage_id
    :return: 结果对象
    """
    template_id_low = template_id % 256
    template_id_high = template_id // 256
    return transfer_midware_resp(ctrl.read(FnCode.IPC_MONTAGE_TEMPLATE_DETAIL, device_id, template_id_low, template_id_high))


@log_wrapper
def ipc_montage_delete(device_id,  data):
    """
    ipc 拼接列表
    :param device_id: 设备Id
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.IPC_MONTAGE_DELETE, [data],  device_id))


@log_wrapper
def ipc_montage_apply(device_id, data):
    """
    ipc 拼接列表
    :param device_id: 设备Id
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.IPC_MONTAGE_APPLY, [data], device_id))


@log_wrapper
def ipc_montage_rename(device_id, data):
    """
    ipc 拼接列表
    :param device_id: 设备Id
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.IPC_MONTAGE_GENERAL, [data], device_id))

# ipc 拼接 结束
@log_wrapper
def read_ipc_stream_list(device_id):
    """
    读取ipc码流详情列表
    :param device_id: 设备Id
    :return: 详细信息
    """
    return transfer_midware_resp(ctrl.read(FnCode.IPC_STREAM_LIST, device_id))


@log_wrapper
def write_ipc_bg_color(device_id, data):
    """
    设置IPC无源显示画面
    :param device_id: 设备id
    :param data: 设备参数
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.IPC_BG_COLOR, [data], device_id))


@log_wrapper
def read_ipc_bg_color(device_id):
    """
    查询IPC无源显示画面
    :param device_id: 设备id
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.read(FnCode.IPC_BG_COLOR, device_id))


@log_wrapper
def write_ipc_stream_rule(device_id, data):
    """
    设置IPC码流规则
    :param device_id: 设备id
    :param data: 码流规则参数
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.IPC_STREAM_SELECT_RULE, [data], device_id))


@log_wrapper
def read_ipc_stream_rule(device_id):
    """
    查询IPC码流规则
    :param device_id: 设备id
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.read(FnCode.IPC_STREAM_SELECT_RULE, device_id))

@log_wrapper
def ipc_source_status_list(device_id, group_id, data):
    """
	ipc组内源信号的处理
	:param device_id: 设备Id
	:param group_id: group id
	:return: 结果对象
	"""
    return transfer_midware_resp(ctrl.write(FnCode.IPC_SOURCE_STATUS_LIST, [data], device_id, group_id))


@log_wrapper
def ipc_montage_template_zorder_update(device_id, data):
    """
    ipc 拼接列表
    :param device_id: 设备Id
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.IPC_MONTAGE_ZORDER_UPDATE, [data], device_id))


@log_wrapper
def ipc_montage_group_update(device_id, data):
    """
    ipc 拼接列表
    :param device_id: 设备Id
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.IPC_MONTAGE_GROUP_UPDATE, [data], device_id))


@log_wrapper
def ipc_montage_poll_status_write(device_id, data):
    """
    ipc 拼接列表
    :param device_id: 设备Id
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.IPC_MONTAGE_POLL_STATUS, [data], device_id))
