"""
Created on 2019年6月28日
@author: 刘益宗
屏幕接口.
对应协议中的<Screen>.
"""
from H_9U.models.result import transfer_midware_resp
from H_9U.models.sysconst import ReadSign
from H_9U.protocol.fncode import FnCode
from H_9U import controller as ctrl
from H_9U.util.cache import cacher
from H_9U.util.logwrapper import log_wrapper
from H_9U.util.formatdata import format_client_data, format_server_data


@log_wrapper
def screen_list_read(device_id):
    """
    读取屏幕列表
    :param device_id: 设备Id
    :return: 参考接口screenList
    """
    rs = transfer_midware_resp(ctrl.read(FnCode.SCREEN_LIST, device_id, ReadSign.Is_Not_ALL))
    if rs['status'] == 0 and rs['data']:
        if rs['data']['screens']:
            rs['data']['screens'] = sorted(rs['data']['screens'], key=lambda x: float(x['createTime']))
    return rs


@log_wrapper
def screen_full_detail_read(device_id, screen_id):
    """
    读取屏幕详情
    :param device_id: 设备Id
    :param screen_id:  屏幕Id
    :return: 参考接口screenFullDetail
    """
    rs = transfer_midware_resp(ctrl.read(FnCode.SCREEN_FULLDETAIL, device_id, screen_id))
    if rs['status'] == 0:
        format_server_data(rs['data'])
    return rs


@log_wrapper
def screen_create_write(device_id, data):
    """
    创建屏幕
    :param device_id: 设备Id
    :param data:  json
    :return:参考接口screenCreate
    """
    format_client_data(data)
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_CREATE, [data], device_id))


@log_wrapper
def screen_delete_write(device_id, screen_id, data):
    """
    删除屏幕
    :param device_id: 设备Id
    :param screen_id:  屏幕Id
    :param data:  json数据
    :return:参考接口screenDelete
    """
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_DELETE, [data], device_id, screen_id))


@log_wrapper
def screen_general_write(device_id, screen_id, data):
    """
    修改screen常规信息
    :param device_id: 设备号
    :param screen_id: 屏幕号
    :param data: json数据
    :return:参考接口screenGeneral
    """
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_GENERAL, [data], device_id, screen_id))


@log_wrapper
def screen_general_read(device_id, screen_id):
    """
    读取screen常规信息
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :return:参考接口screenGeneral
    """
    return transfer_midware_resp(ctrl.read(FnCode.SCREEN_GENERAL, device_id, screen_id))


@log_wrapper
def screen_output_mode_write(device_id, screen_id, data):
    """
    设置指定screen的输出.
    :param device_id: 设备号
    :param screen_id: 屏幕号
    :param data: json数据，参考<screenOutputMode>
    :return:接口对象
    """
    format_client_data(data)
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_OUTMODE, [data], device_id, screen_id))


@log_wrapper
def screen_output_mode_read(device_id, screen_id):
    """
    读取指定screen的输出排列方式
    :param device_id: 设备号
    :param screen_id: 屏幕号
    :return:参考接口screenOutputMode
    """
    return transfer_midware_resp(ctrl.read(FnCode.SCREEN_OUTMODE, device_id, screen_id))


@log_wrapper
def screen_move_interface_write(device_id, screen_id, interface_id, data):
    """
    移动屏幕
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :param interface_id: 接口Id
    :param data: json
    :return:参考接口screenMoveInterface
    """
    return transfer_midware_resp(
        ctrl.write(FnCode.SCREEN_MOVEINTERFACE, [data], device_id, screen_id, interface_id))


@log_wrapper
def screen_add_output_write(device_id, screen_id, interface_id, data):
    """
    将指定的output添加到screen
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :param interface_id: 接口Id
    :param data: json
    :return:参考接口screenAddOuput
    """
    rs = transfer_midware_resp(
        ctrl.write(FnCode.SCREEN_ADDOUTPUT, [data], device_id, screen_id, interface_id))
    if rs['status'] == 0:
        cacher.clear()
    return rs


@log_wrapper
def screen_remove_output_write(device_id, screen_id, interface_id, data):
    """
    将指定的output从screen中移除（移除并非删除）
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :param interface_id: 接口Id
    :param output_id: 输出Id
    :return:参考接口screenRemoveOuput
    """

    return transfer_midware_resp(
        ctrl.write(FnCode.SCREEN_REMOVEOUTPUT, [data], device_id, screen_id, interface_id))


@log_wrapper
def screen_ftb_write(device_id, screen_id, data):
    """
    屏幕的FTB操作，黑屏或者亮屏
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :param data: ftb数据
    :return:参考接口screenFtb
    """
    format_client_data(data)
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_FTB, data if isinstance(data, list) else [data], device_id, screen_id))


@log_wrapper
def screen_freeze_write(device_id, screen_id, data):
    """
    屏幕冻结
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :param data: 是否启用冻结[0：解冻；1：冻结]
    :return:参考接口
    """
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_FREEZE, data if isinstance(data, list) else [data], device_id, screen_id))


@log_wrapper
def screen_bkg_read(device_id, screen_id):
    """
    读取屏幕bkg
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :return:参考接口screenFreeze
    """
    return transfer_midware_resp(ctrl.read(FnCode.SCREEN_BKG, device_id, screen_id))


@log_wrapper
def screen_bkg_write(device_id, screen_id, data):
    """
    屏幕添加BKG，接口screenBKG
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :param data: bkg 数据
    :return: 返回bkgId
    """
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_BKG, [data], device_id, screen_id))


@log_wrapper
def screen_osd_write(device_id, screen_id, data):
    """
    设置屏幕OSD
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :param data: json
    :return: 参考接口screenOSD
    """
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_OSD, [data], device_id, screen_id))


@log_wrapper
def screen_gamma_write(device_id, screen_id, data):
    """
    设置屏幕gamma
    :param device_id: 设备Id
    :param screen_id:  屏幕Id
    :param data:  gamma参数
    :return:  结果对象
    """
    format_client_data(data)
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_GAMMA, [data], device_id, screen_id))


@log_wrapper
def screen_image_quality_write(device_id, screen_id, data):
    """
    设置屏幕画质
    :param device_id: 设备Id
    :param screen_id:  屏幕Id
    :param data:  画质参数
    :return:  结果对象
    """
    format_client_data(data)
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_IMAGE_QUALITY, [data], device_id, screen_id))


@log_wrapper
def screen_test_pattern_write(device_id, screen_id, data):
    """
    设置屏幕测试画面
    :param device_id: 设备Id
    :param screen_id:  屏幕Id
    :param data:  测试画面参数
    :return:  结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_TEST_PATTERN, [data], device_id, screen_id))


@log_wrapper
def screen_brightness_read(device_id, screen_id):
    """
    读取屏幕亮度
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :return: 屏幕亮度
    """
    return transfer_midware_resp(ctrl.read(FnCode.SCREEN_BRIGHTNESS, device_id, screen_id))


@log_wrapper
def screen_brightness_write(device_id, screen_id, data):
    """
    设置屏幕亮度
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :return: 屏幕亮度
    """
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_BRIGHTNESS, [data], device_id, screen_id))


@log_wrapper
def multi_write_brightness(device_id, data):
    """
    设置屏幕亮度
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :return: 屏幕亮度
    """
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_MULTI_BRIGHTNESS,  data if isinstance(data, list) else [data], device_id))


@log_wrapper
def screen_brightness_save(device_id, screen_id, data):
    """
    设置屏幕亮度
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :return: 屏幕亮度
    """
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_BRIGHTNESS_SAVE,  data if isinstance(data, list) else [data], device_id, screen_id))

@log_wrapper
def screen_hdr_read(device_id, screen_id):
    """
    读取屏幕亮度
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :return: 屏幕HDR
    """
    return transfer_midware_resp(ctrl.read(FnCode.SCREEN_HDR, device_id, screen_id))


@log_wrapper
def screen_hdr_write(device_id, screen_id, data):
    """
    设置屏幕HDR
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :param data: hdr数据
    :return: 屏幕hdr
    """
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_HDR, [data], device_id, screen_id))


@log_wrapper
def screen_clear_layers(device_id, screen_id, data):
    """
    清空屏幕图层
    :param device_id: 设备Id
    :param screen_id:  屏幕Id
    :param data: 清空数据
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_CLEAR_LAYERS, [data], device_id, screen_id))


@log_wrapper
def screen_notime_play(device_id, screen_id, data):
    """
    非实时上屏take
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :param data: 图层数据
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_NOTIME_PLAY,  data if isinstance(data, list) else [data],  device_id, screen_id))


@log_wrapper
def screen_nontime_full_detail_read(device_id, screen_id):
    """
    读指定screen的非实时状态下所有图层信息
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :return: 非实时上屏全部数据
    """
    return transfer_midware_resp(ctrl.read(FnCode.SCREEN_NONTIME_FULL_DETAIL, device_id, screen_id))

@log_wrapper
def screen_multi_nontime_full_detail_read(device_id, data):
    """
    读指定screen的非实时状态下所有图层信息
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :return: 非实时上屏全部数据
    """
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_MULTI_NONTIME_FULL_DETAIL, [data], device_id))


@log_wrapper
def screen_3d_write(device_id, screen_id, data):
    """
    设置屏幕3D
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :param data: 3D数据
    :return: 屏幕3D
    """
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_3D, [data], device_id, screen_id))


@log_wrapper
def screen_phase_offset_write(device_id, screen_id, data):
    """
    设置屏幕相位偏移
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :param data: 偏移数据
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_GENLOCK_PHASE_OFFSET, [data], device_id, screen_id))


@log_wrapper
def read_template_list(device_id):
    """
    读取屏幕布局模板列表数据
    :param device_id: 设备Id
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.read(FnCode.SCREEN_TEMPLATE_LIST, device_id))


@log_wrapper
def template_add(device_id, data):
    """
    添加屏幕布局模板列表数据
    :param device_id: 设备Id
    :param data: data
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_TEMPLATE_ADD, [data], device_id))


@log_wrapper
def template_update(device_id, data):
    """
    更新屏幕布局模板列表数据
    :param device_id: 设备Id
    :param data: data
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_TEMPLATE_ADD, [data], device_id))


@log_wrapper
def template_delete(device_id, data):
    """
    删除屏幕布局模板列表数据
    :param device_id: 设备Id
    :param data: data
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_TEMPLATE_DELETE, [data], device_id))


@log_wrapper
def read_bright_adj_list(device_id, screen_id):
    """
    读取屏幕亮度调整列表
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :param data: 偏移数据
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.read(FnCode.SCREEN_AUTO_BRIGHT_ADJ, device_id, screen_id))


@log_wrapper
def write_bright_adj_list(device_id, screen_id, data):
    """
    设置屏幕亮度调整列表
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :param data: 列表数据
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_AUTO_BRIGHT_ADJ, [data], device_id, screen_id))


@log_wrapper
def write_audio_attribute(device_id, screen_id, data):
    """
    设置屏幕音频属性
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_AUDIO_ATTRIBUTE, [data], device_id, screen_id))


@log_wrapper
def volume_write(device_id, screen_id, data):
    """
    设置屏幕音频属性
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :return: 结果对象
    """
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_VOLUME, [data], device_id, screen_id))


@log_wrapper
def screen_osd_read(device_id, screen_id):
    """
    设置屏幕OSD
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :return: 参考接口screenOSD
    """
    return transfer_midware_resp(ctrl.read(FnCode.SCREEN_OSD, device_id, screen_id))


@log_wrapper
def screen_osd_read(device_id, screen_id):
    """
    读取屏幕OSD
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :return: 参考接口screenOSD
    """
    return transfer_midware_resp(ctrl.read(FnCode.SCREEN_OSD, device_id, screen_id))



@log_wrapper
def screen_osd_weather_read(device_id, screen_id, osd_id):
    """
    读取屏幕天气OSD
    :param device_id: 设备Id
    :param screen_id: 屏幕Id
    :param osd_id: osdId
    :return:
    """
    return transfer_midware_resp(ctrl.read(FnCode.SCREEN_OSD_WEATHER, device_id, screen_id, osd_id))

@log_wrapper
def screen_dynamic_subpixel_write(device_id, data):
    """
    设置屏幕动态子像素
    :param device_id: 设备Id
    :param data: bkg 数据
    :return: 返回bkgId
    """
    return transfer_midware_resp(ctrl.write(FnCode.SCREEN_DYNAMIC_SUBPIXEL, [data], device_id))


@log_wrapper
def batch_write_screen_detail(device_id, data):
    """
    批量获取屏幕详情
    :param device_id: 设备Id
    :param screens: 屏幕Id列表
    :return: 屏幕数据列表
    """
    rs = transfer_midware_resp(ctrl.write(FnCode.SCREEN_MULTI_DETAIL, data if isinstance(data, list) else [data], device_id))
    format_server_data(rs['data'])
    return rs