import math

from H_9U.mao.impl import ipcmao
from H_9U.util.cache import cacher
from H_9U.models.cachekey import CacheKey, CacheDuration
from H_9U.models.result import get_result_model
from H_9U.models.sysconst import ModelId, SourceType


# todo: cacher


class IPCApi:
    """
    mao层接口api
    """

    def read_source_list_by_slot_id(self, device_id, slot_id):
        """
        读取ipc列表
        :param device_id: 设备id
        :param slot_id: slot id
        :return: ipc列表
        """
        rs = get_result_model()
        index = 0
        size = 200
        ipc_list = []
        while index >= 0:
            data = {
                "seqPageIndex": index,
                "seqPageSize": size,
                "deviceId": device_id,
                "slotId": slot_id
            }
            rs = ipcmao.ipc_source_list(device_id, data)
            if rs['status'] != 0:
                return rs
            if rs['data']['sourceList']:
                ipc_list.extend(rs['data']['sourceList'])
            if rs['data']['nums'] > index * size + size:
                index += 1
            else:
                index = -1
            rs['data']['sourceList'] = ipc_list
        return rs

    def ipc_source_layout_list(self, device_id, channel_num, data):
        """
        读取信号源布局列表，data为分页数据
        :param device_id: 设备Id
        :param channel_num: 需要的通道数量
        :param data: 分页参数
        :return 结果数据
        """
        data['isMontage'] = 0
        rs = ipcmao.ipc_source_list(device_id, data)
        ipc_list = []
        channel_list = {}
        if rs['status'] == 0 and rs['data']['sourceList']:
            ipc_list.extend(rs['data']['sourceList'])
            if channel_num != 0:
                lists = ipc_list[0:channel_num]
            else:
                lists = ipc_list
            for source in lists:
                source_id = source['sourceId']
                ipc_rs = self.ipc_channel_detail_list(device_id, source_id)
                if ipc_rs['status'] != 0:
                    break
                if ipc_rs['data']['channelList']:
                    channel_list = ipc_rs['data']['channelList']
                source['channelList'] = channel_list
        return rs

    def read_ipc_source_list(self, device_id, data):
        """
        读取ipc列表
        :param device_id: 设备id
        :param data: 查询参数
        :return: ipc列表
        """
        rs = ipcmao.ipc_source_list(device_id, data)
        if rs['status'] != 0 or rs['data']['sourceList'] or rs['data']['nums'] == 0:
            return rs
        data['seqPageIndex'] = math.ceil(rs['data']['nums'] / data['seqPageSize']) - 1
        return ipcmao.ipc_source_list(device_id, data)

    def read_ipc_source_free_list(self, device_id, data):
        """
        读取ipc列表
        :param device_id: 设备id
        :param data: 查询参数
        :return: ipc列表
        """
        return ipcmao.ipc_source_list(device_id, data)

    def read_ipc_camera_list(self, device_id, decode_capacity):
        """
        ipc 摄像头扫描
        :param device_id: 设备id
        :param decode_capacity: 规格
        :return: ipc扫描列表
        """
        return ipcmao.ipc_camera_list(device_id, decode_capacity)

    def read_ipc_channel_list(self, device_id, source_id):
        """
        读取ipc通道列表
        :param device_id: 设备id
        :param source_id: source id
        :return: ipc列表
        """
        key = CacheKey.ipc_source_channel_list(device_id, source_id)
        return cacher.IPCCache.try_get_value(key, CacheDuration.Default, ipcmao.ipc_channel_list, device_id, source_id)

    def ipc_channel_detail_list(self, device_id, source_id):
        """
        ipc通道详情
        :param device_id: 设备id
        :param source_id: source id
        :return: ipc列表
        """
        key = CacheKey.ipc_source_channel_detail_list(device_id, source_id)
        rs = cacher.IPCCache.get(key, None)
        if not rs:
            rs = self.read_ipc_channel_list(device_id, source_id)
            if rs['status'] == 0 and rs['data']:
                channel_list = []
                for channel in rs['data']['channelList']:
                    channel_id = channel['channelId']
                    if channel_id != 65535:
                        channel_rs = ipc_api.read_ipc_channel_detail(device_id, source_id, channel_id)
                        if channel_rs['status'] == 0 and channel_rs['data']:
                            channel_list.append(channel_rs['data'])
                rs['data']['channelList'] = channel_list
                cacher.IPCCache.set(key, rs, CacheDuration.Default)
        return rs

    def read_ipc_channel_detail(self, device_id, source_id, channel_id):
        """
        读取ipc通道详情
        :param device_id: 设备id
        :param source_id: source id
        :param channel_id: channel_id
        :return: ipc列表
        """
        key = CacheKey.ipc_channel_detail(device_id, source_id, channel_id)
        return cacher.IPCCache.try_get_value(key, CacheDuration.Default, ipcmao.ipc_channel_detail,  device_id, source_id, channel_id)


    def write_ipc_channel_edit(self, device_id, params):
        """
        修改通道数据
        :param device_id: 设备Id
        :param params: params
        """
        rs = ipcmao.ipc_channel_edit(device_id, params)
        if rs['status'] == 0:
            source_id = params['sourceId']
            channel_id = params['channelId']
            cacher.IPCCache.delete(CacheKey.ipc_channel_detail(device_id, source_id, channel_id))
            cacher.IPCCache.delete(CacheKey.ipc_source_channel_list(device_id, source_id))
            cacher.IPCCache.delete(CacheKey.ipc_source_channel_detail_list(device_id, source_id))
            cacher.LayerCache.clear()
        return rs

    def write_ipc_channel_add(self, device_id, params):
        """
        add通道数据
        :param device_id: 设备Id
        :return 结果数据
        """
        rs = ipcmao.ipc_source_append(device_id, params)
        if rs['status'] == 0:
            source_id = params['sourceId']
            cacher.IPCCache.delete(CacheKey.ipc_source_channel_list(device_id, source_id))
            cacher.IPCCache.delete(CacheKey.ipc_source_channel_detail_list(device_id, source_id))
        return rs

    def write_ipc_channel_delete(self, device_id, params):
        """
        delete通道数据
        :param device_id: 设备Id
        :return 结果数据
        """
        rs = ipcmao.ipc_channel_delete(device_id, params)
        if rs['status'] == 0 and rs['data']:
            cacher.LayerCache.clear()
            for source in rs['data']['sourceList']:
                source_id = source['sourceId']
                for channel in source['channelList']:
                    if channel['result'] == 0:
                        channel_id = channel['channelId']
                        cacher.IPCCache.delete(CacheKey.ipc_channel_detail(device_id, source_id, channel_id))
                        cacher.IPCCache.delete(CacheKey.ipc_source_channel_list(device_id, source_id))
                        cacher.IPCCache.delete(CacheKey.ipc_source_channel_detail_list(device_id, source_id))
        return rs

    def write_ipc_source_create(self, device_id, slot_id, data):
        """
        创建ipc源
        :param device_id: 设备id
        :param slot_id: slot id
        :param data: 源信息
        :return: 源id
        """
        rs = ipcmao.ipc_source_create(device_id, slot_id, data)
        if rs['status'] == 0:
            cacher.IPCCache.clear()
        return rs

    def ipc_source_update_Info(self, device_id, data):
        """
        编辑ipc源
        :param device_id: 设备id
        :param data: 数据
        :return: 结果对象
        """
        rs = ipcmao.ipc_source_update_Info(device_id, data)
        if rs['status'] == 0:
            cacher.LayerCache.clear()
            source_id = data['sourceId']
            cacher.IPCCache.delete(CacheKey.ipc_source_channel_list(device_id, source_id))
            cacher.IPCCache.delete(CacheKey.ipc_source_channel_detail_list(device_id, source_id))
            cs = ipcmao.ipc_channel_list(device_id, source_id)
            if cs['status'] == 0:
                for channel in cs['data']['channelList']:
                    channel_id = channel['channelId']
                    cacher.IPCCache.delete(CacheKey.ipc_channel_detail(device_id, source_id, channel_id))
        return rs

    def ipc_source_detail_Info(self, device_id, source_id):
        """
        读取ipc源详情
        :param device_id: 设备id
        :param source_id: source_id
        :return: 结果对象
        """
        return ipcmao.ipc_source_general_read(device_id, source_id)

    def write_ipc_source_append(self, device_id, data):
        """
        补充通道数据
        :param device_id: 设备id
        :param data: 源信息
        :return: 源id
        """
        rs = ipcmao.ipc_source_append(device_id, data)
        if rs['status'] == 0:
            source_id = data['sourceId']
            cacher.IPCCache.delete(CacheKey.ipc_source_channel_list(device_id, source_id))
            cacher.IPCCache.delete(CacheKey.ipc_source_channel_detail_list(device_id, source_id))
        return rs

    def write_ipc_source_delete(self, device_id, data):
        """
        删除信号源
        :param device_id: 设备Id
        :param data: 源数据
        :return: 结果对象
        """
        rs = ipcmao.ipc_source_delete(device_id, data)
        if rs['status'] == 0 and rs['data']:
            cacher.LayerCache.clear()
            for source in rs['data']['sourceList']:
                if source['result'] == 0:
                    source_id = source['sourceId']
                    cacher.IPCCache.delete(CacheKey.ipc_source_channel_list(device_id, source_id))
                    cacher.IPCCache.delete(CacheKey.ipc_source_channel_detail_list(device_id, source_id))
        return rs

    def write_ipc_source_general(self, device_id, source_id, data):
        """
        修改信号源名称
        :param device_id: 设备Id
        :param source_id:  信号源id
        :param data: 修改数据
        :return: 结果对象
        """
        rs = ipcmao.ipc_source_general(device_id, source_id, data)
        if rs['status'] == 0:
            cacher.LayerCache.clear()
        return rs

    def read_ipc_crop_list(self, device_id, source_id, source_type=SourceType.IPCSource):
        """
        读取IPC截取列表
        :param device_id: 设备id
        :param source_id: 源id
        :return:
        """
        return ipcmao.ipc_crop_list(device_id, source_id, source_type)

    def write_ipc_crop_create(self, device_id, source_id, data):
        """
        创建IPC截取
        :param device_id: 设备id
        :param source_id: 源id
        :param data:
        :return:
        """
        rs = ipcmao.ipc_crop_create(device_id, source_id, data)
        if rs['status'] == 0:
            cacher.clear_all()
        return rs

    def write_ipc_crop_delete(self, device_id, source_id, data):
        """
        删除IPC截取
        :param device_id: 设备id
        :param source_id: 源id
        :param data:
        :return:
        """
        rs = ipcmao.ipc_crop_delete(device_id, source_id, data)
        if rs['status'] == 0:
            cacher.clear_all()
        return rs

    def write_ipc_crop_general(self, device_id, data):
        """
        修改ipc截取
        :param device_id: 设备id
        :param source_id: 源id
        :param crop_id: crop_id
        :param data:
        :return:
        """
        rs = ipcmao.ipc_crop_general_write(device_id,  data)
        if rs['status'] == 0:
            cacher.clear_all()
        return rs

    def read_ipc_crop_detail(self, device_id, source_id, crop_id):
        """
        读取IPC截取详情
        :param device_id: 设备id
        :param source_id: 源id
        :param crop_id: 截取id
        :return:
        """
        return ipcmao.ipc_crop_general_read(device_id, source_id, crop_id)

    def read_ipc_group_list(self, device_id):
        """
        读取IPC分组列表
        :param device_id: 设备id
        :return:
        """
        return ipcmao.ipc_group_list(device_id)

    def ipc_group_create(self, device_id, data):
        """
        ipc分组创建
        :param device_id: 设备id
        :return:
        """
        return ipcmao.ipc_group_create(device_id, data)

    def ipc_group_update(self, device_id, data):
        """
        ipc分组创建
        :param device_id: 设备id
        :param data: data
        :return:
        """
        return ipcmao.ipc_group_create(device_id, data)

    def ipc_group_node_add(self, device_id, data):
        """ipc分组添加节点"""
        return ipcmao.ipc_group_node_add(device_id, data)

    def ipc_group_delete(self, device_id, data):
        """
        ipc分组删除
        :param device_id: 设备id
        :param data: 分组信息
        :return:
        """
        return ipcmao.ipc_group_delete(device_id, data)

    def ipc_group_node_delete(self, device_id, data):
        """
        ipc分组节点删除
        :param device_id: 设备id
        :param group_id: 分组Id
        :param data: 节点列表
        :return: 结果对象
        """
        return ipcmao.ipc_group_node_delete(device_id, data)

    def read_ipc_group_node_list(self, device_id, data):
        """
        读取IPC分组列表
        :param device_id: 设备id
        :param data: data
        :return:
        """
        return ipcmao.ipc_group_node_list(device_id, data)

    def read_ipc_group_node_list_without_page(self, device_id, group_id):
        """
        读取IPC分组列表 不分页
        :param device_id: 设备id
        :param group_id: group_id
        :return:
        """
        index = 1
        size = 100
        data = {'deviceId': 0, 'groupId': group_id}
        source_list = []
        while index > 0:
            data['seqPageIndex'] = index - 1
            data['seqPageSize'] = size
            rs = self.read_ipc_group_node_list(device_id, data)
            if rs['status'] == 0:
                source_list.extend(rs['data']['sourceList'])
                nums = rs['data']['nums']
                if math.ceil(nums / size) > index:
                    index += 1
                else:
                    index = 0
            else:
                break
        rs = get_result_model()
        rs['data'] = {
                'deviceId': 0, 'groupId': group_id, 'sourceList': source_list
            }
        return rs

    def ipc_group_clear(self, device_id, group_id, data):
        """
        清空分组数据
        :param device_id: 设备id
        :param data: 分组数据
        :param group_id: group id
        :return: ipc列表
        """
        rs = ipcmao.ipc_group_node_clear(device_id, group_id, data)
        return rs

    def read_ipc_slot_source_count(self, device_id, slot_id):
        """
        查询slot源数量
        :param device_id: 设备Id
        :param slot_id: slot Id
        """
        return ipcmao.ipc_slot_source_count_read(device_id, slot_id)

    def read_ipc_slot_source_list(self, device_id, slot_id):
        """
        不分页获取slot 源数据
        :param device_id: 设备Id
        :param slot_id: slot Id
        """
        page_index = 1
        page_size = 100
        page_count = 1
        source_list = []
        while page_index <= page_count:
            data = {'deviceId': device_id, 'slotId': slot_id, 'seqPageIndex': page_index - 1, 'seqPageSize': page_size}
            rs = self.read_ipc_source_list(device_id, data)
            if rs['status'] == 0:
                source_list.extend(rs['data']['ipcSourceList'])
                page_count = math.ceil(rs['data']['nums'] / page_size)
            page_index += 1
        rs = get_result_model()
        rs['data'] = {'deviceId': device_id, 'slotId': slot_id, 'ipcSourceList': source_list}
        return rs

    def read_ipc_search_source_list(self, device_id, data):
        """
        不分页获取slot 源数据
        :param device_id: 设备Id
        :param data: data
        """
        page_index = 1
        page_size = 100
        page_count = 1
        source_list = []
        while page_index <= page_count:
            data['seqPageIndex'] = page_index - 1
            data['seqPageSize'] = page_size
            rs = self.read_ipc_source_list(device_id, data)
            if rs['status'] == 0:
                source_list.extend(rs['data']['sourceList'])
                page_count = math.ceil(rs['data']['nums'] / page_size)
            page_index += 1
        rs = get_result_model()
        rs['data'] = {'deviceId': device_id, 'sourceList': source_list}
        return rs

    def read_ipc_onvif_source_list(self, device_id):
        """
        ipc onvif 源列表
        :param device_id: 设备Id
        :return:
        """
        data = {'deviceId': device_id, 'type': 2}
        page_index = 1
        page_size = 100
        page_count = 1
        source_list = []
        while page_index <= page_count:
            data['seqPageIndex'] = page_index - 1
            data['seqPageSize'] = page_size
            rs = self.read_ipc_source_list(device_id, data)
            if rs['status'] == 0:
                source_list.extend(rs['data']['ipcSourceList'])
                page_count = math.ceil(rs['data']['nums'] / page_size)
            page_index += 1
        rs = get_result_model()
        rs['data'] = {'deviceId': device_id, 'ipcSourceList': source_list}
        return rs

    def read_ipc_stream_detail(self, device_id, data):
        """
        读取ipc码流详情
        :param device_id: 设备Id
        :param data: 码流Id list
        :return: 详细信息
        """
        return ipcmao.read_ipc_stream_detail(device_id, data)

    def read_ipc_stream_list(self, device_id):
        """
        读取ipc码流详情
        :param device_id: 设备Id
        :param data: 码流Id list
        :return: 详细信息
        """
        return ipcmao.read_ipc_stream_list(device_id)
    # ipc 拼接 开始
    def ipc_montage_list(self, device_id, slot_id):
        """ipc拼接源列表"""
        return ipcmao.ipc_montage_list(device_id, slot_id)

    def ipc_montage_group_list(self, device_id, slot_id):
        """ipc拼接源列表"""
        key = CacheKey.ipc_montage_group_list(device_id, slot_id)
        return cacher.IPCCache.try_get_value(key, CacheDuration.Default, ipcmao.ipc_montage_group_list, device_id, slot_id)

    def ipc_montage_template_create(self, device_id, data):
        """ipc拼接源模板创建"""
        cacher.IPCCache.clear()
        return ipcmao.ipc_montage_create(device_id, data)

    def ipc_montage_template_detail(self, device_id, template_id):
        """ipc拼接源模板详情"""
        return ipcmao.ipc_montage_template_detail(device_id, template_id)

    def ipc_montage_template_update(self, device_id, data):
        """ipc拼接源模板修改"""
        cacher.IPCCache.clear()
        return ipcmao.ipc_montage_create(device_id, data)

    def ipc_montage_template_delete(self, device_id, data):
        """ipc拼接源模板删除"""
        cacher.IPCCache.clear()
        return ipcmao.ipc_montage_delete(device_id, data)

    def ipc_montage_template_apply(self, device_id, data):
        """ipc拼接源模板应用"""
        rs = ipcmao.ipc_montage_apply(device_id, data)
        if rs['status'] == 0:
            cacher.clear_all()
        return rs

    def ipc_montage_rename(self, device_id, data):
        """ipc拼接源重命名"""
        rs = ipcmao.ipc_montage_rename(device_id, data)
        if rs['status'] == 0:
            cacher.LayerCache.clear()
        return rs

    def ipc_source_status_list(self, device_id, group_id, data):
        """
        ipc组内源信号的处理
		:param device_id: 设备Id
		:param group_id: group id
		:return: 结果对象
		"""
        return ipcmao.ipc_source_status_list(device_id, group_id, data)


    def write_ipc_bg_color(self, device_id, data):
        """
        设置IPC无源显示画面
        :param device_id: 设备id
        :param data: 设备参数
        :return: 结果对象
        """
        return ipcmao.write_ipc_bg_color(device_id, data)

    def read_ipc_bg_color(self, device_id):
        """
        查询IPC无源显示画面
        :param device_id: 设备id
        :return: 结果对象
        """
        return ipcmao.read_ipc_bg_color(device_id)

    def write_ipc_stream_rule(self, device_id, data):
        """
        设置IPC码流规则
        :param device_id: 设备id
        :param data: 码流规则参数
        :return: 结果对象
        """
        return ipcmao.write_ipc_stream_rule(device_id, data)

    def read_ipc_stream_rule(self, device_id):
        """
        查询IPC码流规则
        :param device_id: 设备id
        :return: 结果对象
        """
        return ipcmao.read_ipc_stream_rule(device_id)

    def ipc_first_channel_detail(self, device_id, source_id):
        rs = self.read_ipc_channel_list(device_id, source_id)
        if rs['status'] != 0 or not rs['data']:
            return rs
        channel_info = None
        if rs['data']['channelList']:
            channel_id = rs['data']['channelList'][0]['channelId']
            if channel_id != 65535:
                channel_rs = ipc_api.read_ipc_channel_detail(device_id, source_id, channel_id)
                if channel_rs['status'] == 0 and channel_rs['data']:
                    channel_info = channel_rs['data']
        rs['data']['firstChannel'] = channel_info
        return rs

    def ipc_montage_template_zorder_update(self, device_id, data):
        """ipc拼接源模板修改"""
        cacher.IPCCache.clear()
        return ipcmao.ipc_montage_template_zorder_update(device_id, data)

    def ipc_montage_group_update(self, device_id, data):
        """ipc拼接源模板修改"""
        cacher.IPCCache.clear()
        return ipcmao.ipc_montage_group_update(device_id, data)

    def ipc_montage_poll_status_write(self, device_id, data):
        """ipc拼接源模板修改"""
        cacher.IPCCache.clear()
        return ipcmao.ipc_montage_poll_status_write(device_id, data)

ipc_api = IPCApi()