"""
Created on 2021年9月07日
@author: 程声清
Screen接口.
对应前端接口
返回值统一为 {data:JSON结构体, msg:"", status:0}
"""
from copy import deepcopy

from flask import Blueprint, g
from flask.json import jsonify

from H_9U.api.output import output_read_all_list
from H_9U.api.screen import screen_detail, screen_list
from H_9U.api.websender import push_data, push_data_org
from H_9U.models.result import get_result_model, ResInfo
from H_9U.open.service import open_screen_svc
from H_9U.util.common import get_params
from H_9U.util.logwrapper import open_log
from H_9U.models.syncdataname import SyncDataName

open_screen_bp = Blueprint('open_screen', __name__)


@open_screen_bp.route('/createScreen', methods=['GET', 'POST'])
@open_log
def screen_create():
    """
    创建Screen
    :return:参考文档createScreen
    """
    params = g.data
    values = get_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = open_screen_svc.screen_create(device_id, params)
    if rs['status'] == 0:
        screen_id = rs['data']['id']
        push_data(output_read_all_list, SyncDataName.Output_Read_List, device_id)
        push_data(screen_detail, SyncDataName.Screen_Create, device_id, screen_id)
    return jsonify((rs))


@open_screen_bp.route('/deleteScreen', methods=['GET', 'POST'])
@open_log
def screen_delete():
    """
    删除Screen
    :return: 参考文档deleteScreen
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenIdList')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_list = values
    rs = open_screen_svc.screen_delete(device_id, screen_list)
    if rs['status'] == 0:
        push_data(output_read_all_list, SyncDataName.Output_Read_List, device_id)
        for screen_id in screen_list:
            push_data_org(SyncDataName.Screen_Delete, {'deviceId': device_id, 'screenId': screen_id})
    return jsonify((rs))


@open_screen_bp.route('/clear', methods=['GET', 'POST'])
@open_log
def screen_delete_screen_batch():
    """
    批量删除Screen
    :return: 参考文档deleteScreen
    """
    params = g.data
    device_id = params['deviceId']
    rs = open_screen_svc.screen_delete_batch(device_id)
    if rs['status'] == 0:
        push_data(output_read_all_list, SyncDataName.Output_Read_List, device_id)
        push_data(screen_list, SyncDataName.Screen_Read_All_List, device_id)
        push_data(screen_list, SyncDataName.Screen_Clear_data, device_id)
    return jsonify((rs))


@open_screen_bp.route('/readDetail', methods=['GET', 'POST'])
@open_log
def screen_read_detail():
    """
    读取屏幕详情
    :return: 参考readDetail
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    rs = open_screen_svc.screen_read_detail(device_id, screen_id)
    return jsonify((rs))


@open_screen_bp.route('/readList', methods=['GET', 'POST'])
@open_log
def screen_read_list():
    """
    读取Screen列表
    :return: readAllList
    """
    params = g.data
    values = get_params(params, 'deviceId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id = values
    rs = open_screen_svc.screen_read_list(device_id)
    return jsonify((rs))


@open_screen_bp.route('/rename', methods=['GET', 'POST'])
@open_log
def screen_rename():
    """
    屏幕重命名
    :return:
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId', 'name')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, name = values
    rs = open_screen_svc.screen_rename(device_id, screen_id, params)
    return jsonify((rs))


@open_screen_bp.route('/ftb', methods=['GET', 'POST'])
@open_log
def screen_ftb():
    """
    ftb
    :return:参考文档ftb
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    rs = open_screen_svc.ftb(device_id, screen_id, params)
    if rs['status'] == 0:
        params['screenIdList'] = [screen_id]
        push_data_org(SyncDataName.Screen_Ftb, params)
        push_data(output_read_all_list, SyncDataName.Output_Read_List, device_id)
    return jsonify((rs))


@open_screen_bp.route('/writeBKG', methods=['GET', 'POST'])
@open_log
def screen_write_bkg():
    """
    设置屏幕BKG
    :return:参考文档 writeBkg
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId', 'bkgId', 'enable')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, bkg_id, enable = values
    rs = open_screen_svc.screen_bkg_write(
        device_id, screen_id, bkg_id, enable, params)
    if rs['status'] == 0:
        if rs['status'] == 0:
            image_url = ''
            if enable == 1:
                from H_9U.api.bkg import bkg_image_url
                bkg_rs = bkg_image_url(device_id, bkg_id)
                if bkg_rs['status'] == 0 and bkg_rs['data'] and bkg_rs['data']['imageUrl']:
                    image_url = bkg_rs['data']['imageUrl']
            params['imageUrl'] = image_url
            push_data_org(SyncDataName.Screen_Write_BKG, params)
    return jsonify((rs))


@open_screen_bp.route('/writeFreeze', methods=['GET', 'POST'])
@open_log
def screen_write_freeze():
    """
    freeze
    :return:参考文档freeze
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    rs = open_screen_svc.screen_write_freeze(device_id, screen_id, params)
    if rs['status'] == 0:
        params['screenIdList'] = [screen_id]
        push_data_org(SyncDataName.Screen_Write_Freeze, params)
        push_data(output_read_all_list, SyncDataName.Output_Read_List, device_id)
    return jsonify((rs))


@open_screen_bp.route('/writeImageOSD', methods=['GET', 'POST'])
@open_log
def screen_write_image_osd():
    """
    设置屏幕osd
    :return:参考文档writeOsd
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    rs = open_screen_svc.screen_write_image_osd(
        device_id, screen_id, params)
    if rs['status'] == 0:
        screen_detail = open_screen_svc.read_detail(device_id, screen_id)
        if screen_detail['status'] == 0 and screen_detail['data']:
            osd_data = {
                "deviceId": device_id,
                "screenId": screen_id,
                "Osd": screen_detail['data']['Osd'],
                "OsdImage": screen_detail['data']['OsdImage']
            }
            push_data_org(SyncDataName.Screen_Write_OSD, osd_data)
            rs['data'] = {'Osd': screen_detail['data']['Osd']}
    return jsonify((rs))


@open_screen_bp.route('/writeOSD', methods=['GET', 'POST'])
@open_log
def screen_write_osd():
    """
    设置屏幕osd
    :return:参考文档writeOsd
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    osd_detail = open_screen_svc.osd_read(device_id, screen_id)
    if osd_detail['status'] != 0:
        return jsonify(get_result_model(ResInfo.Screen_Not_Exist))
    osd = deepcopy(osd_detail['data'])
    osd['enable'] = params['enable']
    osd['screenId'] = screen_id
    osd['deviceId'] = device_id

    from H_9U.util.common import valid_str_value
    for text in osd['text']:
        if not valid_str_value(text['words'], 0, 1024):
            return jsonify(get_result_model(ResInfo.Screen_OSD_Words_Length_Err))

    rs = open_screen_svc.osd_write(device_id, screen_id, osd)
    return jsonify((rs))


@open_screen_bp.route('/nonTimeDetail', methods=['GET', 'POST'])
@open_log
def non_time_detail():
    """
   读指定 screen 的非实时状态下所有图层信息
    :return:参考文档nonTimeDetail
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    rs = open_screen_svc.screen_non_time_detail(device_id, screen_id)
    return jsonify((rs))


@open_screen_bp.route('/write3D', methods=['GET', 'POST'])
@open_log
def screen_write_3d():
    """
   读指定 screen 的非实时状态下所有图层信息
    :return:参考文档nonTimeDetail
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    rs = open_screen_svc.screen_write_3d(
        device_id, screen_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Screen_Write_3D, params)
    return jsonify((rs))


@open_screen_bp.route('/writeOutputMode', methods=['GET', 'POST'])
@open_log
def screen_write_output_mode():
    """
    读取屏幕output布局
    :return: 见文档writeOutputMode
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    rs = open_screen_svc.screen_write_output_mode(
        device_id, screen_id, params)
    if rs['status'] == 0:
        push_data(output_read_all_list, SyncDataName.Output_Read_List, device_id)
        push_data(screen_detail, SyncDataName.Screen_Write_Output_Mode, device_id, screen_id)
    return jsonify((rs))


@open_screen_bp.route('/writeGamma', methods=['GET', 'POST'])
@open_log
def screen_write_gamma():
    """
    :return:参考文档nonTimeDetail
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values
    rs = open_screen_svc.screen_write_gamma(
        device_id, screen_id, params)
    if rs['status'] == 0:
        push_data_org(SyncDataName.Screen_Write_Gamma, params)
    return jsonify((rs))


@open_screen_bp.route('/writeBrightness', methods=['GET', 'POST'])
@open_log
def screen_write_brightness():
    """
    设置屏幕Brightness
    :return: 见文档writeBrightness
    """
    params = g.data
    values = get_params(params, 'deviceId', 'screenId')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id = values

    rs = open_screen_svc.write_brightness(device_id, screen_id, params)
    return jsonify((rs))


@open_screen_bp.route('/saveBrightness', methods=['GET', 'POST'])
@open_log
def screen_save_brightness():
    params = g.data
    values = get_params(params, 'deviceId', 'screenId', 'brightness')
    if values is None:
        return jsonify(get_result_model(ResInfo.Params_Error))
    device_id, screen_id, brightness = values
    rs = open_screen_svc.save_brightness(device_id, screen_id, params)
    return jsonify((rs))
