"""
Created on 2019年6月28日
@author: 刘益宗
中间件路径处理函数

1、 中间件工作目录动态获取，以 data为关键字
2、 中间件data路径创建软连接，放到static文件夹中，名称为data
3、 中间件返回文件路径为data的相对路径，以./data开头
4、 使用时自动处理
"""
import os
import threading
import shutil

from H_9U.conf.syssettings import SysSettings
from H_9U import BASE_DIR
from H_9U.util.fileopt import getdirsize
from H_9U.service.common import make_thumbnail
from H_9U.api.log import logger
from H_9U.models.sysconst import DeviceType


class BaseManager:

    def __init__(self, slot_link_from=SysSettings.Middleware_Data_Abs_Path,
                 slot_link_to=SysSettings.Middleware_Web_Data_Path):
        self._slot_link_from = slot_link_from  # 中间件根路径
        self._slot_link_to = slot_link_to  # 中间件对应的软连接
        self.device_type = DeviceType.Unknown
        self._web_base_path = BASE_DIR  # web根路径
        self._lock = threading.Lock()

    def init_path(self):
        pass

    def make_symlink(self):
        """
        生软连接
        :return:
        """
        if self._slot_link_from is None or self._slot_link_to is None:
            self.init_path()
        if self._slot_link_from is None or self._slot_link_to is None:
            return

        # 软连接不存在则生成
        self._lock.acquire()
        try:
            if not os.path.exists(os.path.join(self._web_base_path, self._slot_link_to)) \
                    and os.path.exists(self._slot_link_from):
                if not os.path.exists(os.path.join(self._web_base_path, self._slot_link_to)):
                    os.symlink(self._slot_link_from, os.path.join(self._web_base_path, self._slot_link_to))
        except Exception as e:
            logger.error(e)
        finally:
            self._lock.release()

    def _get_device_type(self):
        if self.device_type == DeviceType.Unknown:
            from H_9U.service.device import device_svc
            rs = device_svc.device_detail_read(SysSettings.Device_Id)
            if rs['status'] == 0:
                model_id = rs['data']['modelId']
                if model_id in DeviceType.HS_ALL or model_id in DeviceType.Alita_ALL or model_id in DeviceType.P_Main_All:
                    self.device_type = DeviceType.HS
        return self.device_type


class BkgManager(BaseManager):
    """
    bkg路径处理
    """

    def __init__(self):
        BaseManager.__init__(self, slot_link_from=SysSettings.Middleware_Data_Abs_Path,
                             slot_link_to=SysSettings.Middleware_Web_Data_Path)
        self._bkg_folder_name = SysSettings.Middleware_BKG_Folder
        self.thumbnail_path = SysSettings.Image_Thumbnail_Path

    def get_bkg_path_info(self, bkg_relative_path):
        """
        获取bkg图片绝对路径、相对url、缩略图绝对路径，相对url
        :param bkg_relative_path: bkg相对路径
        :return: bkg图片绝对路径、相对url、缩略图绝对路径，相对url
        """
        self.make_symlink()
        if bkg_relative_path.startswith('./data'):
            bkg_relative_path = bkg_relative_path[7:]
        file_path = os.path.join(self._slot_link_from, bkg_relative_path)
        file_url = '/data/' + bkg_relative_path

        thumbnail_path = ''
        thumbnail_url = ''
        # 文件存在生成缩略图
        if os.path.exists(file_path):
            path, file_info = os.path.split(file_path)
            name, ext = os.path.splitext(file_info)
            name = name + '_thumbnail' + ext
            thumbnail_path = os.path.join(BASE_DIR, SysSettings.Image_Thumbnail_Path, name)
            if not os.path.exists(thumbnail_path):
                # 不存在则创建缩略图，创建失败返回None
                make_thumbnail(file_path, thumbnail_path=thumbnail_path)

            thumbnail_url = 'thumbnail/' + name

        return file_path, file_url, thumbnail_path, thumbnail_url

    def _get_bkg_forder(self):
        """
        获取bkg文件夹路径
        :return:
        """
        return os.path.join(self._slot_link_from, self._bkg_folder_name)

    def valid_bkg_file_size(self, bkg_path):
        """
        验证bkg文件夹容量是否超限
        :param bkg_path: bkg文件夹路径
        :return: true 不超限 false 超限
        """
        bkg_folder_path = self._get_bkg_forder()
        if not os.path.exists(bkg_folder_path):
            return True
        file_size = os.path.getsize(bkg_path)
        total_size = getdirsize(bkg_folder_path)
        return SysSettings.Max_BKG_File_Length > total_size + file_size


class InputOsdManager(BaseManager):
    """
    输入源osd路径处理
    """

    def __init__(self):
        BaseManager.__init__(self, slot_link_from=SysSettings.Middleware_Data_Abs_Path,
                             slot_link_to=SysSettings.Middleware_Web_Data_Path)

    def get_osd_path_info(self, osd_relative_path):
        """
        osd图片绝对路径和相对url
        :param osd_relative_path:
        :return: osd图片绝对路径和相对url
        """
        self.make_symlink()
        if osd_relative_path.startswith('./data'):
            osd_relative_path = osd_relative_path[7:]
        file_path = os.path.join(self._slot_link_from, osd_relative_path)
        file_url = '/data/' + osd_relative_path
        return file_path, file_url


class ScreenOsdManager(BaseManager):
    """
    屏幕osd路径处理
    """

    def __init__(self):
        BaseManager.__init__(self,
                             slot_link_from=None, slot_link_to=SysSettings.Middleware_Web_Data_Path)

    def init_path(self):
        if self._slot_link_from is None:
            self._get_device_type()
            if self.device_type == DeviceType.ES:
                self._slot_link_from = SysSettings.E_Middleware_Data_Abs_Path
            elif self.device_type == DeviceType.HS:
                self._slot_link_from = SysSettings.Middleware_Data_Abs_Path

    def get_osd_path_info(self, osd_relative_path):
        """
        osd图片绝对路径和相对url
        :param osd_relative_path:
        :return: osd图片绝对路径和相对url
        """
        file_path = file_url = ''
        if osd_relative_path:
            self.make_symlink()
            if osd_relative_path.startswith('./data'):
                osd_relative_path = osd_relative_path[7:]
            file_path = os.path.join(self._slot_link_from, osd_relative_path)
            file_url = '/data/' + osd_relative_path
        return file_path, file_url


class NomarkModeManager(BaseManager):
    def __init__(self):
        BaseManager.__init__(self,
                             slot_link_from=None,
                             slot_link_to=os.path.join(SysSettings.Nomark_Mode_File_Path, 'data'))
        self.lock = threading.Lock()

    def init_path(self):
        if self._slot_link_from is None:
            self._get_device_type()
            if self.device_type == DeviceType.ES:
                self._slot_link_from = SysSettings.E_Middleware_Nomark_Data_Abs_Path
            elif self.device_type == DeviceType.HS:
                self._slot_link_from = SysSettings.Middleware_Nomark_Data_Abs_Path

    def get_default_image_urls(self):
        logo_url = '/nomarkdata/img_logo_default_copy.png'
        icon_url = '/nomarkdata/img_icon_default_copy.png'
        bkg_url = '/nomarkdata/img_bkg_default_copy.png'
        self.get_default_image_paths()

        return logo_url, icon_url, bkg_url

    def get_default_image_paths(self):
        logo_path = os.path.join(BASE_DIR, SysSettings.Nomark_Mode_File_Path, 'img_logo_default_copy.png')
        icon_path = os.path.join(BASE_DIR, SysSettings.Nomark_Mode_File_Path, 'img_icon_default_copy.png')
        bkg_path = os.path.join(BASE_DIR, SysSettings.Nomark_Mode_File_Path, 'img_bkg_default_copy.png')
        try:
            with self.lock:
                if not os.path.exists(logo_path):
                    shutil.copy(os.path.join(BASE_DIR, SysSettings.Nomark_Mode_File_Path, 'img_logo_default.png'),
                                logo_path)
                if not os.path.exists(icon_path):
                    shutil.copy(os.path.join(BASE_DIR, SysSettings.Nomark_Mode_File_Path, 'img_icon_default.png'),
                                icon_path)
                if not os.path.exists(bkg_path):
                    shutil.copy(os.path.join(BASE_DIR, SysSettings.Nomark_Mode_File_Path, 'img_bkg_default.png'),
                                bkg_path)
        except Exception as e:
            logger.error('中性设置处理默认图片错误')
            logger.exception(e)
        return logo_path, icon_path, bkg_path

    def format_image_url(self, path):
        self.make_symlink()
        if path:
            filename = os.path.split(path)[1]
            path = os.path.join(os.path.split(SysSettings.Nomark_Mode_File_Path)[1], 'data', filename)
            if not path.startswith('/'):
                path = '/' + path
        return path


class FontManager(BaseManager):
    """
    输入源osd路径处理
    """

    def __init__(self):
        BaseManager.__init__(self,
                             slot_link_from=SysSettings.Middleware_Font_Data_Abs_Path,
                             slot_link_to=SysSettings.Link_Font_File_Path)

    def get_font_path_info(self, font_abs_path):
        """
        osd图片绝对路径和相对url
        :param osd_relative_path:
        :return: osd图片绝对路径和相对url
        """
        self.make_symlink()
        file_url = os.path.join('/midfont', font_abs_path.replace(SysSettings.Middleware_Font_Data_Abs_Path, ''))
        return file_url


bkg_manager = BkgManager()
input_osd_manager = InputOsdManager()
nomark_manager = NomarkModeManager()
screen_osd_manager = ScreenOsdManager()
font_manager = FontManager()


