"""
Created on 2019年6月28日
@author: 刘益宗
场景接口.
返回值统一为 {data:JSON结构体, msg:"", status:0}
"""

from H_9U.dao.modelconf import model_conf_dao
from H_9U.models.sysconst import ModelConfKey
from H_9U.models.result import get_result_model, ResInfo
from H_9U.mao.preset_mao import preset_api



class PresetSvc:

    def preset_create(self, device_id, screen_id, params):
        """
        创建场景
        :param device_id: 设备Id
        :param screen_id: 屏幕号
        :param params: 创建场景参数
        :return: 新增场景Id
        """
        return preset_api.write_create(device_id, screen_id, params)

    def preset_copy(self, device_id, screen_id, params):
        """
        场景复制
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :param params:  复制场景参数
        :return:  复制后场景id
        """
        return preset_api.write_copy(device_id, screen_id, params)

    def preset_delete(self, device_id, screen_id, preset_id, params):
        """
        删除场景
        :param params: 删除场景参数
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :param preset_id: 场景Id
        :return: 结果对象
        """
        return preset_api.write_delete(device_id, screen_id, preset_id, params)

    def preset_general_write(self, device_id, screen_id, preset_id, params):
        """
        读取场景基本信息
        :param params: 场景信息参数
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :param preset_id: 场景Id
        :return: 参考中间件协议PresetGeneral部分
        """
        return preset_api.write_general(device_id, screen_id, preset_id, params)

    def preset_play_write(self, device_id, screen_id, preset_id, params):
        """
        场景播放
        :param params:播放场景参数
        :return: 结果对象
        """
        rs = preset_api.write_play(device_id, screen_id, preset_id, params)
        if rs['status'] == 0:
            key = str(device_id) + '|' + str(screen_id)
            model_conf_dao.set_value(ModelConfKey.PresetPlay, key, preset_id)
        return rs

    def preset_play_screen(self, device_id, screen_id):
        """
        读取屏幕播放的场景
        :param device_id:  设备id
        :param screen_id:  屏幕Id
        :return:  场景Id
        """
        key = str(device_id) + '|' + str(screen_id)
        rs = model_conf_dao.get_value(ModelConfKey.PresetPlay, key)
        if rs['status'] == 0:
            preset_id = rs['data']['value']
            rs['data'] = {
                'deviceId': device_id,
                'screenId': screen_id,
                'presetId': preset_id
            }
        return rs

    def preset_detail_read(self, device_id, screen_id, preset_id):
        """
        场景详细信息
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :param preset_id: 场景Id
        :return: 参考中间件协议场景详情
        """
        return preset_api.read_detail(device_id, screen_id, preset_id)

    def preset_delete_all(self, device_id, screen_id):
        """
        删除屏幕下全部场景
        :param device_id:  设备Id
        :param screen_id:  屏幕Id
        :return: 删除场景列表
        """
        # 如果中间件不提供接口则查询后删除
        rs = self.preset_list_read(device_id, screen_id)
        if rs['status'] == 0 and rs['data']:
            del_result = []
            success = True
            for p in rs['data']['presets']:
                preset_id = p['presetId']
                params = {'deviceId': device_id, 'screenId': screen_id, 'presetId': preset_id}
                del_rs = self.preset_delete(device_id, screen_id, preset_id, params)
                if del_rs['status'] != 0:
                    success = False
                params['status'] = del_rs['status']
                del_result.append(params)
            if success:
                rs = get_result_model()
            else:
                rs = get_result_model(ResInfo.Preset_Batch_Delete_Part_Success)
                rs['data'] = del_result

        else:
            rs = get_result_model(ResInfo.Preset_List_Error)
        return rs

    def preset_list_read(self, device_id, screen_id):
        """
        读取屏幕下所有场景
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :return: 场景列表，参考中间件协议
        """
        rs = preset_api.read_list(device_id, screen_id)
        if rs['status'] == 0 and rs['data']:
            preset_list = []
            # 循环所有场景，读取详细信息
            for preset in rs['data']['presets']:
                preset_id = preset['presetId']
                detail_rs = self.preset_detail_read(device_id, screen_id, preset_id)
                if detail_rs['status'] == 0:
                    preset_list.append(detail_rs['data'])
                else:  # 读取单个详情失败，返回
                    rs = get_result_model(ResInfo.Preset_List_Error)
                    return rs

            rs['data']['presets'] = preset_list

        return rs

    def preset_origin_list_read(self, device_id, screen_id):
        return preset_api.read_list(device_id, screen_id)

    def preset_group_list(self, device_id, screen_id):
        """
        读取屏幕下场景分组列表
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :return: 参考前端接口文档场景分组信息
        """
        rs = preset_api.read_group_list(device_id, screen_id)
        if rs['status'] != 0 or not rs['data']:
            return rs
        des = rs
        for group in des['data']['presetGroups']:
            group_id = group['presetGroupId']
            group_node = preset_api.read_group_node_list(device_id, screen_id, group_id)
            if group_node['status'] == 0 and group_node['data']:
                presets = []
                for preset in group_node['data']['presets']:
                    preset_id = preset['presetId']
                    detail_rs = self.preset_detail_read(device_id, screen_id, preset_id)
                    if detail_rs['status'] == 0:
                        presets.append(detail_rs['data'])
                group['presets'] = presets
        return des

    def preset_group_create(self, device_id, screen_id, params):
        """
        创建场景分组
        :param device_id: 设备Id
        :param screen_id:  屏幕Id
        :param params: 分组信息
        :return:  创建分组Id
        """
        return preset_api.write_group_create(device_id, screen_id, params)

    def preset_group_update(self, device_id, screen_id, params):
        """
        创建场景分组
        :param device_id: 设备Id
        :param screen_id:  屏幕Id
        :param params: 分组信息
        :return:  结果对象
        """
        return preset_api.write_group_update(device_id, screen_id, params)

    def preset_group_delete(self, device_id, screen_id, group_id):
        """
        删除场景分组
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :param group_id: 分组Id
        :return: 结果数据
        """
        params = {
            "presetGroupId": group_id,
            "screenId": screen_id,
            "deviceId": device_id
        }
        return preset_api.write_group_delete(device_id, screen_id, params)

    def preset_group_node_list(self, device_id, screen_id, group_id):
        """
        读取场景分组节点列表
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :param group_id:  分组Id
        :return:  分组节点列表信息
        """
        return preset_api.read_group_node_list(device_id, screen_id, group_id)

    def preset_group_node_add(self, device_id, screen_id, group_id, preset_id):
        """
        读取场景分组节点列表
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :param group_id:  分组Id
        :param preset_id: 场景Id
        :return:  分组节点列表信息
        """
        params = {
            "presetId": preset_id,
            "presetGroupId": group_id,
            "screenId": screen_id,
            "deviceId": device_id
        }
        return preset_api.write_group_node_add(device_id, screen_id, group_id, params)

    def preset_group_node_delete(self, device_id, screen_id, group_id, preset_id):
        """
        读取场景分组节点列表
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :param group_id:  分组Id
        :param preset_id: 场景Id
        :return:  分组节点删除场景
        """
        params = {
            "presetId": preset_id,
            "presetGroupId": group_id,
            "screenId": screen_id,
            "deviceId": device_id
        }
        return preset_api.write_group_node_delete(device_id, screen_id, group_id, params)

    def preset_poll(self, device_id, screen_id, params):
        """
        场景轮询
        :param device_id: 设备Id
        :param screen_id:  屏幕Id
        :param params: 轮询参数
        :return: 结果数据
        """
        enable = params['enable']
        if enable == 1:
            # 关闭原来场景轮循
            rs = self.preset_poll_read(device_id, screen_id)
            if rs['status'] == 0 and rs['data']:
                enable_old = rs['data']['enable']
                if enable_old == 1:
                    group_id = rs['data']['presetGroupId']
                    preset_api.write_poll(device_id, screen_id, {
                        "screenId": screen_id,
                        "deviceId": device_id,
                        "presetGroupId": group_id,
                        "enable": 0
                    })
            # 校验场景分组中是否有数据
            group_id = params['presetGroupId']
            rs = preset_api.read_group_node_list(device_id, screen_id, group_id)
            if rs['status'] != 0 or not rs['data'] or not rs['data'].get('presets', None):
                return get_result_model(ResInfo.Preset_Group_Empty)
        return preset_api.write_poll(device_id, screen_id, params)

    def preset_poll_read(self, device_id, screen_id):
        """
        读取场景轮询数据
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :return: 轮询场景组信息
        """
        return preset_api.read_poll(device_id, screen_id)

    def preset_nontime_save(self, device_id, screen_id, data):
        """
        非实时上屏保存场景
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param data: 场景信息
        :return: 结果对象
        """
        return preset_api.write_nontime_save(device_id, screen_id, data)

    def preset_config_read(self, device_id):
        """
        读取场景切换特效
        :param device_id:设备Id
        :return: 结果对象
        """
        rs = preset_api.read_preset_config(device_id)
        return rs

    def preset_config_write(self, device_id, data):
        """
        设置场景切换特效
        :param device_id:  设备Id
        :param data: 特效数据
        :return: 结果对象
        """
        enable = data['effectEnable']
        if enable > 0:
            data = {
                "deviceId": 0,
                "effectEnable": enable,
                "effectTime": data['effectTime']
            }
        return preset_api.write_preset_config(device_id, data)

    def preset_show_id_read(self, device_id):
        """读取是否显示场景id"""
        rs = model_conf_dao.get_value(ModelConfKey.PresetShowId, str(device_id))
        if rs['status'] == 0:
            rs['data'] = {
                'deviceId': device_id,
                'enable': 0 if rs['data']['value'] == -1 else rs['data']['value']
            }
        return rs

    def preset_show_id_write(self, device_id, enable):
        """读取是否显示场景id"""
        key = str(device_id)
        return model_conf_dao.set_value(ModelConfKey.PresetShowId, key, enable)


    def preset_collection_create(self, device_id, params):
        """
        创建场景组合
        :return:  场景组合Id
        """
        return preset_api.write_collection_create(device_id, params)

    def preset_collection_update(self, device_id, params):
        """
        编辑场景组合
        :return:  操作结果
        """
        return preset_api.write_collection_update(device_id, params)

    def preset_collection_delete(self, device_id, preset_collection_id):
        """
        删除场景分组
        :param device_id: 设备Id
        :param preset_collection_id: 场景组合Id
        :return: 结果数据
        """
        params = {
            "presetCollectionId": preset_collection_id,
            "deviceId": device_id
        }
        return preset_api.write_collection_delete(device_id,params)

    def preset_collection_del_all(self, device_id, params):
        """
        清空场景分组
        """
        return preset_api.write_collection_del_all(device_id,params)

    def preset_collection_play(self, device_id, params):
        """
        播放场景分组
        :param device_id: 设备Id
        :param params: 组合id
        """
        return preset_api.write_collection_play(device_id, params)

    def preset_collection_read(self, device_id, params):
        """
        读取场景组合数据
        :param device_id: 设备Id
        :param params: 组合id
        :return: 场景组合信息
        """
        return preset_api.read_collection(device_id, params)

    def preset_collection_list(self, device_id):
        """
        读取场景组合列表数据
        :param device_id: 设备Id
        :return: 场景组合列表信息
        """
        return preset_api.read_collection_list(device_id)

    def read_multi_detail(self, device_id, params):
        """
        批量查询多个场景详情
        :param device_id: 设备Id
        :param params: 场景id
        :return: 场景详情列表
        """
        return preset_api.read_multi_detail(device_id, params)


preset_svc = PresetSvc()
